/*
 *    Copyright 2009-2023 the original author or authors.
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *       https://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.apache.ibatis.annotations;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.apache.ibatis.type.JdbcType;
import org.apache.ibatis.type.TypeHandler;
import org.apache.ibatis.type.UnknownTypeHandler;

// @formatter:off
/**
 * The annotation that be grouping conditional mapping definitions.
 * <p>
 * <b>How to use:</b>
 *
 * <pre>
 * public interface UserMapper {
 *   &#064;Select("SELECT id, name, type FROM users ORDER BY id")
 *   &#064;TypeDiscriminator(
 *     column = "type",
 *     javaType = String.class,
 *     cases = {
 *       &#064;Case(value = "1", type = PremiumUser.class),
 *       &#064;Case(value = "2", type = GeneralUser.class),
 *       &#064;Case(value = "3", type = TemporaryUser.class)
 *     }
 *   )
 *   List&lt;User&gt; selectAll();
 * }
 * </pre>
 *
 * @author Clinton Begin
 */
// @formatter:on
@Documented
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.METHOD)
public @interface TypeDiscriminator {

  /**
   * Returns the column name(column label) that hold conditional value.
   *
   * @return the column name(column label)
   */
  String column();

  /**
   * Return the java type for conditional value.
   *
   * @return the java type
   */
  Class<?> javaType() default void.class;

  /**
   * Return the jdbc type for column that hold conditional value.
   *
   * @return the jdbc type
   */
  JdbcType jdbcType() default JdbcType.UNDEFINED;

  /**
   * Returns the {@link TypeHandler} type for retrieving a column value from result set.
   *
   * @return the {@link TypeHandler} type
   */
  Class<? extends TypeHandler> typeHandler() default UnknownTypeHandler.class;

  /**
   * Returns conditional mapping definitions.
   *
   * @return conditional mapping definitions
   */
  Case[] cases();
}
