/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.memory;

import java.io.IOException;
import java.util.Arrays;
import java.util.BitSet;
import java.util.HashSet;
import javax.annotation.concurrent.GuardedBy;
import org.apache.spark.memory.MemoryConsumer;
import org.apache.spark.memory.MemoryManager;
import org.apache.spark.memory.MemoryMode;
import org.apache.spark.unsafe.memory.MemoryBlock;
import org.apache.spark.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spark-project.guava.annotations.VisibleForTesting;

public class TaskMemoryManager {
    private final Logger logger = LoggerFactory.getLogger(TaskMemoryManager.class);
    private static final int PAGE_NUMBER_BITS = 13;
    @VisibleForTesting
    static final int OFFSET_BITS = 51;
    private static final int PAGE_TABLE_SIZE = 8192;
    public static final long MAXIMUM_PAGE_SIZE_BYTES = 0x3FFFFFFF8L;
    private static final long MASK_LONG_LOWER_51_BITS = 0x7FFFFFFFFFFFFL;
    private static final long MASK_LONG_UPPER_13_BITS = -2251799813685248L;
    private final MemoryBlock[] pageTable = new MemoryBlock[8192];
    private final BitSet allocatedPages = new BitSet(8192);
    private final MemoryManager memoryManager;
    private final long taskAttemptId;
    final MemoryMode tungstenMemoryMode;
    @GuardedBy(value="this")
    private final HashSet<MemoryConsumer> consumers;

    public TaskMemoryManager(MemoryManager memoryManager, long taskAttemptId) {
        this.tungstenMemoryMode = memoryManager.tungstenMemoryMode();
        this.memoryManager = memoryManager;
        this.taskAttemptId = taskAttemptId;
        this.consumers = new HashSet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long acquireExecutionMemory(long required, MemoryMode mode, MemoryConsumer consumer) {
        assert (required >= 0L);
        TaskMemoryManager taskMemoryManager = this;
        synchronized (taskMemoryManager) {
            long got = this.memoryManager.acquireExecutionMemory(required, this.taskAttemptId, mode);
            if (got < required) {
                for (MemoryConsumer c : this.consumers) {
                    if (c == consumer || c.getUsed() <= 0L) continue;
                    try {
                        long released = c.spill(required - got, consumer);
                        if (released <= 0L || mode != this.tungstenMemoryMode) continue;
                        this.logger.debug("Task {} released {} from {} for {}", new Object[]{this.taskAttemptId, Utils.bytesToString(released), c, consumer});
                        if ((got += this.memoryManager.acquireExecutionMemory(required - got, this.taskAttemptId, mode)) < required) continue;
                        break;
                    }
                    catch (IOException e) {
                        this.logger.error("error while calling spill() on " + c, (Throwable)e);
                        throw new OutOfMemoryError("error while calling spill() on " + c + " : " + e.getMessage());
                    }
                }
            }
            if (got < required && consumer != null) {
                try {
                    long released = consumer.spill(required - got, consumer);
                    if (released > 0L && mode == this.tungstenMemoryMode) {
                        this.logger.debug("Task {} released {} from itself ({})", new Object[]{this.taskAttemptId, Utils.bytesToString(released), consumer});
                        got += this.memoryManager.acquireExecutionMemory(required - got, this.taskAttemptId, mode);
                    }
                }
                catch (IOException e) {
                    this.logger.error("error while calling spill() on " + consumer, (Throwable)e);
                    throw new OutOfMemoryError("error while calling spill() on " + consumer + " : " + e.getMessage());
                }
            }
            if (consumer != null) {
                this.consumers.add(consumer);
            }
            this.logger.debug("Task {} acquire {} for {}", new Object[]{this.taskAttemptId, Utils.bytesToString(got), consumer});
            return got;
        }
    }

    public void releaseExecutionMemory(long size, MemoryMode mode, MemoryConsumer consumer) {
        this.logger.debug("Task {} release {} from {}", new Object[]{this.taskAttemptId, Utils.bytesToString(size), consumer});
        this.memoryManager.releaseExecutionMemory(size, this.taskAttemptId, mode);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void showMemoryUsage() {
        this.logger.info("Memory used in task " + this.taskAttemptId);
        TaskMemoryManager taskMemoryManager = this;
        synchronized (taskMemoryManager) {
            long memoryAccountedForByConsumers = 0L;
            for (MemoryConsumer c : this.consumers) {
                long totalMemUsage = c.getUsed();
                memoryAccountedForByConsumers += totalMemUsage;
                if (totalMemUsage <= 0L) continue;
                this.logger.info("Acquired by " + c + ": " + Utils.bytesToString(totalMemUsage));
            }
            long memoryNotAccountedFor = this.memoryManager.getExecutionMemoryUsageForTask(this.taskAttemptId) - memoryAccountedForByConsumers;
            this.logger.info("{} bytes of memory were used by task {} but are not associated with specific consumers", (Object)memoryNotAccountedFor, (Object)this.taskAttemptId);
            this.logger.info("{} bytes of memory are used for execution and {} bytes of memory are used for storage", (Object)this.memoryManager.executionMemoryUsed(), (Object)this.memoryManager.storageMemoryUsed());
        }
    }

    public long pageSizeBytes() {
        return this.memoryManager.pageSizeBytes();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public MemoryBlock allocatePage(long size, MemoryConsumer consumer) {
        int pageNumber;
        if (size > 0x3FFFFFFF8L) {
            throw new IllegalArgumentException("Cannot allocate a page with more than 17179869176 bytes");
        }
        long acquired = this.acquireExecutionMemory(size, this.tungstenMemoryMode, consumer);
        if (acquired <= 0L) {
            return null;
        }
        TaskMemoryManager taskMemoryManager = this;
        synchronized (taskMemoryManager) {
            pageNumber = this.allocatedPages.nextClearBit(0);
            if (pageNumber >= 8192) {
                this.releaseExecutionMemory(acquired, this.tungstenMemoryMode, consumer);
                throw new IllegalStateException("Have already allocated a maximum of 8192 pages");
            }
            this.allocatedPages.set(pageNumber);
        }
        MemoryBlock page = this.memoryManager.tungstenMemoryAllocator().allocate(acquired);
        page.pageNumber = pageNumber;
        this.pageTable[pageNumber] = page;
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Allocate page number {} ({} bytes)", (Object)pageNumber, (Object)acquired);
        }
        return page;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void freePage(MemoryBlock page, MemoryConsumer consumer) {
        assert (page.pageNumber != -1) : "Called freePage() on memory that wasn't allocated with allocatePage()";
        assert (this.allocatedPages.get(page.pageNumber));
        this.pageTable[page.pageNumber] = null;
        TaskMemoryManager taskMemoryManager = this;
        synchronized (taskMemoryManager) {
            this.allocatedPages.clear(page.pageNumber);
        }
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Freed page number {} ({} bytes)", (Object)page.pageNumber, (Object)page.size());
        }
        long pageSize = page.size();
        this.memoryManager.tungstenMemoryAllocator().free(page);
        this.releaseExecutionMemory(pageSize, this.tungstenMemoryMode, consumer);
    }

    public long encodePageNumberAndOffset(MemoryBlock page, long offsetInPage) {
        if (this.tungstenMemoryMode == MemoryMode.OFF_HEAP) {
            offsetInPage -= page.getBaseOffset();
        }
        return TaskMemoryManager.encodePageNumberAndOffset(page.pageNumber, offsetInPage);
    }

    @VisibleForTesting
    public static long encodePageNumberAndOffset(int pageNumber, long offsetInPage) {
        assert (pageNumber != -1) : "encodePageNumberAndOffset called with invalid page";
        return (long)pageNumber << 51 | offsetInPage & 0x7FFFFFFFFFFFFL;
    }

    @VisibleForTesting
    public static int decodePageNumber(long pagePlusOffsetAddress) {
        return (int)((pagePlusOffsetAddress & 0xFFF8000000000000L) >>> 51);
    }

    private static long decodeOffset(long pagePlusOffsetAddress) {
        return pagePlusOffsetAddress & 0x7FFFFFFFFFFFFL;
    }

    public Object getPage(long pagePlusOffsetAddress) {
        if (this.tungstenMemoryMode == MemoryMode.ON_HEAP) {
            int pageNumber = TaskMemoryManager.decodePageNumber(pagePlusOffsetAddress);
            assert (pageNumber >= 0 && pageNumber < 8192);
            MemoryBlock page = this.pageTable[pageNumber];
            assert (page != null);
            assert (page.getBaseObject() != null);
            return page.getBaseObject();
        }
        return null;
    }

    public long getOffsetInPage(long pagePlusOffsetAddress) {
        long offsetInPage = TaskMemoryManager.decodeOffset(pagePlusOffsetAddress);
        if (this.tungstenMemoryMode == MemoryMode.ON_HEAP) {
            return offsetInPage;
        }
        int pageNumber = TaskMemoryManager.decodePageNumber(pagePlusOffsetAddress);
        assert (pageNumber >= 0 && pageNumber < 8192);
        MemoryBlock page = this.pageTable[pageNumber];
        assert (page != null);
        return page.getBaseOffset() + offsetInPage;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long cleanUpAllAllocatedMemory() {
        TaskMemoryManager taskMemoryManager = this;
        synchronized (taskMemoryManager) {
            Arrays.fill(this.pageTable, null);
            for (MemoryConsumer c : this.consumers) {
                if (c == null || c.getUsed() <= 0L) continue;
                this.logger.warn("leak " + Utils.bytesToString(c.getUsed()) + " memory from " + c);
            }
            this.consumers.clear();
        }
        for (MemoryBlock page : this.pageTable) {
            if (page == null) continue;
            this.memoryManager.tungstenMemoryAllocator().free(page);
        }
        Arrays.fill(this.pageTable, null);
        return this.memoryManager.releaseAllExecutionMemoryForTask(this.taskAttemptId);
    }

    public long getMemoryConsumptionForThisTask() {
        return this.memoryManager.getExecutionMemoryUsageForTask(this.taskAttemptId);
    }
}

