/**
 * fshows.com
 * Copyright (C) 2013-2025 All Rights Reserved.
 */
package com.fshows.ysepay;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.definition.IResponseDefinition;
import com.fshows.sdk.core.client.base.handler.IHttpRequestHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.client.component.http.FromHttpRequestHandler;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import com.fshows.ysepay.apienum.YsepayIncomeApiDefinitionEnum;
import com.fshows.ysepay.request.income.YsepayIncomeBizRequest;
import com.fshows.ysepay.request.income.YsepayIncomeFileBizRequest;
import com.fshows.ysepay.response.income.YsepayIncomeBizResponse;
import com.fshows.ysepay.util.AESUtil;
import com.fshows.ysepay.util.ByteUtil;
import com.fshows.ysepay.util.CertUtil;
import com.fshows.ysepay.util.YsepaySignatureUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;

/**
 * @author mengqf
 * @version YsepayUploadApiClient.java, v 0.1 2025-06-09 15:10 mengqf
 */
@Slf4j
public class YsepayUploadApiClient extends AbstractApiClient<YsepayIncomeFileBizRequest, YsepayIncomeBizResponse, YsepayIncomeApiDefinitionEnum> {
    private PublicKey publicKey;

    private PrivateKey privateKey;

    private static final String ALLCHAR = "0123456789ABCDEF";

    /**
     * 请求执行器
     */
    protected IHttpRequestHandler fromHttpRequestHandler = new FromHttpRequestHandler();

    public YsepayUploadApiClient(YsepayIncomeClientConfigModel apiClientConfig) throws Exception {
        super(apiClientConfig);
        publicKey = CertUtil.getValidateCert(apiClientConfig.getYsepayPublicKeyPath()).getPublicKey();
        privateKey = YsepaySignatureUtil.getPrivateKey(apiClientConfig.getPrivateKeyPath());
    }

    @Override
    public YsepayIncomeBizResponse execute(YsepayIncomeFileBizRequest request, YsepayIncomeApiDefinitionEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public YsepayIncomeBizResponse execute(YsepayIncomeFileBizRequest request, YsepayIncomeApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

    @Override
    protected YsepayIncomeBizResponse doExecute(YsepayIncomeFileBizRequest request, YsepayIncomeApiDefinitionEnum iApiDefinition, DefaultClientConfigModel customConfig) throws FsApiException {
        long beginTime = System.currentTimeMillis();
        LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}", getClientInfo().getClientDesc(), iApiDefinition, request);

        // 构建请求上下文
        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, customConfig);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;

        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);
            // 请求开始时间
            long reqBeginTime = System.currentTimeMillis();
            // 执行post请求
            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            // 请求结束时间
            long reqEndTime = System.currentTimeMillis();

            YsepayIncomeBizResponse response = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);

            LogUtil.info(log, "{} >> 请求结束[明文] >> url={}, method={}, request={}, response={}, totalcost={}ms, reqcost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), requestContext.getIApiDefinition(), JSONObject.toJSONString(apiRequestModel.getRequest()),
                    JSONObject.toJSONString(apiResponseModel), System.currentTimeMillis() - beginTime, reqEndTime - reqBeginTime);

            return response;
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求业务异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求未知异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            LogUtil.info(log, "{} >> 请求未知异常[明文] >> url={}, method={}, request={}, cost={}ms", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel == null ? "" : apiRequestModel.getApiURL(),
                    requestContext.getIApiDefinition(),
                    JSONObject.toJSONString(apiRequestModel == null ? "" : apiRequestModel.getRequest()),
                    System.currentTimeMillis() - beginTime);
            throw new FsApiException(e.getMessage(), e);
        }
    }

    @Override
    protected DefaultRequestContext buildRequestContext(YsepayIncomeApiDefinitionEnum incomeApiDefinitionEnum, YsepayIncomeFileBizRequest request, DefaultClientConfigModel customConfig) {
        DefaultRequestContext context = new DefaultRequestContext();
        context.setIApiDefinition(incomeApiDefinitionEnum);

        YsepayIncomeClientConfigModel clientConfig = (YsepayIncomeClientConfigModel) this.apiClientConfig;
        // 复制默认参数
        YsepayIncomeClientConfigModel config = new YsepayIncomeClientConfigModel();
        if (StringUtils.isBlank(request.getCertId())) {
            request.setCertId(clientConfig.getCertId());
        }
        config.setCertId(request.getCertId());


        // 需要复制所有必要的配置参数
        config.setApiParentURL(clientConfig.getApiParentURL());
        config.setCharset(clientConfig.getCharset());
        config.setSignType(clientConfig.getSignType());
        config.setPrivateKeyPath(clientConfig.getPrivateKeyPath());
        config.setYsepayPublicKeyPath(clientConfig.getYsepayPublicKeyPath());

        context.setApiClientConfig(config);
        context.setClientInfoModel(getClientInfo());
        return context;
    }

    @Override
    protected ApiRequestModel buildApiRequestModel(YsepayIncomeFileBizRequest request, DefaultRequestContext context) {
        // sdk客户端配置
        YsepayIncomeClientConfigModel  configModel = (YsepayIncomeClientConfigModel ) context.getApiClientConfig();

        ApiRequestModel apiRequestModel = new ApiRequestModel();
        // 设置网关地址
        YsepayIncomeApiDefinitionEnum iApiDefinition = (YsepayIncomeApiDefinitionEnum)context.getIApiDefinition();
        apiRequestModel.setApiURL(context.getApiClientConfig().getApiParentURL() + iApiDefinition.getApiURI());

        // 设置Content-Type
        apiRequestModel.setContentType("application/json");

        // 构建公共参数
        /** 1、封装请求参数，bizContent（业务参数，需要加密）和check（需要生成密钥并加密）后面设置值*/
        Map<String,String> reqMap = new HashMap<>();
        //接口方法名
        reqMap.put("method", iApiDefinition.getMethod());
        //请求时间
        reqMap.put("timeStamp", getCurrentDateTime());
        //固定utf-8
        reqMap.put("charset","utf-8");
        //请求流水号
        reqMap.put("reqId",String.valueOf(System.currentTimeMillis()));
        //发起方商户号，服务商在银盛给自己开设的商户号，即可当作发起方商户号，由银盛生成并下发。 注意：不同于子商户号，服务商发展的商户即为子商户号
        reqMap.put("certId", ((YsepayIncomeClientConfigModel) context.getApiClientConfig()).getCertId());
        //版本号，默认1.0，按接口文档来
        reqMap.put("version", iApiDefinition.getVersion());

        /** 2、生成对业务参数加密的密钥*/
        StringBuilder sb = new StringBuilder();
        Random random = new Random();
        for (int i = 0; i < 16; i++) {
            sb.append(ALLCHAR.charAt(random.nextInt(ALLCHAR.length())));
        }
        String key = ByteUtil.toHexString(sb.toString());
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("key", key);
        /** 3、使用银盛公钥对密钥key进行加密，得到加密的key并设置到请求参数check中。publicFilePath为存放银盛公钥的地址*/
        byte[] byte1= null;
        try {
            byte1 = YsepaySignatureUtil.encrypt(publicKey, ByteUtil.hexStringToBytes(key));
        }catch (Exception e) {
            throw new FsApiException("YsepayIncomeApiClient --》使用银盛公钥对密钥key进行加密异常", e);
        }
        String encryptKeyCheck = Base64.encodeBase64String(byte1);
        //加密后的密钥
        reqMap.put("check", encryptKeyCheck);

        /** 5、使用生成的密钥key对业务参数进行加密，并将加密后的业务参数放入请求参数bizContent中*/
        byte[] bte= null;
        try {
            bte = AESUtil.encrypt(JSONObject.toJSONBytes(request, SerializerFeature.WriteNullStringAsEmpty), ByteUtil.hexStringToBytes(key));
        }catch (Exception e) {
            throw new FsApiException("YsepayIncomeApiClient --》使用生成的密钥key对业务参数进行加密异常", e);
        }
        String msgBizContent = Base64.encodeBase64String(bte);
        reqMap.put("bizContent", msgBizContent);

        /** 6、将请求参数进行sort排序，生成拼接的字符床，并使用接入方私钥对请求参数进行加签，并将加签的值存入请求参数sign中*/
        // 6.1 排序生成拼接字符串
        List<String> keys = new ArrayList<String>(reqMap.keySet());
        Collections.sort(keys);
        StringBuilder sb1 = new StringBuilder();
        for(String k : keys){
            if("sign".equals(k)) {
                continue;
            }
            sb1.append(k).append("=");
            sb1.append(reqMap.get(k));
            sb1.append("&");
        }
        if(sb1.length() > 0) {
            sb1.setLength(sb1.length() - 1);
        }
        String sign = "";
        //  6.2 生成签名
        try {
            sign = YsepaySignatureUtil.signWithRsa256(sb1.toString(), privateKey);
        } catch (Exception e) {
            throw new FsApiException("YsepayIncomeApiClient --》生成签名异常", e);
        }
        reqMap.put("sign",sign);

        apiRequestModel.setRequestBody(JSONObject.toJSONString(reqMap));
        apiRequestModel.setRequest(request);
        apiRequestModel.setParamMap(paramMap);
        apiRequestModel.setRequestForm(reqMap);
        return apiRequestModel;
    }

    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientName("银盛支付");
        clientInfoModel.setClientCode("ysepay-sdk");
        return clientInfoModel;
    }

    @Override
    protected YsepayIncomeBizResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        try {
            String result = apiResponseModel.getResponseBody();
            byte[] res = Base64.decodeBase64(result);
            Map<String, String> resMap = (Map<String, String>) JSONObject.parse(new String(res,"UTF-8"));
            /** 对结果进行解密，并使用银盛公钥验签*/
            if(StringUtils.isNotBlank(resMap.get("sign"))) {
                byte[] srcData = YsepaySignatureUtil.getSignDataStr1(resMap).getBytes("UTF-8");
                boolean validateSignResult= YsepaySignatureUtil.validateSignBySoft(publicKey,Base64.decodeBase64(resMap.get("sign")),srcData);
                if(!validateSignResult){
                    LogUtil.error(log,"验签失败,可能是银盛未配置发起方或者发起方证书类型配置有误,返回结果提示为:{}",resMap.get("msg"));
                    throw new FsApiException("验签失败,可能是银盛未配置发起方或者发起方证书类型配置有误,返回结果提示为:"+resMap.get("msg"));
                }
            }else{
                throw new FsApiException("验签失败,未返回加签信息,可能是银盛未配置发起方或者发起方证书类型配置有误,返回结果提示为:"+resMap.get("msg"));
            }
            String key = apiRequestModel.getParamMap().get("key").toString();
            YsepayIncomeBizResponse baseResponse = JSONObject.parseObject(JSONObject.toJSONString(resMap), YsepayIncomeBizResponse.class);
            if(StringUtils.isNotBlank(resMap.get("businessData"))) {
                byte[] data_ = Base64.decodeBase64(resMap.get("businessData"));
                byte[] data = AESUtil.decrypt(data_,ByteUtil.hexStringToBytes(key));
                String stringBody = new String(data, "UTF-8");
                baseResponse.setBusinessData( stringBody);
                LogUtil.info(log,"解密后的业务参数:{}",stringBody);

                // 设置响应
                // 解析业务响应
                IResponseDefinition iResponseDefinition = JSON.parseObject(stringBody,
                        requestContext.getIApiDefinition().getResponseClass());
                baseResponse.setDecryptBusinessContext(iResponseDefinition);
            }
            return baseResponse;

        } catch (Exception e) {
            throw new FsApiException("解析响应失败", e);
        }
    }

    @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
        HttpPost httpPost = new HttpPost(apiRequestModel.getApiURL());
        File file = ((YsepayIncomeFileBizRequest)apiRequestModel.getRequest()).getFile();
        MultipartEntityBuilder entity = MultipartEntityBuilder.create()
                .setContentType(ContentType.MULTIPART_FORM_DATA)
                .setMode(HttpMultipartMode.BROWSER_COMPATIBLE)
                .addBinaryBody("file", Files.newInputStream(file.toPath()), ContentType.DEFAULT_BINARY, file.getName())
                .setCharset(StandardCharsets.UTF_8);
        // 请求参数
        for (String key : apiRequestModel.getRequestForm().keySet()) {
            entity.addTextBody(key, apiRequestModel.getRequestForm().get(key), ContentType.DEFAULT_TEXT);
        }
        httpPost.setEntity(entity.build());
        RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(50000).setConnectTimeout(30000)
                .build();// 设置请求和传输超时时间
        httpPost.setConfig(requestConfig);
        CloseableHttpClient httpclient = HttpClients.createDefault();
        CloseableHttpResponse response = httpclient.execute(httpPost);
        String responseString = null;
        if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
            HttpEntity httpEntity = response.getEntity();
            try {
                if (httpEntity != null) {
                    responseString = EntityUtils.toString(httpEntity);
                    EntityUtils.consume(httpEntity);
                }
            } finally {
                response.close();
            }
        }
        ApiResponseModel apiResponseModel = new ApiResponseModel();
        apiResponseModel.setResponseBody(responseString);
        return apiResponseModel;
    }
    /**
     * 格式化当前日期时间字符串
     * @return
     */
    private String getCurrentDateTime(){
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:dd");
        return sdf.format(new Date());
    }
}