package com.fshows.ysepay;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.definition.IResponseDefinition;
import com.fshows.sdk.core.client.base.handler.IHttpRequestHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.client.component.http.FromHttpRequestHandler;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import com.fshows.ysepay.apienum.YsepayTradeApiDefinitionEnum;
import com.fshows.ysepay.response.YsepayBaseResponse;
import com.fshows.ysepay.util.YsepaySignatureUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.lang.reflect.Field;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * 银盛支付API客户端
 */
@Slf4j
public class YsepayApiClient extends AbstractApiClient<com.fshows.ysepay.request.YsepayBizRequest, com.fshows.ysepay.response.YsepayBaseResponse, YsepayTradeApiDefinitionEnum> {
	private PublicKey publicKey;
	
	private PrivateKey privateKey;
	/**
	 * 请求执行器
	 */
	protected IHttpRequestHandler httpRequestHandler = new FromHttpRequestHandler();
	
	public YsepayApiClient(YsepayClientConfigModel apiClientConfig) throws Exception {
		super(apiClientConfig);
		publicKey = YsepaySignatureUtil.getPublicKeyFromCert(apiClientConfig.getYsepayPublicKeyPath());
		privateKey = YsepaySignatureUtil.getPrivateKey(apiClientConfig.getPrivateKeyPath());
	}
	
	@Override
	public com.fshows.ysepay.response.YsepayBaseResponse execute(com.fshows.ysepay.request.YsepayBizRequest request, YsepayTradeApiDefinitionEnum apiDefinition) throws FsApiException {
		return doExecute(request, apiDefinition);
	}
	
	@Override
	public com.fshows.ysepay.response.YsepayBaseResponse execute(com.fshows.ysepay.request.YsepayBizRequest request, YsepayTradeApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
		return doExecute(request, apiDefinition, configModel);
	}
	
	@Override
	protected com.fshows.ysepay.response.YsepayBaseResponse doExecute(com.fshows.ysepay.request.YsepayBizRequest request, YsepayTradeApiDefinitionEnum iApiDefinition, DefaultClientConfigModel customConfig) throws FsApiException {
		long beginTime = System.currentTimeMillis();
		LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}", getClientInfo().getClientDesc(), iApiDefinition, request);
		
		// 构建请求上下文
		DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, customConfig);
		ApiRequestModel apiRequestModel = null;
		ApiResponseModel apiResponseModel = null;
		
		try {
			// 入参数校验
			checkParam(request, requestContext);
			// 构建请求数据
			apiRequestModel = buildApiRequestModel(request, requestContext);
			// 请求开始时间
			long reqBeginTime = System.currentTimeMillis();
			// 执行post请求
			apiResponseModel = httpRequest(apiRequestModel, requestContext);
			// 请求结束时间
			long reqEndTime = System.currentTimeMillis();
			
			LogUtil.debug(log, "{} >> 请求结束[密文] >> url={}, method={}, request={}, response={}, cost={}ms, reqcost={}ms",
					requestContext.getClientInfoModel().getClientDesc(),
					apiRequestModel.getApiURL(), iApiDefinition, apiRequestModel.getRequestBody(),
					apiResponseModel.getResponseBody(), System.currentTimeMillis() - beginTime, reqEndTime - reqBeginTime);
			
			com.fshows.ysepay.response.YsepayBaseResponse response = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);
			
			LogUtil.info(log, "{} >> 请求结束[明文] >> url={}, method={}, request={}, response={}, totalcost={}ms, reqcost={}ms",
					requestContext.getClientInfoModel().getClientDesc(),
					apiRequestModel.getApiURL(), requestContext.getIApiDefinition(), JSONObject.toJSONString(apiRequestModel.getRequest()),
					JSONObject.toJSONString(apiResponseModel.getResponse()), System.currentTimeMillis() - beginTime, reqEndTime - reqBeginTime);
			
			return response;
		} catch (FsApiException e) {
			LogUtil.error(log, "{} >> 请求业务异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
					requestContext.getClientInfoModel().getClientDesc(),
					iApiDefinition, request, apiRequestModel, apiResponseModel);
			throw e;
		} catch (Exception e) {
			LogUtil.error(log, "{} >> 请求未知异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
					requestContext.getClientInfoModel().getClientDesc(),
					iApiDefinition, request, apiRequestModel, apiResponseModel);
			LogUtil.info(log, "{} >> 请求未知异常[明文] >> url={}, method={}, request={}, cost={}ms", e,
					requestContext.getClientInfoModel().getClientDesc(),
					apiRequestModel == null ? "" : apiRequestModel.getApiURL(),
					requestContext.getIApiDefinition(),
					JSONObject.toJSONString(apiRequestModel == null ? "" : apiRequestModel.getRequest()),
					System.currentTimeMillis() - beginTime);
			throw new FsApiException(e.getMessage(), e);
		}
	}
	
	@Override
	protected DefaultRequestContext buildRequestContext(YsepayTradeApiDefinitionEnum tradeApiDefinitionEnum, com.fshows.ysepay.request.YsepayBizRequest request, DefaultClientConfigModel customConfig) {
		DefaultRequestContext context = new DefaultRequestContext();
		context.setIApiDefinition(tradeApiDefinitionEnum);
		
		YsepayClientConfigModel clientConfig = (YsepayClientConfigModel) this.apiClientConfig;
		// 复制默认参数
		YsepayClientConfigModel config = new YsepayClientConfigModel();
		config.setPartnerId(StringUtils.isEmpty(request.getPartnerId()) ? clientConfig.getPartnerId() : request.getPartnerId());
		
		
		// 需要复制所有必要的配置参数
		config.setNotifyUrl(request.getNotifyUrl());
		config.setApiParentURL(clientConfig.getApiParentURL());
		config.setCharset(clientConfig.getCharset());
		config.setSignType(clientConfig.getSignType());
		config.setVersion(tradeApiDefinitionEnum.getVersion());
		config.setPrivateKeyPath(clientConfig.getPrivateKeyPath());
		config.setYsepayPublicKeyPath(clientConfig.getYsepayPublicKeyPath());
		
		context.setApiClientConfig(config);
		context.setClientInfoModel(getClientInfo());
		return context;
	}
	
	@Override
	protected ApiRequestModel buildApiRequestModel(com.fshows.ysepay.request.YsepayBizRequest request, DefaultRequestContext context) {
		// sdk客户端配置
		YsepayClientConfigModel configModel = (YsepayClientConfigModel) context.getApiClientConfig();
		
		ApiRequestModel apiRequestModel = new ApiRequestModel();
		// 设置网关地址
		apiRequestModel.setApiURL(context.getApiClientConfig().getApiParentURL());
		
		// 设置Content-Type
		apiRequestModel.setContentType("application/x-www-form-urlencoded;charset=UTF-8");
		
		// 构建公共参数
		Map<String, String> formParams = new HashMap<>();
		formParams.put("method", context.getIApiDefinition().getApiURI());
		formParams.put("partner_id", configModel.getPartnerId());
		formParams.put("timestamp", new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()));
		formParams.put("charset", configModel.getCharset());
		formParams.put("sign_type", configModel.getSignType());
		if (StringUtils.isNotBlank(configModel.getNotifyUrl())) {
			formParams.put("notify_url", configModel.getNotifyUrl());
		}
		formParams.put("version", configModel.getVersion());
		
		// 构建业务参数并转JSON
// 构建业务参数并转JSON
		SerializeConfig config = new SerializeConfig();
		config.propertyNamingStrategy = PropertyNamingStrategy.SnakeCase;
		String bizContent = JSON.toJSONString(request, config);
		formParams.put("biz_content", bizContent);
		
		// 生成签名
		try {
			String sign = YsepaySignatureUtil.sign(formParams, privateKey);
			formParams.put("sign", sign);
		} catch (FsApiException e) {
			throw new FsApiException("生成签名失败", e);
		}
		
		// 设置表单参数
		apiRequestModel.setRequestForm(formParams);
		apiRequestModel.setRequest(request);
		return apiRequestModel;
	}
	
	/**
	 * 构建业务参数
	 */
	private Map<String, Object> buildBizParams(com.fshows.ysepay.request.YsepayBizRequest request) {
		Map<String, Object> bizParams = new HashMap<>();
		
		// 通过反射获取所有字段
		Field[] fields = request.getClass().getDeclaredFields();
		for (Field field : fields) {
			try {
				field.setAccessible(true);
				Object value = field.get(request);
				// 跳过null值和serialVersionUID字段
				if (value != null && !"serialVersionUID".equals(field.getName())) {
					String paramName = camelToUnderline(field.getName());
					bizParams.put(paramName, value);
				}
			} catch (IllegalAccessException e) {
				log.error("构建业务参数异常", e);
			}
		}
		
		return bizParams;
	}
	
	/**
	 * 驼峰转下划线
	 */
	private String camelToUnderline(String camelStr) {
		if (StringUtils.isEmpty(camelStr)) {
			return "";
		}
		
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < camelStr.length(); i++) {
			char c = camelStr.charAt(i);
			if (Character.isUpperCase(c)) {
				if (i > 0) {
					sb.append("_");
				}
				sb.append(Character.toLowerCase(c));
			} else {
				sb.append(c);
			}
		}
		
		return sb.toString();
	}
	
	@Override
	protected ClientInfoModel getClientInfo() {
		ClientInfoModel clientInfoModel = new ClientInfoModel();
		clientInfoModel.setClientName("银盛支付");
		clientInfoModel.setClientCode("ysepay-sdk");
		return clientInfoModel;
	}
	
	@Override
	protected YsepayBaseResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
		try {
			// 响应结果
			// JSONObject resJson = JSONObject.parseObject(apiResponseModel.getResponseBody());
		
			// 获取响应key
			String responseKey = requestContext.getIApiDefinition().getApiURI().replace(".", "_") + "_response";
			JSONObject jsonObject = JSONObject.parseObject(apiResponseModel.getResponseBody());
			LogUtil.info(log,"ysepay response jsonObject:{}, responseKey:{}" ,jsonObject, responseKey);
			String content = jsonObject.get(responseKey).toString();
			// 获取签名
			String sign = jsonObject.get("sign").toString();
			// 去除前后的双引号
			sign = sign.substring(1, sign.length() - 1);
//			// 获取业务响应内容
//			JSONObject responseJson = resJson.getJSONObject(responseKey);
			
			// 验证签名
			YsepayClientConfigModel configModel = (YsepayClientConfigModel) requestContext.getApiClientConfig();
			try {
//				// 使用原始的响应内容进行验签
//				String content = responseJson.toString();
				
				boolean verifyResult = YsepaySignatureUtil.verifyResponse(content, sign, publicKey);
				if (!verifyResult) {
					throw new FsApiException("银盛响应签名验证失败, sign=" + sign);
				}
			} catch (FsApiException e) {
				throw new FsApiException("银盛验证签名失败", e);
			}

			YsepayBaseResponse baseResponse = JSONObject.parseObject(content, YsepayBaseResponse.class);
			// 设置响应
			// 解析业务响应
			IResponseDefinition iResponseDefinition = JSON.parseObject(content,
					requestContext.getIApiDefinition().getResponseClass());
			baseResponse.setDecryptBusinessContext(iResponseDefinition);
			return baseResponse;
		} catch (Exception e) {
			throw new FsApiException("解析响应失败", e);
		}
	}
	
	@Override
	protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
		return httpRequestHandler.httpRequest(apiRequestModel, requestContext);
	}
}