package com.fshows.ysepay.util;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.fshows.sdk.core.util.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;

import com.fshows.sdk.core.exception.FsApiException;

/**
 * 银盛支付签名工具类
 */
@Slf4j
public class YsepaySignatureUtil {

    private static final String SIGN_TYPE_RSA = "RSA";
    private static final String SIGN_TYPE_SM = "SM";
    private static final String SIGN_ALGORITHMS = "SHA1WithRSA";
    private static final String CHARSET_UTF8 = "UTF-8";

    /**
     * 签名
     *
     * @param params         待签名参数
     * @param privateKeyPath 私钥路径
     * @return 签名结果
     * @throws FsApiException 签名异常
     */
    public static String sign(Map<String, String> params, String privateKeyPath) throws FsApiException {
        try {
            String content = getSignContent(params);
            LogUtil.info(log,"ysepay 待签名内容: " + content); // 添加调试日志
            
            PrivateKey privateKey = getPrivateKey(privateKeyPath);
            Signature signature = Signature.getInstance("SHA1withRSA");
            signature.initSign(privateKey);
            signature.update(content.getBytes("UTF-8"));
            byte[] signed = signature.sign();
            
            String sign = Base64.encodeBase64String(signed);
            LogUtil.info(log,"ysepay 签名结果: " + sign); // 添加调试日志
            return sign;
        } catch (Exception e) {
            LogUtil.error(log,"ysepay RSA签名失败", e);
            throw new FsApiException("RSA签名失败", e);
        }
    }
    public static String sign(Map<String, String> params,  PrivateKey privateKey ) throws FsApiException {
        try {
            String content = getSignContent(params);
            LogUtil.info(log,"ysepay 待签名内容: " + content); // 添加调试日志
            Signature signature = Signature.getInstance("SHA1withRSA");
            signature.initSign(privateKey);
            signature.update(content.getBytes("UTF-8"));
            byte[] signed = signature.sign();
            
            String sign = Base64.encodeBase64String(signed);
            LogUtil.info(log,"ysepay 签名结果: " + sign); // 添加调试日志
            return sign;
        } catch (Exception e) {
            LogUtil.error(log,"ysepay RSA签名失败", e);
            throw new FsApiException("ysepay RSA签名失败", e);
        }
    }

    /**
     * 验证签名
     *
     * @param content        待验证内容
     * @param sign           签名
     * @param publicKeyPath  公钥路径
     * @return 验证结果
     * @throws FsApiException 验证异常
     */
    public static boolean verifyResponse(String content, String sign, String publicKeyPath) throws FsApiException {
        try {
            // 打印调试信息
            // {"msg":"业务参数验证错误","code":"40003","sub_msg":"无效的业务参数对象名：biz_content, 错误数量:1 具体错误： [收款方商户号(seller_id)或对应的商户名称(seller_name)校验不通过] ,out_trade_no:20250518980897627614128480578870","sub_code":"ACQ.INVALID_PARAMETER"}
            LogUtil.info(log,"ysepay 验签内容：" + content);
            LogUtil.info(log,"ysepay 签名值：" + sign);
            
            // 获取公钥
            PublicKey publicKey = getPublicKeyFromCert(publicKeyPath);
            
            Signature signetcheck = Signature.getInstance("SHA1WithRSA");
            signetcheck.initVerify(publicKey);
            signetcheck.update(content.getBytes("UTF-8"));
            LogUtil.info(log,"ysepay 公钥算法：" + publicKey.getAlgorithm());
            LogUtil.info(log,"ysepay 公钥格式：" + publicKey.getFormat());
            return signetcheck.verify(Base64.decodeBase64(sign.getBytes("UTF-8")));
//            // 使用SHA1WithRSA算法
//            Signature signature = Signature.getInstance("SHA1WithRSA");
//            signature.initVerify(publicKey);
//
//            // 更新签名内容
//            byte[] contentBytes = content.getBytes("UTF-8");
//            signature.update(contentBytes);
//
//            // Base64解码签名
//            byte[] signBytes = Base64.decodeBase64(sign);
//
//            // 打印签名长度信息
//            System.out.println("签名长度：" + signBytes.length);

            
//            return signature.verify(signBytes);
        } catch (Exception e) {
            LogUtil.error(log,"ysepay RSA验签失败", e);
            throw new FsApiException("ysepay RSA验签失败", e);
        }
    }
    
    /**
     * 验证签名
     *
     * @param content        待验证内容
     * @param sign           签名
     * @return 验证结果
     * @throws FsApiException 验证异常
     */
    public static boolean verifyResponse(String content, String sign, PublicKey publicKey ) throws FsApiException {
        try {
            // 打印调试信息
            // {"msg":"业务参数验证错误","code":"40003","sub_msg":"无效的业务参数对象名：biz_content, 错误数量:1 具体错误： [收款方商户号(seller_id)或对应的商户名称(seller_name)校验不通过] ,out_trade_no:20250518980897627614128480578870","sub_code":"ACQ.INVALID_PARAMETER"}
            LogUtil.info(log,"ysepay 验签内容：" + content);
            LogUtil.info(log,"ysepay 签名值：" + sign);
            
            Signature signetcheck = Signature.getInstance("SHA1WithRSA");
            signetcheck.initVerify(publicKey);
            signetcheck.update(content.getBytes("UTF-8"));
            LogUtil.info(log,"ysepay 公钥算法：" + publicKey.getAlgorithm());
            LogUtil.info(log,"ysepay 公钥格式：" + publicKey.getFormat());
            return signetcheck.verify(Base64.decodeBase64(sign.getBytes("UTF-8")));
//            // 使用SHA1WithRSA算法
//            Signature signature = Signature.getInstance("SHA1WithRSA");
//            signature.initVerify(publicKey);
//
//            // 更新签名内容
//            byte[] contentBytes = content.getBytes("UTF-8");
//            signature.update(contentBytes);
//
//            // Base64解码签名
//            byte[] signBytes = Base64.decodeBase64(sign);
//
//            // 打印签名长度信息
//            System.out.println("签名长度：" + signBytes.length);


//            return signature.verify(signBytes);
        } catch (Exception e) {
            LogUtil.error(log,"ysepay RSA验签失败", e);
            throw new FsApiException("ysepay RSA验签失败", e);
        }
    }

    /**
     * 获取签名内容
     *
     * @param params 参数
     * @return 待签名字符串
     */
    private static String getSignContent(Map<String, String> params) {
        if (params == null) {
            return null;
        }

        StringBuilder content = new StringBuilder();
        List<String> keys = new ArrayList<>(params.keySet());
        Collections.sort(keys);

        int index = 0;
        for (String key : keys) {
            if ("sign".equals(key)) {
                continue;
            }
            
            String value = params.get(key);
            if (value != null && StringUtils.isNotBlank(value)) {
                content.append(index == 0 ? "" : "&")
                       .append(key)
                       .append("=")
                       .append(value);
                index++;
            }
        }

        return content.toString();
    }

    /**
     * 获取私钥
     *
     * @param privateKeyPath 私钥路径
     * @return 私钥对象
     * @throws Exception 异常
     */
    public static PrivateKey getPrivateKey(String privateKeyPath) throws Exception {
        String privateKey = readKey(privateKeyPath);
        
        // 尝试使用 BouncyCastle 提供的方法来解析私钥
        try {
            // 移除PEM格式的头尾和换行符
            privateKey = privateKey.replaceAll("-----BEGIN PRIVATE KEY-----", "")
                                 .replaceAll("-----END PRIVATE KEY-----", "")
                                 .replaceAll("-----BEGIN RSA PRIVATE KEY-----", "")
                                 .replaceAll("-----END RSA PRIVATE KEY-----", "")
                                 .replaceAll("\n", "")
                                 .replaceAll("\r", "")
                                 .trim();

            // 直接使用原始的PKCS8格式
            byte[] keyBytes = Base64.decodeBase64(privateKey);
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            return keyFactory.generatePrivate(keySpec);
        } catch (Exception e) {
            // 如果解析失败，尝试直接读取文件内容
            try (FileInputStream fis = new FileInputStream(privateKeyPath)) {
                byte[] keyBytes = new byte[fis.available()];
                fis.read(keyBytes);
                PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
                KeyFactory keyFactory = KeyFactory.getInstance("RSA");
                return keyFactory.generatePrivate(keySpec);
            } catch (Exception ex) {
                throw new FsApiException("ysepay 私钥格式不正确", ex);
            }
        }
    }
    /**
     * 读取公钥，x509格式
     *
     * @return
     * @throws Exception
     * @see
     */
    public static PublicKey getPublicKeyFromCert(String publicKeyPath) throws Exception {
        InputStream ins = new FileInputStream(publicKeyPath);
        
        PublicKey pubKey = null;
        try {
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            Certificate cac = cf.generateCertificate(ins);
            pubKey = cac.getPublicKey();
        } catch (Exception e) {
            if (ins != null){
                ins.close();
            }
            throw e;
        } finally {
            if (ins != null) {
                ins.close();
            }
        }
        
        return pubKey;
    }
    /**
     * 获取公钥
     *
     * @param publicKeyPath 公钥路径
     * @return 公钥对象
     * @throws Exception 异常
     */
    private static PublicKey getPublicKey(String publicKeyPath) throws Exception {
        String publicKeyPEM = readKey(publicKeyPath);
        
        // 移除PEM格式的头尾和换行符
        String publicKeyContent = publicKeyPEM
                .replace("-----BEGIN PUBLIC KEY-----", "")
                .replace("-----END PUBLIC KEY-----", "")
                .replaceAll("\\s+", "");
        
        // Base64解码
        byte[] keyBytes = Base64.decodeBase64(publicKeyContent);
        
        // 使用RSA算法，不指定提供者
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(keyBytes);
        return keyFactory.generatePublic(keySpec);
    }

    /**
     * 读取密钥文件
     *
     * @param keyPath 密钥路径
     * @return 密钥内容
     * @throws IOException 异常
     */
    private static String readKey(String keyPath) throws IOException {
        StringBuilder sb = new StringBuilder();
        try (BufferedReader br = new BufferedReader(new FileReader(keyPath))) {
            String line;
            while ((line = br.readLine()) != null) {
                sb.append(line);
                sb.append("\n");
            }
        }
        return sb.toString();
    }
} 