/**
 * fshows.com
 * Copyright (C) 2013-2020 All Rights Reserved.
 */
package com.fshows.vbill.sdk.client;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fshows.vbill.sdk.exception.VbillApiException;
import com.fshows.vbill.sdk.request.VbillBizRequest;
import com.fshows.vbill.sdk.response.BaseVbillResponse;
import com.fshows.vbill.sdk.response.VbillBizResponse;
import com.fshows.vbill.sdk.sign.DefaultSigner;
import com.fshows.vbill.sdk.sign.Signer;
import com.fshows.vbill.sdk.util.DefaultHttpRequest;
import com.fshows.vbill.sdk.util.HttpRequest;
import com.fshows.vbill.sdk.util.ReqIdUtil;
import com.fshows.vbill.sdk.util.StringPool;
import com.fshows.vbill.sdk.util.ValidateUtil;
import com.fshows.vbill.sdk.util.VbillHashMap;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;

import java.text.MessageFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * @author wujn
 * @version DefaultVbillClient.java, v 0.1 2020-04-23 5:34 PM wujn
 */
@Slf4j
public class DefaultVbillClient implements VbillClient {
    /**
     * 请求成功标识
     */
    private static final String SUCCESS_CODE = "0000";
    /**
     * 签名对象
     */
    private Signer signer;

    /**
     * HTTP请求对象
     */
    private HttpRequest httpRequest;

    /**
     * 付呗私钥
     */
    private String fubeiPrivateKey;

    /**
     * 随行付的公钥
     */
    private String vbillPublicKey;

    public DefaultVbillClient() {

    }

    /**
     * @param signChecker     签名校验
     * @param request         请求参数
     * @param fubeiPrivateKey 私钥
     * @param vbillPublicKey  公钥
     */
    public DefaultVbillClient(Signer signChecker, HttpRequest request, String vbillPublicKey, String fubeiPrivateKey) {
        this.signer = signChecker;
        this.httpRequest = request;
        this.fubeiPrivateKey = fubeiPrivateKey;
        this.vbillPublicKey = vbillPublicKey;
    }

    /**
     * @param vbillPublicKey  私钥
     * @param fubeiPrivateKey 公钥
     */
    public DefaultVbillClient(String vbillPublicKey, String fubeiPrivateKey) {
        this(new DefaultSigner(), new DefaultHttpRequest(), vbillPublicKey, fubeiPrivateKey);
    }

    /**
     * 请求接口
     *
     * @param serverUrl 请求地址
     * @param request   请求参数
     * @param orgId     合作方机构号，类似APP_ID，由随行付颁发
     * @param reqId     请求ID，每次请求的唯一标识
     * @param version   版本号
     * @param timeout   超时时间
     * @param <T>       com.fshows.vbill.sdk.response.VbillBizResponse
     * @return com.fshows.vbill.sdk.response.BaseVbillResponse
     * @throws VbillApiException com.fshows.vbill.sdk.exception.VbillApiException
     */
    @Override
    public <T extends VbillBizResponse> BaseVbillResponse<T> execute(String serverUrl,
                                                                     VbillBizRequest<T> request,
                                                                     String orgId, String reqId,
                                                                     String version,
                                                                     boolean isSign,
                                                                     boolean isVerifySign,
                                                                     Integer timeout) throws VbillApiException {
        //1. 记录日志
        long beginTime = System.currentTimeMillis();
        log.info("【vbill-sdk】请求开始 reqId={},url={},request={},begin={}", reqId, serverUrl, JSON.toJSONString(request, SerializerFeature.PrettyFormat), beginTime);
        //2. 验证参数
        validateParam(request, serverUrl);
        if (signer == null) {
            signer = new DefaultSigner();
        }
        if (httpRequest == null) {
            httpRequest = new DefaultHttpRequest();
        }
        try {
            //3. 准备加签参数
            String requestData = getRequestData(request, orgId, reqId, version, isSign);
            //4. 请求接口
            String result = httpRequest.post(serverUrl, requestData, timeout);
            log.info("【vbill-sdk】请求结束 reqId={},url={},request={},response={},cost={}ms", reqId, serverUrl, JSON.toJSONString(request, SerializerFeature.PrettyFormat),
                    result, System.currentTimeMillis() - beginTime);
            //5. 解析返回值 todo: 写死为false方便测试，上线前一定要去掉
            return parseResponse(request, false, result);
        } catch (Exception ex) {
            throw new VbillApiException(ex.getMessage(), ex);
        }
    }

    /**
     * @param orgId   机构ID
     * @param reqId   请求ID，每次都要唯一
     * @param request 业务参数
     * @param <T>     业务参数的类型
     * @return com.fshows.vbill.sdk.response.BaseVbillResponse
     * @throws VbillApiException com.fshows.vbill.sdk.exception.VbillApiException
     */
    @Override
    public <T extends VbillBizResponse> BaseVbillResponse<T> execute(String serverUrl, VbillBizRequest<T> request, String orgId, String reqId) throws VbillApiException {
        return execute(serverUrl, request, orgId, reqId, null, true, true, 10 * 1000);
    }

    @SuppressWarnings("unchecked")
    private <T extends VbillBizResponse> BaseVbillResponse<T> parseResponse(VbillBizRequest<T> request, boolean isVerifySign, String result) throws VbillApiException {
        JSONObject jsonObject = JSON.parseObject(result);
        String code = jsonObject.getString("code");
        String msg = jsonObject.getString("msg");
        //6. 验证返回结果签名
        //如果接口没有返回0000则请求异常
        if (StringUtils.isBlank(code) || !code.equals(SUCCESS_CODE)) {
            throw new VbillApiException(MessageFormat.format("[{0}]{1}", code, msg));
        }
        if (isVerifySign) {
            String respSign = jsonObject.getString("sign");
            String respSignType = jsonObject.getString("signType");
            //签名为空或者签名类型为空，则签名异常
            if (StringUtils.isBlank(respSign) || StringUtils.isBlank(respSign)) {
                throw new VbillApiException("返回签名为空或签名类型为空，请检查签名信息");
            }
            //验证签名
            boolean signCheckResult = signer.verifySign(jsonObject.getInnerMap(), respSignType, respSign, vbillPublicKey);
            if (!signCheckResult) {
                throw new VbillApiException("签名校验失败，请检查签名信息");
            }
        }
        //7. 设置结果
        BaseVbillResponse response = new BaseVbillResponse<>();
        response.setReqId(jsonObject.getString("reqId"));
        response.setMsg(jsonObject.getString("msg"));
        response.setCode(jsonObject.getString("code"));
        response.setOrgId(jsonObject.getString("orgId"));
        response.setRespData(JSON.parseObject(jsonObject.getString("respData"), request.getResponseClass()));
        return response;
    }

    private <T extends VbillBizResponse> void validateParam(VbillBizRequest<T> request, String serverUrl) {
        if (request == null) {
            throw new IllegalArgumentException("接口请求参数不能为空");
        }
        if (StringUtils.isBlank(fubeiPrivateKey)) {
            throw new IllegalArgumentException("私钥不能为空");
        }
        if (StringUtils.isBlank(vbillPublicKey)) {
            throw new IllegalArgumentException("公钥不能为空");
        }
        if (StringUtils.isBlank(serverUrl)) {
            throw new IllegalArgumentException("api请求地址不能为空");
        }
        //注解验证
        ValidateUtil.validateWithThrow(request);
    }

    private <T extends VbillBizResponse> String getRequestData(VbillBizRequest<T> request, String orgId, String reqId, String version, boolean isSign) {
        Map<String, String> signMap = new VbillHashMap();
        signMap.put("orgId", orgId);
        signMap.put("reqId", StringUtils.isBlank(reqId) ? ReqIdUtil.getId() : reqId);
        signMap.put("timestamp", DateFormatUtils.format(new Date(), StringPool.DEFAULT_TIMESTAMP_FORMAT));
        signMap.put("version", StringUtils.isBlank(version) ? "1.0" : version);
        signMap.put("signType", StringPool.DEFAULT_SIGN_TYPE);
        signMap.put("reqData", JSON.toJSONString(request));
        if (isSign) {
            String sign = signer.sign(signMap, StringPool.DEFAULT_SIGN_TYPE, fubeiPrivateKey);
            signMap.put("sign", sign);
        }
        Map<String, Object> param = new HashMap<>(16);
        param.putAll(signMap);
        // 随行付接口接收的reqData为“json对象”而为“json格式的字符串”
        param.put("reqData", request);
        return JSON.toJSONString(param);
    }

}
