/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.umpay.sdk.request.trade.pay;

import com.fshows.umpay.sdk.request.UmBizRequest;
import com.fshows.umpay.sdk.request.trade.pay.item.UmpayAlipayExtendParamsRequest;
import com.fshows.umpay.sdk.response.trade.pay.UmpayScanCardResponse;
import lombok.Data;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;

/**
 * @author
 * @version UmpayScanCardRequest.java, v 0.1 2022-03-16 11:58:42
 */
@Data
public class UmpayScanCardRequest extends UmBizRequest<UmpayScanCardResponse> {

    private static final long serialVersionUID = -2213838555238145061L;


    /**
     * 聚合收单平台为商户分配的唯一ID,商户入驻后由平台返回
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 20, message = "storeId长度不能超过20")
    private String storeId;

    /**
     * 服务商订单号,64个字符以内。可包含字母、数字、下划线;需保证在接入的服务商系统中不重复
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 64, message = "outTradeNo长度不能超过64")
    private String outTradeNo;

    /**
     * 订单标题。示例：Iphone 16G
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 256, message = "subject长度不能超过256")
    private String subject;

    /**
     * 商品描述商品或支付单简单描述，格式要求：门店品牌名-城市分店名-实际商品名称示例:image-深圳腾大- QQ
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 128, message = "body长度不能超过128")
    private String body;

    /**
     * 订单总金额，单位为元，精确到小数点后两位，取值范围[0.01,100000000]
     * <p>
     * (必填)
     */
    @NotNull(message = "订单总金额不能为空")
    private BigDecimal totalAmount;

    /**
     * 商户终端费率，精确到小数点后5位注：银联二维码1000元以上不支持该接口交易时上送费率，请调用后补上账接口（敬请期待）
     * <p>
     * (选填)
     */
    private BigDecimal merchantRate;

    /**
     * 可使用指定的微信、支付宝子商户号进行交易，子商户号多渠道号模式的情况下必填
     * <p>
     * (特殊必填)
     */
    @Length(max = 32, message = "subMerchantNo长度不能超过32")
    private String subMerchantNo;

    /**
     * 交易补贴金额，单位元下游服务商给商户的交易补贴，未开通补贴金功能时传0
     * <p>
     * (必填)
     */
    @NotNull(message = "交易补贴金额不能为空")
    private BigDecimal subsidyFee;

    /**
     * 支付授权码，用户的付款码。根据支付授权码自动识别支付通道，目前支持微信、支付宝、银联二维码
     * <p>
     * (必填)
     */
    @NotBlank(message = "支付授权码不能为空")
    @Length(max = 64, message = "authCode长度不能超过64")
    private String authCode;

    /**
     * 支付成功后回调地址，如果不填会用聚合收单平台默认的地址
     * <p>
     * (选填)
     */
    private String notifyUrl;

    /**
     * 终端IP，调用支付API 的机器IP
     * <p>
     * (选填)
     */
    @Length(max = 16, message = "spbillCreateIp长度不能超过16")
    private String spbillCreateIp;

    /**
     * 终端设备号，可由服务商自行定义
     * <p>
     * (必填)
     */
    @NotBlank(message = "终端设备号不能为空")
    @Length(max = 64, message = "deviceInfo长度不能超过64")
    private String deviceInfo;

    /**
     * 订单优惠标记透传给微信，代金券或立减优惠功能的参数（使用单品券时必传）
     * <p>
     * (特殊必填)
     */
    @Length(max = 32, message = "goodsTag长度不能超过32")
    private String goodsTag;

    /**
     * 支付宝业务拓展参数--花呗分期
     * <p>
     * (选填)
     */
    private UmpayAlipayExtendParamsRequest alipayExtendParams;


    /**
     * 该笔交易需禁用的支付渠道，多个支付渠道需要用英文逗号,分开；注意该字段与enable_pay_channels互斥示例：pcredit,moneyFund仅对支付宝交易有效，支付宝支付渠道详见&lt;支付渠道列表&gt;
     * <p>
     * (选填)
     */
    @Length(max = 128, message = "disablePayChannels长度不能超过128")
    private String disablePayChannels;

    /**
     * 该笔交易可用的支付渠道，多个支付渠道需要用英文逗号,分开；注意该字段与disable_pay_channels互斥示例：pcredit,moneyFund仅对支付宝交易有效，支付宝支付渠道详见&lt;支付渠道列表&gt;
     * <p>
     * (选填)
     */
    @Length(max = 128, message = "enablePayChannels长度不能超过128")
    private String enablePayChannels;

    /**
     * 微信禁用信用卡该字段传no_credit
     * <p>
     * (选填)
     */
    @Length(max = 10, message = "limitPay长度不能超过10")
    private String limitPay;

    /**
     * 分账标识；只有分账订单才可调用&lt;订单分账&gt;接口进行分账0-无需分账；1-分账订单；默认0-无需分账；
     * <p>
     * (选填)
     */
    private Integer shareOrderType;

    /**
     * 订单失效时间；格式：yyyyMMddHHmmss；
     * <p>
     * (选填)
     */
    @Length(max = 14, message = "timeExpire长度不能超过14")
    private String timeExpire;

    /**
     * 是否刷脸支付：默认扫码，0：扫码1：刷脸
     * <p>
     * (选填)
     */
    @Length(max = 2, message = "facepay长度不能超过2")
    private String facepay;

    /**
     * 销售产品码：FACE_TO_FACE_PAYMENT
     * <p>
     * (选填)
     */
    @Length(max = 32, message = "productCode长度不能超过32")
    private String productCode;

    /**
     * 门店编号（微信支付宝的门店编号）针对微信、支付宝有效，透传至银联。微信：数字、字母、下划线格式
     * <p>
     * (选填)
     */
    @Length(max = 32, message = "storeNo长度不能超过32")
    private String storeNo;

    /**
     * 终端设备号；不传取默认终端设备号；
     * <p>
     * (特殊必填)
     */
    @Length(max = 8, message = "terminalId长度不能超过8")
    private String terminalId;

    /**
     * 银行卡受理终端产品应用认证编号。该编号由“中国银联标识产品企业资质认证办公室”为通过入网认证的终端进行分配。银联直连终端必填
     * <p>
     * (选填)
     */
    @Length(max = 5, message = "networkLicense长度不能超过5")
    private String networkLicense;

    /**
     * （付款 APP）设备 GPS 位置，格式为纬度/经度，+表示北纬、东经，-表示南纬、西经。
     * <p>
     * (选填)
     */
    @Length(max = 32, message = "location长度不能超过32")
    private String location;
    /**
     * 商品详情
     * 该字段为单品营销所需上送，微信支付宝单品营销通用字段。
     * 微信上送格式：
     * 支付宝上送格式：
     */
    @Length(max = 8000, message = "goodsDetail长度不能超过8000")
    private String goodsDetail;
    /**
     * 参与优惠计算的金额，单位为元，精确到小数点后两位，取值范围[0.01,100000000]。 如果该值未传入，但传入了【订单总金额】和【不可打折金额】，则该值默认为【订单总金额】-【不可打折金额】
     */
    private BigDecimal discountableAmount;
    /**
     * 不参与优惠计算的金额，单位为元，精确到小数点后两位，取值范围 [0.01,100000000]。如果该值未传入，但传入了【订单总金额】和【可打折金额】，则该值默认为【订单总金额】-【可打折金额】
     */
    private BigDecimal undiscountableAmount;
    /**
     * 商户传入业务信息，应用于安全，营销等参数直传场景，格式为json格式。
     */
    @Length(max = 512, message = "businessParams长度不能超过512")
    private String businessParams;

    @Override
    public Class<UmpayScanCardResponse> getResponseClass() {
        return UmpayScanCardResponse.class;
    }
}