/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.umpay.sdk.request.trade.pay;

import com.fshows.umpay.sdk.request.UmBizRequest;
import com.fshows.umpay.sdk.request.trade.pay.item.UmpayAlipayExtendParamsRequest;
import com.fshows.umpay.sdk.response.trade.pay.UmpayPrePayResponse;
import lombok.Data;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;

/**
 * 统一下单请求参数
 *
 * @author
 * @version UmpayPrePayRequest.java, v 0.1 2022-03-16 13:35:53
 */
@Data
public class UmpayPrePayRequest extends UmBizRequest<UmpayPrePayResponse> {

    private static final long serialVersionUID = -2213838555692096913L;


    /**
     * 聚合收单平台为商户分配的惟一 ID,商户入驻后由平台返回
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 20, message = "storeId长度不能超过20")
    private String storeId;

    /**
     * 服务商订单号,64个字符以内。可包含字母、数字、下划线;需保证在接入的服务商系统中不重复
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 64, message = "outTradeNo长度不能超过64")
    private String outTradeNo;

    /**
     * 订单总金额，单位为元，精确到小数点后两位，取值范围[0.01,100000000]
     * <p>
     * (必填)
     */
    @NotNull(message = "订单总金额不能为空")
    private BigDecimal totalAmount;

    /**
     * 商户终端费率，精确到小数点后5位注：银联二维码1000元以上不支持该接口交易时上送费率，请调用后补上账接口
     * <p>
     * (选填)
     */
    private BigDecimal merchantRate;

    /**
     * 交易补贴金额，单位元下游服务商给商户的交易补贴，若没有补贴业务，传0
     * <p>
     * (必填)
     */
    @NotNull(message = "交易补贴金额不能为空")
    private BigDecimal subsidyFee;

    /**
     * 支付平台：1-支付宝；2-微信；6-云闪付；
     * <p>
     * (必填)
     */
    @NotBlank(message = "支付平台不能为空")
    @Length(max = 1, message = "payPlatformType长度不能超过1")
    private String payPlatformType;

    /**
     * 订单类型：1-条码支付；2-native，原生扫码支付（支持支付宝、银联二维码）；3-JSAPI，公众号支付，服务窗支付；4-小程序支付；6-云闪付JS支付
     * <p>
     * (必填)
     */
    @NotBlank(message = "订单类型不能为空")
    @Length(max = 1, message = "orderType长度不能超过1")
    private String orderType;

    /**
     * 可使用指定的微信、支付宝子商户号进行交易，一个商户有多个子商户号的情况下必填
     * <p>
     * (特殊必填)
     */
    @Length(max = 32, message = "subMerchantNo长度不能超过32")
    private String subMerchantNo;

    /**
     * 微信公众号ID ；微信公众号支付的公众号id或微信小程序支付的小程序appid;如果传了会使用此appid进行下单；没传会使用商户进件时最新配置的appid
     * <p>
     * (选填)
     */
    @Length(max = 32, message = "subAppid长度不能超过32")
    private String subAppid;

    /**
     * 付款用户id，“微信sub_openid”，“支付宝 user_id”，“云闪付user_id”微信公众号、小程序、支付宝服务窗、支付宝小程序、云闪付JS支付必填（订单类型3和4）
     * <p>
     * (特殊必填)
     */
    @Length(max = 128, message = "userId长度不能超过128")
    private String userId;

    /**
     * 订单标题。示例：Iphone 16G
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 256, message = "subject长度不能超过256")
    private String subject;

    /**
     * 商品描述商品或支付单简单描述，格式要求：门店品牌名-城市分店名-实际商品名称Eg:image-深圳腾大- QQ
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 128, message = "body长度不能超过128")
    private String body;

    /**
     * 支付成功后回调地址，如果不填会用聚合收单平台默认的地址
     * <p>
     * (选填)
     */
    private String notifyUrl;

    /**
     * APP和网页支付提交用户端ip；
     * <p>
     * (选填)
     */
    @Length(max = 16, message = "spbillCreateIp长度不能超过16")
    private String spbillCreateIp;

    /**
     * 订单优惠标记透传给微信，代金券或立减优惠功能的参数（使用单品券时必传）
     * <p>
     * (特殊必填)
     */
    @Length(max = 32, message = "goodsTag长度不能超过32")
    private String goodsTag;

    /**
     * 公众号、小程序支付成功后的跳转地址例: "https://www.baidu.com/"
     * <p>
     * (选填)
     */
    private String callbackUrl;

    /**
     * 公众号、小程序支付过程中用户取消支付或支付失败跳转地址
     * <p>
     * (选填)
     */
    private String cancelUrl;

    /**
     * 拉起支付方式1-调用微信支付宝官方接口方式；
     * <p>
     * (特殊必填)
     */
    private Integer pullUpType;

    /**
     * 支付宝业务拓展参数--花呗分期
     * <p>
     * (选填)
     */
    private UmpayAlipayExtendParamsRequest alipayExtendParams;

    /**
     * 该笔交易需禁用的支付渠道，多个支付渠道需要用英文逗号,分开；注意该字段与enable_pay_channels互斥示例：pcredit,moneyFund支付宝支付渠道详见&lt;支付渠道列表&gt;
     * <p>
     * (选填)
     */
    @Length(max = 128, message = "disablePayChannels长度不能超过128")
    private String disablePayChannels;

    /**
     * 该笔交易可用的支付渠道，多个支付渠道需要用英文逗号,分开；注意该字段与disable_pay_channels互斥示例：pcredit,moneyFund支付宝支付渠道详见&lt;支付渠道列表&gt;
     * <p>
     * (选填)
     */
    @Length(max = 128, message = "enablePayChannels长度不能超过128")
    private String enablePayChannels;

    /**
     * 分账标识；只有分账订单才可调用&lt;分账&gt;接口进行分账0-无需分账；1-分账订单；默认0-无需分账；
     * <p>
     * (选填)
     */
    private Integer shareOrderType;

    /**
     * 订单失效时间；格式：yyyyMMddHHmmss
     * <p>
     * (选填)
     */
    @Length(max = 14, message = "timeExpire长度不能超过14")
    private String timeExpire;

    /**
     * 微信禁用信用卡该字段传no_credit
     * <p>
     * (选填)
     */
    @Length(max = 10, message = "limitPay长度不能超过10")
    private String limitPay;

    /**
     * 门店编号（微信支付宝的门店编号）针对微信、支付宝有效，透传至银联。微信：数字、字母、下划线格式
     * <p>
     * (选填)
     */
    @Length(max = 32, message = "storeNo长度不能超过32")
    private String storeNo;

    /**
     * 终端设备号；不传取默认终端设备号；
     * <p>
     * (特殊必填)
     */
    @Length(max = 8, message = "terminalId长度不能超过8")
    private String terminalId;

    /**
     * 银行卡受理终端产品应用认证编号。该编号由“中国银联标识产品企业资质认证办公室”为通过入网认证的终端进行分配。银联直连终端必填
     * <p>
     * (选填)
     */
    @Length(max = 5, message = "networkLicense长度不能超过5")
    private String networkLicense;

    /**
     * （付款 APP）设备 GPS 位置，格式为纬度/经度，+表示北纬、东经，-表示南纬、西经。
     * <p>
     * (选填)
     */
    @Length(max = 32, message = "location长度不能超过32")
    private String location;
    /**
     * 参与优惠计算的金额，单位为元，精确到小数点后两位，取值范围[0.01,100000000]。 如果该值未传入，但传入了【订单总金额】和【不可打折金额】，则该值默认为【订单总金额】-【不可打折金额】
     */
    private BigDecimal discountableAmount;
    /**
     * 不参与优惠计算的金额，单位为元，精确到小数点后两位，取值范围 [0.01,100000000]。如果该值未传入，但传入了【订单总金额】和【可打折金额】，则该值默认为【订单总金额】-【可打折金额】
     */
    private BigDecimal undiscountableAmount;
    /**
     * 商品详情
     * 该字段为单品营销所需上送，微信支付宝单品营销通用字段。
     * 微信上送格式：
     * 支付宝上送格式：
     */
    @Length(max = 8000, message = "goodsDetail长度不能超过8000")
    private String goodsDetail;
    /**
     * 商户传入业务信息，应用于安全，营销等参数直传场景，格式为json格式。
     */
    @Length(max = 512, message = "businessParams长度不能超过512")
    private String businessParams;

    @Override
    public Class<UmpayPrePayResponse> getResponseClass() {
        return UmpayPrePayResponse.class;
    }
}