/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.umpay.sdk.client.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.umpay.sdk.client.UmBaseClient;
import com.fshows.umpay.sdk.exception.UmPayException;
import com.fshows.umpay.sdk.request.UmBizRequest;
import com.fshows.umpay.sdk.response.UmBaseResponse;
import com.fshows.umpay.sdk.util.FsHttpUtil;
import com.fshows.umpay.sdk.util.SignUtil;
import com.fshows.umpay.sdk.util.StringPool;
import com.fshows.umpay.sdk.util.ValidateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

import java.util.HashMap;
import java.util.Map;

/**
 * @author youmingming
 * @version UmBaseClientImpl.java, v 0.1 2022-02-09 下午11:06 youmingming
 */
@Slf4j
public class UmBaseClientImpl implements UmBaseClient {
    /**
     * 版本号
     */
    private static final String VERSION = "1.0";
    /**
     * 应用appId;
     */
    private String appId;
    /**
     * 私钥
     */
    private String fbPrivateKey;
    /**
     * 公钥
     */
    private String umPublicKey;

    private static SerializeConfig SNAKE_CASE_CONFIG = new SerializeConfig();

    static {
        SNAKE_CASE_CONFIG.propertyNamingStrategy = PropertyNamingStrategy.SnakeCase;
    }

    public UmBaseClientImpl(String appId, String fbPrivateKey, String umPublicKey) {
        this.appId = appId;
        this.fbPrivateKey = fbPrivateKey;
        this.umPublicKey = umPublicKey;
    }

    /**
     * 请求联动接口
     *
     * @param request            请求参数
     * @param umpayApiDefinition 方法枚举
     * @param postUrl            请求地址
     * @param <R>                返参
     * @return UmBaseResponse
     */
    @Override
    public <R> UmBaseResponse<R> excute(UmBizRequest<R> request, UmpayApiDefinition umpayApiDefinition, String postUrl) throws UmPayException {
        String method = umpayApiDefinition.getMethod();
        //验证参数
        validateParam(request, method);

        //获取开始时间
        final long startTime = System.currentTimeMillis();

        //获取请求参数
        final Map<String, String> requestData = getRequestData(request, method, true);
        log.info("【umpay-sdk】接口调用开始 >> url={},method={},request={}", postUrl, method, requestData);

        try {
            final String result = FsHttpUtil.post(postUrl, requestData);
            log.info("【umpay-sdk】接口调用结束 >> url={},method={},request={},response={},cost={}", postUrl, method, requestData, result, System.currentTimeMillis() - startTime);
            return parseResponse(result, request);
        } catch (Exception ex) {
            log.error("【umpay-sdk】接口调用失败 >> url={},method={},request={},ex={},cost={}", postUrl, method, requestData, ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            throw new UmPayException(ex.getMessage());
        }
    }

    /**
     * 参数校验
     *
     * @param request 请求参数
     * @param method  请求方法
     * @param <R>
     */
    private <R> void validateParam(UmBizRequest<R> request, String method) {
        if (request == null) {
            throw new IllegalArgumentException("接口请求参数不能为空");
        }
        if (StringUtils.isBlank(fbPrivateKey)) {
            throw new IllegalArgumentException("私钥不能为空");
        }
        if (StringUtils.isBlank(umPublicKey)) {
            throw new IllegalArgumentException("公钥不能为空");
        }
        if (StringUtils.isBlank(method)) {
            throw new IllegalArgumentException("请求方法不能为空");
        }
        //注解验证
        ValidateUtil.validateWithThrow(request);
    }

    /**
     * 获取签名参数
     *
     * @param request 请求参数
     * @param method  方法名
     * @param isSign  是否签名
     * @param <R>     返参
     * @return Map<String, String>
     */
    private <R> Map<String, String> getRequestData(UmBizRequest<R> request, String method, boolean isSign) {
        Map<String, String> signMap = new HashMap<>(16);
        signMap.put("app_id", appId);
        signMap.put("method", method);
        signMap.put("version", VERSION);
        signMap.put("content", JSON.toJSONString(request, SNAKE_CASE_CONFIG));
        if (isSign) {
            String sign = SignUtil.sign(signMap, StringPool.DEFAULT_SIGN_TYPE, fbPrivateKey);
            signMap.put("sign", sign);
        }
        return signMap;
    }

    /**
     * 结果解析
     *
     * @param result  返回结果
     * @param request 请求参数
     * @param <R>     结果
     * @return UmBaseResponse
     */
    @SuppressWarnings("unchecked")
    private <R> UmBaseResponse<R> parseResponse(String result, UmBizRequest<R> request) {
        JSONObject jsonObject = JSON.parseObject(result);
        //7. 设置结果
        UmBaseResponse response = new UmBaseResponse<>();
        response.setSuccess(jsonObject.getBoolean("success"));
        response.setErrorCode(jsonObject.getString("error_code"));
        response.setErrorMessage(jsonObject.getString("error_message"));
        response.setReturnValue(JSON.parseObject(jsonObject.getString("return_value"), request.getResponseClass()));
        return response;
    }
}