/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.fshows.sdk.sf.api;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fshows.sdk.sf.api.enums.SfResultCodeEnum;
import com.fshows.sdk.sf.api.utils.FsRequestUtil;
import com.fshows.sdk.sf.api.utils.LogUtil;
import com.fshows.sdk.sf.api.utils.Misc;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.util.List;

/**
 * @author CoderMa
 * @version DefaultElemeClient.java, v 0.1 2020-04-01 09:46 CoderMa
 */
@Slf4j
@Data
public class DefaultSfClient implements SfClient {

    private String serverUrl;

    private String devid;

    private String devKey;

    public DefaultSfClient(String serverUrl, String devid, String devKey) {
        this.serverUrl = serverUrl;
        this.devid = devid;
        this.devKey = devKey;
    }

    @Override
    public <T extends SfResponse<SfResponse>> T execute(SfRequest<T> request) {
        SfResponse<SfResponse> sfResponse = executeRequest(request);
        T t1 = JSON.parseObject(JSONObject.toJSONString(sfResponse), request.getResponseClazz());
        T t2 = JSON.parseObject(sfResponse.getResult(), request.getResponseClazz());
        t1.setData(t2);
        t1.setResult(null);
        return t1;
    }

    @Override
    public <T extends SfResponse<SfResponse>> String executeString(SfRequest<T> request) {
        return executeRequest(request).getResult();
    }

    @Override
    public <T extends SfResponse<SfResponse>> List<T> executeArray(SfRequest<T> request) {
        String data = executeString(request);
        List<T> response = JSON.parseArray(data, request.getResponseClazz());
        return response;
    }

    public SfResponse<SfResponse> executeRequest(SfRequest request) {
        JSONObject json = JSON.parseObject(request.getBusinessParam(), JSONObject.class);
        json.put("dev_id", this.devid);

        //加签
        String sign = Signer.sign(json.toString(), devid, devKey);
        String response = null;
        try {
            response = FsRequestUtil.postByJson(serverUrl + request.getApiMethodName().toLowerCase() + "?sign=" + sign, json.toString(), 10000);
        } catch (IOException e) {
            LogUtil.error(log, "顺丰接口 >> 调用异常 >> param :{}", request);
            return new SfResponse<SfResponse>(SfResultCodeEnum.SF_IO_ERROR);
        } catch (Exception e) {
            LogUtil.error(log, "顺丰接口 >> 异常 >> error >> param :{}", request);
            return new SfResponse<SfResponse>(SfResultCodeEnum.SF_ERROR);
        }
        SfResponse<SfResponse> sfResponse = null;
        try {
            sfResponse = JSON.parseObject(response, SfResponse.class);
        } catch (Exception e) {
            LogUtil.error(log, "顺丰接口 >> 序列化异常 >> param :{}", request);
            return new SfResponse<SfResponse>(SfResultCodeEnum.SF_SERIAL_ERROR);
        }

        if (Misc.isNull(sfResponse)) {
            sfResponse = new SfResponse<SfResponse>(SfResultCodeEnum.RESULT_IS_NULL);
            return sfResponse;
        }

        return sfResponse;
    }

}