package com.fshows.msfpay.client;

import com.fshows.msfpay.utils.MsfpayUtil;
import msfpay.cfca.sadk.util.Base64;
import msfpay.cfca.sadk.util.CertUtil;
import msfpay.cfca.sadk.x509.certificate.X509Cert;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.parser.Feature;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fshows.msfpay.apienum.MsfpayPayApiDefinitionEnum;
import com.fshows.msfpay.client.model.MsfpayPayClientConfigModel;
import com.fshows.msfpay.constant.MsfpayConstant;
import com.fshows.msfpay.request.MsfpayBizRequest;
import com.fshows.msfpay.response.MsfpayBaseResponse;
import com.fshows.msfpay.utils.SM2Utils;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.definition.IResponseDefinition;
import com.fshows.sdk.core.client.base.handler.IHttpRequestHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.security.PrivateKey;
import java.util.HashMap;
import java.util.Map;

/**
 * 中投科信支付API客户端
 */
@Slf4j
public class MsfpayPayApiClient extends AbstractApiClient<MsfpayBizRequest, MsfpayBaseResponse, MsfpayPayApiDefinitionEnum> {

    private PrivateKey merchantSignPrivateKey;
    private PrivateKey merchantEncPrivateKey;
    private X509Cert platformEncCert;
    private X509Cert platformSignCert;
    private X509Cert merchantEncCert;
    private X509Cert merchantSignCert;


    /**
     * 请求执行器
     */
    protected IHttpRequestHandler httpRequestHandler = new com.fshows.sdk.core.client.component.http.FromHttpRequestHandler();

    public MsfpayPayApiClient(MsfpayPayClientConfigModel apiClientConfig) throws Exception {
        super(apiClientConfig);
        initCertificatesAndKeys(apiClientConfig);
    }

    private void initCertificatesAndKeys(MsfpayPayClientConfigModel config) throws Exception {
        try {
            // 1. 加载商户签名私钥
            merchantSignPrivateKey = SM2Utils.getPrivateKeyFromSM2(config.getMerchantSignPath(), config.getMerchantSignPwd());

            // 2. 加载商户加密私钥
            merchantEncPrivateKey = SM2Utils.getPrivateKeyFromSM2(config.getMerchantEncPath(), config.getMerchantEncPwd());

            // 3. 加载平台加密证书
            platformEncCert = MsfpayUtil.getX509Cert(config.getPlatformEncCertPath());

            // 4. 加载平台签名证书
            platformSignCert = MsfpayUtil.getX509Cert(config.getPlatformSignCertPath());

            merchantEncCert = CertUtil.getCertFromSM2(config.getMerchantEncPath());

            merchantSignCert = CertUtil.getCertFromSM2(config.getMerchantSignPath());

            log.info("证书和密钥初始化成功");
        } catch (Exception e) {
            log.error("初始化证书和密钥失败", e);
            throw new Exception("初始化证书和密钥失败", e);
        }
    }

    @Override
    public MsfpayBaseResponse execute(MsfpayBizRequest request, MsfpayPayApiDefinitionEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public MsfpayBaseResponse execute(MsfpayBizRequest request, MsfpayPayApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

    @Override
    protected MsfpayBaseResponse doExecute(MsfpayBizRequest request, MsfpayPayApiDefinitionEnum iApiDefinition, DefaultClientConfigModel customConfig) throws FsApiException {
        long beginTime = System.currentTimeMillis();
        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, customConfig);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;

        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);

            LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}", getClientInfo().getClientDesc(), iApiDefinition, JSONObject.toJSONString(apiRequestModel.getParamMap()));
            // 请求开始时间
            long reqBeginTime = System.currentTimeMillis();
            // 执行post请求
            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            // 请求结束时间
            long reqEndTime = System.currentTimeMillis();

            // 构建响应数据
            MsfpayBaseResponse response = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);

            LogUtil.info(log, "{} >> 执行请求结束 >> url={}, method={}, request={}, response={}, totalcost={}ms, reqcost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), requestContext.getIApiDefinition(), JSONObject.toJSONString(apiRequestModel.getParamMap()),
                    JSONObject.toJSONString(response), System.currentTimeMillis() - beginTime, reqEndTime - reqBeginTime);

            return response;
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求业务异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, JSONObject.toJSONString(request), JSONObject.toJSONString(apiRequestModel), JSONObject.toJSONString(apiResponseModel));
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求未知异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, JSONObject.toJSONString(request), JSONObject.toJSONString(apiRequestModel), JSONObject.toJSONString(apiResponseModel));
            throw new FsApiException("请求未知异常", e);
        }
    }

    @Override
    protected DefaultRequestContext buildRequestContext(MsfpayPayApiDefinitionEnum iApiDefinition, MsfpayBizRequest request, DefaultClientConfigModel customConfig) {
        DefaultRequestContext context = new DefaultRequestContext();
        context.setIApiDefinition(iApiDefinition);

        MsfpayPayClientConfigModel config = new MsfpayPayClientConfigModel();
        MsfpayPayClientConfigModel clientConfig = (MsfpayPayClientConfigModel) this.apiClientConfig;

        config.setApiParentURL(clientConfig.getApiParentURL());
        config.setCharset(clientConfig.getCharset());
        config.setSignType(clientConfig.getSignType());
        config.setVersion(clientConfig.getVersion());
        config.setPlatmerid(clientConfig.getPlatmerid());
        config.setTimeout(clientConfig.getTimeout());

        context.setApiClientConfig(config);
        context.setClientInfoModel(getClientInfo());
        return context;
    }

    @Override
    protected ApiRequestModel buildApiRequestModel(MsfpayBizRequest request, DefaultRequestContext context) {
        MsfpayPayClientConfigModel configModel = (MsfpayPayClientConfigModel) context.getApiClientConfig();
        MsfpayPayApiDefinitionEnum iApiDefinition = (MsfpayPayApiDefinitionEnum) context.getIApiDefinition();

        ApiRequestModel apiRequestModel = new ApiRequestModel();
        apiRequestModel.setApiURL(configModel.getApiParentURL());
        apiRequestModel.setContentType(MsfpayConstant.Http.CONTENT_TYPE_FORM);

        try {
            fillCommonHead(request, configModel, iApiDefinition);
            apiRequestModel.setRequest(request);

            Map<String, String> messageData = new HashMap<>();
            Map<String, Object> dataMap = new HashMap<>();
            Map<String, String> headMap = new HashMap<>();
            Map<String, Object> bodyMap = new HashMap<>();

            headMap.put(MsfpayConstant.HeadField.MERTRANDATE, request.getMertrandate());
            headMap.put(MsfpayConstant.HeadField.MERTRANTIME, request.getMertrantime());
            headMap.put(MsfpayConstant.HeadField.PLATMERID, request.getPlatmerid());
            headMap.put(MsfpayConstant.HeadField.TRANCODE, request.getTrancode());
            headMap.put(MsfpayConstant.HeadField.TRANFLOW, request.getTranflow());
            headMap.put(MsfpayConstant.HeadField.ZTPAGE, request.getZtpage());
            headMap.put(MsfpayConstant.HeadField.RESPCODE, "");
            headMap.put(MsfpayConstant.HeadField.RESPMSG, "");


            String requestJson = JSON.toJSONString(request);
            Map<String, Object> requestMap = JSON.parseObject(requestJson, Map.class);
            requestMap.remove(MsfpayConstant.HeadField.PLATMERID);
            requestMap.remove(MsfpayConstant.HeadField.TRANFLOW);
            requestMap.remove(MsfpayConstant.HeadField.TRANCODE);
            requestMap.remove(MsfpayConstant.HeadField.MERTRANDATE);
            requestMap.remove(MsfpayConstant.HeadField.MERTRANTIME);
            requestMap.remove(MsfpayConstant.HeadField.ZTPAGE);
            bodyMap.putAll(requestMap);

            dataMap.put(MsfpayConstant.MessageField.HEAD, headMap);
            dataMap.put(MsfpayConstant.MessageField.BODY, bodyMap);

            String dataJson = JSON.toJSONString(dataMap, SerializerFeature.WriteNullStringAsEmpty);
            String sign = sign(dataJson);

            messageData.put(MsfpayConstant.MessageField.DATA, dataJson);
            messageData.put(MsfpayConstant.MessageField.SIGN, sign);

            Map<String, Object> message = new HashMap<>();
            message.put(MsfpayConstant.MessageField.MESSAGE, messageData);

            apiRequestModel.setParamMap(message);

            String messageJson = JSON.toJSONString(message,SerializerFeature.WriteNullStringAsEmpty);
            byte[] envelopdata = SM2Utils.envelopeMessage(messageJson, MsfpayConstant.Encryption.SM4_ECB_PKCS7, platformEncCert);

            Map<String,String> paramMap = new HashMap<String,String>();
            paramMap.put(MsfpayConstant.RequestField.FORMAT_TYPE, MsfpayConstant.RequestValue.FORMAT_JSON);
            paramMap.put(MsfpayConstant.RequestField.CRYPT_TYPE, MsfpayConstant.RequestValue.CRYPT_SM2);
            paramMap.put(MsfpayConstant.RequestField.SIGN_TYPE, MsfpayConstant.RequestValue.SIGN_SM2);
            paramMap.put(MsfpayConstant.RequestField.DATA, new String(envelopdata));

            apiRequestModel.setRequestForm(paramMap);
        } catch (Exception e) {
            throw new FsApiException("构建请求失败", e);
        }

        return apiRequestModel;
    }

    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientName(MsfpayConstant.ClientInfo.CLIENT_NAME);
        clientInfoModel.setClientCode(MsfpayConstant.ClientInfo.CLIENT_CODE);
        return clientInfoModel;
    }

    @Override
    protected MsfpayBaseResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        try {
            String responseBody = apiResponseModel.getResponseBody();

            byte[] decryptedData = SM2Utils.openEvelopedMessage(responseBody,
                    merchantEncPrivateKey,
                    merchantEncCert,
                    SM2Utils.session);

            String msg = new String(decryptedData, MsfpayConstant.Http.CHARSET_UTF8);

            JSONObject Obj = JSON.parseObject(msg, Feature.OrderedField);
            String messageStr = Obj.getString(MsfpayConstant.ResponseField.MESSAGE);
            JSONObject messageObj = JSON.parseObject(messageStr,Feature.OrderedField);
            String dataStr = messageObj.getString(MsfpayConstant.MessageField.DATA);
            JSONObject body = JSON.parseObject(dataStr,Feature.OrderedField);
            String bodyStr = body.getString(MsfpayConstant.ResponseField.BODY);

            String signStr = messageObj.getString(MsfpayConstant.MessageField.SIGN);

            boolean verify = SM2Utils.P7VerifyMessageDetach(
                    new String(Base64.encode(dataStr.getBytes(MsfpayConstant.Http.CHARSET_UTF8))),
                    signStr,
                    platformSignCert,
                    SM2Utils.session);

            if(!verify){
                throw new FsApiException("验签失败");
            }

            MsfpayBaseResponse baseResponse = JSONObject.parseObject(msg, MsfpayBaseResponse.class);
            baseResponse.setCode(baseResponse.getMessage().getData().getHead().getRespcode());
            baseResponse.setMsg(baseResponse.getMessage().getData().getHead().getRespmsg());
            baseResponse.setTranflow(baseResponse.getMessage().getData().getHead().getTranflow());

            IResponseDefinition iResponseDefinition = JSON.parseObject(bodyStr, requestContext.getIApiDefinition().getResponseClass());
            baseResponse.setiResponseDefinition(iResponseDefinition);

            return baseResponse;
        } catch (Exception e) {
            LogUtil.error(log, "解析响应失败", e);
            throw new FsApiException("解析响应失败", e);
        }
    }

    private String sign(String content) {
        try {
            return SM2Utils.P7SignMessageDetach(MsfpayConstant.Encryption.SM3_WITH_SM2,
                    new String(Base64.encode(content.getBytes(MsfpayConstant.Http.CHARSET_UTF8))),
                    merchantSignPrivateKey,
                    merchantSignCert,
                    SM2Utils.session);
        } catch (Exception e) {
            log.error("签名失败", e);
            throw new FsApiException("签名失败", e);
        }
    }

    private void fillCommonHead(MsfpayBizRequest request, MsfpayPayClientConfigModel configModel, MsfpayPayApiDefinitionEnum iApiDefinition) {
        request.setPlatmerid(configModel.getPlatmerid());
        request.setTrancode(iApiDefinition.getTrancode());
        request.setMertrandate(MsfpayUtil.getCurrentDateTime(MsfpayConstant.DateTimeFormat.DATE_FORMAT));
        request.setMertrantime(MsfpayUtil.getCurrentDateTime(MsfpayConstant.DateTimeFormat.TIME_FORMAT));
        request.setZtpage(MsfpayConstant.BooleanString.FALSE);

        if (request.getTranflow() == null || request.getTranflow().isEmpty()) {
            request.setTranflow(MsfpayUtil.generateTranflow(configModel.getPlatmerid()));
        }
    }

    @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
        return httpRequestHandler.httpRequest(apiRequestModel, requestContext);
    }
}