package com.fshows.msfpay.client;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fshows.msfpay.apienum.MsfpayMerchantApiDefinitionEnum;
import com.fshows.msfpay.client.model.MsfpayMerchantClientConfigModel;
import com.fshows.msfpay.constant.MsfpayConstant;
import com.fshows.msfpay.request.MsfpayBizRequest;
import com.fshows.msfpay.response.MsfpayBaseResponse;
import com.fshows.msfpay.utils.SM2Utils;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.definition.IResponseDefinition;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import lombok.extern.slf4j.Slf4j;
import msfpay.cfca.sadk.util.Base64;
import msfpay.cfca.sadk.util.CertUtil;
import msfpay.cfca.sadk.x509.certificate.X509Cert;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.PrivateKey;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

@Slf4j
public class MsfpaySignApiClient extends AbstractApiClient<MsfpayBizRequest, MsfpayBaseResponse, MsfpayMerchantApiDefinitionEnum> {
    private PrivateKey merchantSignPrivateKey;
    private PrivateKey merchantEncPrivateKey;
    private X509Cert platformEncCert;
    private X509Cert platformSignCert;
    private X509Cert merchantEncCert;
    private X509Cert merchantSignCert;

    public MsfpaySignApiClient(MsfpayMerchantClientConfigModel apiClientConfig) throws Exception {
        super(apiClientConfig);
        initCertificatesAndKeys(apiClientConfig);
    }

    private static String getCurrentDateTime(String pattern) {
        SimpleDateFormat sdf = new SimpleDateFormat(pattern);
        return sdf.format(new Date());
    }

    private static String generateTranflow(String platmerid) {
        String dateTime = getCurrentDateTime(MsfpayConstant.DateTimeFormat.DATETIME_FORMAT);
        int random = (int) (Math.random() * 900) + 100;
        return platmerid + dateTime + random;
    }

    private void initCertificatesAndKeys(MsfpayMerchantClientConfigModel config) throws Exception {
        try {
            // 1. 加载商户签名私钥
            merchantSignPrivateKey = SM2Utils.getPrivateKeyFromSM2(config.getMerchantSignPath(), config.getMerchantSignPwd());

            // 2. 加载商户加密私钥
            merchantEncPrivateKey = SM2Utils.getPrivateKeyFromSM2(config.getMerchantEncPath(), config.getMerchantEncPwd());

            // 3. 加载平台加密证书
            String platformEncCertContent = new String(Files.readAllBytes(Paths.get(config.getPlatformEncCertPath())))
                    .replaceAll(MsfpayConstant.StringPattern.LINE_FEED, MsfpayConstant.StringPattern.EMPTY).replace(MsfpayConstant.StringPattern.CARRIAGE_RETURN, MsfpayConstant.StringPattern.EMPTY);
            platformEncCert = new X509Cert(platformEncCertContent.getBytes(MsfpayConstant.Http.CHARSET_UTF8));

            // 4. 加载平台签名证书
            String platformSignCertContent = new String(Files.readAllBytes(Paths.get(config.getPlatformSignCertPath())))
                    .replaceAll(MsfpayConstant.StringPattern.LINE_FEED, MsfpayConstant.StringPattern.EMPTY).replace(MsfpayConstant.StringPattern.CARRIAGE_RETURN, MsfpayConstant.StringPattern.EMPTY);
            platformSignCert = new X509Cert(platformSignCertContent.getBytes(MsfpayConstant.Http.CHARSET_UTF8));

            merchantEncCert = CertUtil.getCertFromSM2(config.getMerchantEncPath());

            merchantSignCert = CertUtil.getCertFromSM2(config.getMerchantSignPath());

            log.info("证书和密钥初始化成功");
        } catch (Exception e) {
            log.error("初始化证书和密钥失败", e);
            throw new Exception("初始化证书和密钥失败", e);
        }
    }

    @Override
    protected DefaultRequestContext buildRequestContext(MsfpayMerchantApiDefinitionEnum iApiDefinition, MsfpayBizRequest request, DefaultClientConfigModel customConfig) {
        DefaultRequestContext context = new DefaultRequestContext();
        context.setIApiDefinition(iApiDefinition);

        MsfpayMerchantClientConfigModel config = new MsfpayMerchantClientConfigModel();
        MsfpayMerchantClientConfigModel clientConfig = (MsfpayMerchantClientConfigModel) this.apiClientConfig;

        config.setApiParentURL(clientConfig.getApiParentURL());
        config.setCharset(clientConfig.getCharset());
        config.setSignType(clientConfig.getSignType());
        config.setVersion(clientConfig.getVersion());
        config.setPlatmerid(clientConfig.getPlatmerid());
        config.setTimeout(clientConfig.getTimeout());

        context.setApiClientConfig(config);
        context.setClientInfoModel(getClientInfo());
        return context;
    }

    @Override
    protected ApiRequestModel buildApiRequestModel(MsfpayBizRequest request, DefaultRequestContext context) {
        MsfpayMerchantClientConfigModel configModel = (MsfpayMerchantClientConfigModel) context.getApiClientConfig();
        MsfpayMerchantApiDefinitionEnum iApiDefinition = (MsfpayMerchantApiDefinitionEnum) context.getIApiDefinition();

        ApiRequestModel apiRequestModel = new ApiRequestModel();
        apiRequestModel.setApiURL(configModel.getApiParentURL());
        apiRequestModel.setContentType(MsfpayConstant.Http.CONTENT_TYPE_FORM);

        try {
            fillCommonHead(request, configModel, iApiDefinition);
            apiRequestModel.setRequest(request);

            Map<String, String> messageData = new HashMap<>();
            Map<String, Object> dataMap = new HashMap<>();
            Map<String, String> headMap = new HashMap<>();
            Map<String, Object> bodyMap = new HashMap<>();

            headMap.put(MsfpayConstant.HeadField.MERTRANDATE, request.getMertrandate());
            headMap.put(MsfpayConstant.HeadField.MERTRANTIME, request.getMertrantime());
            headMap.put(MsfpayConstant.HeadField.PLATMERID, request.getPlatmerid());
            headMap.put(MsfpayConstant.HeadField.TRANCODE, request.getTrancode());
            headMap.put(MsfpayConstant.HeadField.TRANFLOW, request.getTranflow());
            headMap.put(MsfpayConstant.HeadField.ZTPAGE, request.getZtpage());
            headMap.put(MsfpayConstant.HeadField.RESPCODE, "");
            headMap.put(MsfpayConstant.HeadField.RESPMSG, "");


            String requestJson = JSON.toJSONString(request);
            Map<String, Object> requestMap = JSON.parseObject(requestJson, Map.class);
            requestMap.remove(MsfpayConstant.HeadField.PLATMERID);
            requestMap.remove(MsfpayConstant.HeadField.TRANFLOW);
            requestMap.remove(MsfpayConstant.HeadField.TRANCODE);
            requestMap.remove(MsfpayConstant.HeadField.MERTRANDATE);
            requestMap.remove(MsfpayConstant.HeadField.MERTRANTIME);
            requestMap.remove(MsfpayConstant.HeadField.ZTPAGE);
            bodyMap.putAll(requestMap);

            dataMap.put(MsfpayConstant.MessageField.HEAD, headMap);
            dataMap.put(MsfpayConstant.MessageField.BODY, bodyMap);

            String dataJson = JSON.toJSONString(dataMap, SerializerFeature.WriteNullStringAsEmpty);
            String sign = sign(dataJson);

            messageData.put(MsfpayConstant.MessageField.DATA, dataJson);
            messageData.put(MsfpayConstant.MessageField.SIGN, sign);

            Map<String, Object> message = new HashMap<>();
            message.put(MsfpayConstant.MessageField.MESSAGE, messageData);

            apiRequestModel.setParamMap(message);

            String messageJson = JSON.toJSONString(message, SerializerFeature.WriteNullStringAsEmpty);
            byte[] envelopdata = SM2Utils.envelopeMessage(messageJson, MsfpayConstant.Encryption.SM4_ECB_PKCS7, platformEncCert);

            Map<String, String> paramMap = new HashMap<String, String>();
            paramMap.put(MsfpayConstant.RequestField.FORMAT_TYPE, MsfpayConstant.RequestValue.FORMAT_JSON);
            paramMap.put(MsfpayConstant.RequestField.CRYPT_TYPE, MsfpayConstant.RequestValue.CRYPT_SM2);
            paramMap.put(MsfpayConstant.RequestField.SIGN_TYPE, MsfpayConstant.RequestValue.SIGN_SM2);
            paramMap.put(MsfpayConstant.RequestField.DATA, new String(envelopdata));

            apiRequestModel.setRequestForm(paramMap);
        } catch (Exception e) {
            throw new FsApiException("构建请求失败", e);
        }

        return apiRequestModel;
    }

    private void fillCommonHead(MsfpayBizRequest request, MsfpayMerchantClientConfigModel configModel, MsfpayMerchantApiDefinitionEnum iApiDefinition) {
        request.setPlatmerid(configModel.getPlatmerid());
        request.setTrancode(iApiDefinition.getTrancode());
        request.setMertrandate(getCurrentDateTime(MsfpayConstant.DateTimeFormat.DATE_FORMAT));
        request.setMertrantime(getCurrentDateTime(MsfpayConstant.DateTimeFormat.TIME_FORMAT));
        request.setZtpage(MsfpayConstant.BooleanString.FALSE);

        if (request.getTranflow() == null || request.getTranflow().isEmpty()) {
            request.setTranflow(generateTranflow(configModel.getPlatmerid()));
        }
    }

    private String sign(String content) {
        try {
            return SM2Utils.P7SignMessageDetach(MsfpayConstant.Encryption.SM3_WITH_SM2,
                    new String(Base64.encode(content.getBytes(MsfpayConstant.Http.CHARSET_UTF8))),
                    merchantSignPrivateKey,
                    merchantSignCert,
                    SM2Utils.session);
        } catch (Exception e) {
            log.error("签名失败", e);
            throw new FsApiException("签名失败", e);
        }
    }

    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientName(MsfpayConstant.ClientInfo.CLIENT_NAME);
        clientInfoModel.setClientCode(MsfpayConstant.ClientInfo.CLIENT_CODE);
        return clientInfoModel;
    }

    @Override
    protected MsfpayBaseResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        JSONObject message = new JSONObject();
        message.put(MsfpayConstant.SignResponseField.SERVICE_URL, requestContext.getApiClientConfig().getApiParentURL());
        message.put(MsfpayConstant.SignResponseField.DATA, apiRequestModel.getRequestForm().get(MsfpayConstant.RequestField.DATA));

        MsfpayBaseResponse baseResponse = new MsfpayBaseResponse();
        baseResponse.setCode(MsfpayConstant.SignResponseField.SUCCESS_CODE);
        baseResponse.setMsg(MsfpayConstant.SignResponseField.MSG);

        IResponseDefinition iResponseDefinition = JSON.parseObject(message.toJSONString(), requestContext.getIApiDefinition().getResponseClass());
        baseResponse.setiResponseDefinition(iResponseDefinition);
        return baseResponse;
    }

    @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext defaultRequestContext) throws IOException {
        return null;
    }

    @Override
    public MsfpayBaseResponse execute(MsfpayBizRequest request, MsfpayMerchantApiDefinitionEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public MsfpayBaseResponse execute(MsfpayBizRequest request, MsfpayMerchantApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

    @Override
    protected MsfpayBaseResponse doExecute(MsfpayBizRequest request, MsfpayMerchantApiDefinitionEnum iApiDefinition, DefaultClientConfigModel customConfig) throws FsApiException {
        long beginTime = System.currentTimeMillis();

        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, customConfig);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;

        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);
            // 构建响应数据
            MsfpayBaseResponse response = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);

            return response;
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求业务异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, JSONObject.toJSONString(request), JSONObject.toJSONString(apiRequestModel), JSONObject.toJSONString(apiResponseModel));
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求未知异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, JSONObject.toJSONString(request), JSONObject.toJSONString(apiRequestModel), JSONObject.toJSONString(apiResponseModel));
            throw new FsApiException("请求未知异常", e);
        }
    }
}
