/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows;

import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.fshows.apienum.LzccbApiDefinitionEnum;
import com.fshows.component.LzccbClientConfigModel;
import com.fshows.component.Sm3ApiSignHandler;
import com.fshows.constant.LzccbConstant;
import com.fshows.request.LzccbBaseRequest;
import com.fshows.request.LzccbGetSm4KeyReq;
import com.fshows.response.LzccbBaseResponse;
import com.fshows.response.LzccbGetSm4KeyDataRes;
import com.fshows.response.LzccbGetSm4KeyRes;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.handler.IApiSignHandler;
import com.fshows.sdk.core.client.base.handler.IHttpRequestHandler;
import com.fshows.sdk.core.client.base.handler.ISerializableHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.client.component.http.PostHttpRequestHandler;
import com.fshows.sdk.core.client.component.serializable.JsonSerializableHandler;
import com.fshows.sdk.core.client.component.signer.RsaSignHandlerImpl;
import com.fshows.sdk.core.client.tempalte.single.apienum.SingleApiDefinitionEnum;
import com.fshows.sdk.core.client.tempalte.single.request.SingleBaseRequest;
import com.fshows.sdk.core.client.tempalte.single.response.SingleBaseResponse;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import com.fshows.sdk.core.util.ReqIdUtil;
import com.fshows.sdk.core.util.RequestParamUtils;
import com.fshows.util.SMSaltSigner;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * 泸州银行客户端
 *
 * @author liluqing
 * @version NestTemplateApiClinet.java, v 0.1 2023-12-12 15:40
 */
@Slf4j
public class LzccbPayApiClinet extends AbstractApiClient<LzccbBaseRequest, LzccbBaseResponse, LzccbApiDefinitionEnum> {

     /**
      * 加签验签处理器
      */
     protected IApiSignHandler iApiSignHandler = new Sm3ApiSignHandler();

    /**
     * 加密
     */
    private SMSaltSigner signer = new SMSaltSigner();

     /**
      * 序列化处理器
      */
     protected ISerializableHandler paramSerializable = new JsonSerializableHandler();

     /**
      * 请求执行器
      */
     protected IHttpRequestHandler httpRequestHandler = new PostHttpRequestHandler();

    public LzccbPayApiClinet(DefaultClientConfigModel apiClientConfig) throws FsApiException {
        super(apiClientConfig);
    }

    @Override
    public LzccbBaseResponse execute(LzccbBaseRequest request, LzccbApiDefinitionEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public LzccbBaseResponse execute(LzccbBaseRequest request, LzccbApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

    /**
     * 执行请求
     *
     * @param request
     * @param iApiDefinition
     * @param merchantConfigModel
     * @return
     * @throws FsApiException
     */
    @Override
    protected LzccbBaseResponse doExecute(LzccbBaseRequest request, LzccbApiDefinitionEnum iApiDefinition, DefaultClientConfigModel merchantConfigModel) throws FsApiException {
        long beginTime = System.currentTimeMillis();
        LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}",
                getClientInfo().getClientDesc(), iApiDefinition, request);
        // 构建请求上下文
        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, merchantConfigModel);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;
        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);
            // 执行post请求
            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            Object requestBody = ObjectUtils.defaultIfNull(apiRequestModel.getRequestBody(), apiRequestModel.getRequestForm());
            LogUtil.info(log, "{} >> 请求结束[密文] >> url={}, method={}, request={}, response={}, cost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), iApiDefinition, requestBody,
                    apiResponseModel.getResponseBody(), System.currentTimeMillis() - beginTime);

            LzccbBaseResponse response = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);

            LogUtil.info(log, "{} >> 请求结束[明文] >> url={}, method={}, request={}, response={}, cost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), requestContext.getIApiDefinition(), JSONObject.toJSONString(apiRequestModel.getRequest()),
                    apiResponseModel.getResponseBody(), System.currentTimeMillis() - beginTime);
            // 反序列化构建响应结果
            return response;
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc() ,
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc() ,
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw new FsApiException(e.getMessage(), e);
        }
    }

    /**
     * 构建请求上下文
     *
     * @param tradeApiDefinitionEnum
     * @param request
     * @return
     */
    @Override
    protected DefaultRequestContext buildRequestContext(LzccbApiDefinitionEnum tradeApiDefinitionEnum, LzccbBaseRequest request, DefaultClientConfigModel merchantConfig) {

        DefaultRequestContext context = new DefaultRequestContext();
        context.setRequestId(ReqIdUtil.getId());
        context.setIApiDefinition(tradeApiDefinitionEnum);
        // 复制默认参数
        LzccbClientConfigModel clientConfig = (LzccbClientConfigModel) this.apiClientConfig;
        //
        LzccbClientConfigModel config = newLzccbClientConfigModel(clientConfig);
        if (merchantConfig != null) {
            // merchantId
            config.setAgentId(StringUtils.isEmpty(merchantConfig.getAgentId()) ? clientConfig.getAgentId() : merchantConfig.getAgentId());
            // merchantId
            config.setAppId(StringUtils.isEmpty(merchantConfig.getAppId()) ? clientConfig.getAppId() : merchantConfig.getAppId());
            // 私钥
            config.setFubeiPrivateKey(StringUtils.isEmpty(merchantConfig.getFubeiPrivateKey())? clientConfig.getFubeiPrivateKey(): merchantConfig.getFubeiPrivateKey());
            // 公钥
            config.setPayCompanyPublicKey(StringUtils.isEmpty(merchantConfig.getPayCompanyPublicKey()) ? clientConfig.getPayCompanyPublicKey(): merchantConfig.getPayCompanyPublicKey());
            // md5 密钥
            config.setMd5SignKey(merchantConfig.getMd5SignKey());
            // sm4key
            config.setSm4Key(((LzccbClientConfigModel) merchantConfig).getSm4Key());
        }
        // 如果sm4key不存在,则重新获取sm4key
        if (!LzccbApiDefinitionEnum.GET_SM4_KEY.equals(tradeApiDefinitionEnum) && StringUtils.isBlank(config.getSm4Key())) {
            LzccbGetSm4KeyReq lzccbGetSm4KeyReq = new LzccbGetSm4KeyReq(config.getAppId(), config.getFubeiPrivateKey(), config.getMd5SignKey());
            // 调用泸州银行接口获取sm4key
            String sm4Key = getSM4Key(lzccbGetSm4KeyReq);
            config.setSm4Key(sm4Key);
        }
        context.setApiClientConfig(config);

        context.setClientInfoModel(getClientInfo());
        return context;
    }

    /**
     * 对象复制
     * @param clientConfigModel
     * @return {@link DefaultClientConfigModel}
     */
    private LzccbClientConfigModel newLzccbClientConfigModel(LzccbClientConfigModel clientConfigModel) {
        LzccbClientConfigModel configModel = new LzccbClientConfigModel();
        configModel.setAppId(clientConfigModel.getAppId());
        configModel.setAgentId(clientConfigModel.getAgentId());
        configModel.setFubeiPrivateKey(clientConfigModel.getFubeiPrivateKey());
        configModel.setPayCompanyPublicKey(clientConfigModel.getPayCompanyPublicKey());
        configModel.setPassword(clientConfigModel.getPassword());
        configModel.setApiParentURL(clientConfigModel.getApiParentURL());
        configModel.setMd5SignKey(clientConfigModel.getMd5SignKey());
        configModel.setSignTypeEnum(clientConfigModel.getSignTypeEnum());
        configModel.setVrifySignResponse(clientConfigModel.isVrifySignResponse());
        configModel.setCheckParam(clientConfigModel.isCheckParam());
        configModel.setConnectionTimeout(clientConfigModel.getConnectionTimeout());
        configModel.setReadTimeout(clientConfigModel.getReadTimeout());
        configModel.setCharset(clientConfigModel.getCharset());
        configModel.setHump(clientConfigModel.isHump());
        configModel.setExtendParam(new HashMap<>(clientConfigModel.getExtendParam()));
        configModel.setSingIgnoreNull(clientConfigModel.isSingIgnoreNull());
        return configModel;
    }

    /**
     * 获取接口加解密需要用到的sm4key
     *
     * @return
     */
    public String getSM4Key(LzccbGetSm4KeyReq lzccbGetSm4KeyReq) {
        LzccbClientConfigModel configModel = new LzccbClientConfigModel();
        configModel.setAppId(lzccbGetSm4KeyReq.getMerchantNo());
        configModel.setFubeiPrivateKey(lzccbGetSm4KeyReq.getSm2Key());
        configModel.setMd5SignKey(lzccbGetSm4KeyReq.getSm3Key());
        LzccbGetSm4KeyRes lzccbGetSm4KeyRes = (LzccbGetSm4KeyRes) doExecute(lzccbGetSm4KeyReq, LzccbApiDefinitionEnum.GET_SM4_KEY, configModel);
        System.out.println(lzccbGetSm4KeyRes.getData().getSm4Key());
        if (!LzccbConstant.GET_SM4KEY_RES_CODE.equals(lzccbGetSm4KeyRes.getCode())) {
            LogUtil.debug(log, "{} >> 请求异常 >> sm4key获取失败 >> apiDefinition={}, response={}",
                    getClientInfo().getClientDesc() ,
                    LzccbApiDefinitionEnum.GET_SM4_KEY,
                    lzccbGetSm4KeyRes);
            throw new FsApiException(lzccbGetSm4KeyRes.getMsg());
        }
        return lzccbGetSm4KeyRes.getData().getSm4Key();
    }



    /**
      * 构建请求参数
      *
      * @param request
      * @param context
      * @return
      */
    @Override
    protected ApiRequestModel buildApiRequestModel(LzccbBaseRequest request, DefaultRequestContext context) {
        // todo: 一般来说该方法需要完成以下事项
        // 1. 设置网关地址  2. 请求参数填充  3. 请求加签  4. 请求参数序列化

        // sdk客户端配置
        LzccbClientConfigModel defaultClientConfigModel = (LzccbClientConfigModel) context.getApiClientConfig();

        ApiRequestModel apiRequestModel = new ApiRequestModel();
        // 设置网关地址
        apiRequestModel.setApiURL(context.getApiClientConfig().getApiParentURL() + context.getIApiDefinition().getApiURI());

        // 请求头信息
        Map<String, String> headerMap = new HashMap<String, String>();
        apiRequestModel.setHeadMap(headerMap);

        headerMap.put("Mer-Id", defaultClientConfigModel.getAppId());
        headerMap.put("Key-Sn", "1");
        headerMap.put("Alg-Sn", "1");
        headerMap.put("Time-Stamp", DateUtil.format(new Date(), DatePattern.PURE_DATETIME_PATTERN));
        headerMap.put("Content-Type","application/json;charset=utf-8");
        headerMap.put("Accept", "application/json");
        headerMap.put("Accept-Encoding", "identity");
        // 请求参数
        apiRequestModel.setRequest(request);
        // 参数序列化
        String requestBody = paramSerializable.serializeObject(apiRequestModel, context);
        apiRequestModel.setRequestBody(requestBody);
        // 如果不是获取sm4key接口，则对请求参数进行加密
        if (!LzccbApiDefinitionEnum.GET_SM4_KEY.equals(context.getIApiDefinition())) {
            String encRequestBody = signer.encryptData(defaultClientConfigModel.getSm4Key(), requestBody);
            apiRequestModel.setRequestBody(encRequestBody);
        }

        // 加签
        apiRequestModel.setRequestSign(iApiSignHandler.sign(apiRequestModel, context));
        headerMap.put("Sign", apiRequestModel.getRequestSign());
        return apiRequestModel;
    }

     /**
      * 处理客户端信息
      *
      * @return
      */
    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientName("泸州银行");
        clientInfoModel.setClientCode("lzccb-sdk");
        return clientInfoModel;
    }

     @Override
     protected LzccbBaseResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
         // todo: 该方法需要完成以下内容
         // 1.响应结果反序列化    2. 返回结果验签
        // 客户端配置
         LzccbClientConfigModel clientConfigModel = (LzccbClientConfigModel) requestContext.getApiClientConfig();
        String ciphertext = apiResponseModel.getResponseBody();
        if (!LzccbApiDefinitionEnum.GET_SM4_KEY.equals(requestContext.getIApiDefinition())
                && !StringUtils.startsWith(ciphertext, "{") && !StringUtils.startsWith(ciphertext, "[")) {
            ciphertext = signer.decryptData(clientConfigModel.getSm4Key(), ciphertext);
        }
        apiResponseModel.setResponseBody(ciphertext);
        apiResponseModel.setResponse(JSONObject.parseObject(ciphertext, requestContext.getIApiDefinition().getResponseClass()));
        // 是否验签
        Boolean verifySign = iApiSignHandler.verifySign(apiResponseModel, apiRequestModel,  requestContext);
        // 如果需要验签
        if (clientConfigModel.isVrifySignResponse() && !Boolean.TRUE.equals(verifySign)) {
            throw new FsApiException("验签失败");
        }
        if (apiResponseModel.getResponse() instanceof LzccbGetSm4KeyRes) {
            LzccbGetSm4KeyRes lzccbGetSm4KeyRes = (LzccbGetSm4KeyRes) apiResponseModel.getResponse();
            LzccbGetSm4KeyDataRes dataRes = lzccbGetSm4KeyRes.getData();
            if (dataRes != null) {
                dataRes.setSm4Key(signer.decryptSm2(clientConfigModel.getFubeiPrivateKey(), dataRes.getSm4Key()));
            }
        }
        return (LzccbBaseResponse) apiResponseModel.getResponse();
     }

     @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
        return httpRequestHandler.httpRequest(apiRequestModel, requestContext);
    }
}