/**
 * Copyright (c) 2017, buh@fshows.com(BuHao) All Rights Reserved.
 */
package com.fshows.lifecircle.user.service.business.impl.biz;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.mapper.Condition;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.fshows.lifecircle.service.user.openapi.facade.domain.agent.AgentAdminIndexResult;
import com.fshows.lifecircle.service.user.openapi.facade.domain.agent.AgentDetailDTO;
import com.fshows.lifecircle.service.user.openapi.facade.domain.agent.AgentInfo;
import com.fshows.lifecircle.service.user.openapi.facade.domain.agent.AgentQuery;
import com.fshows.lifecircle.service.user.openapi.facade.domain.agent.AgentSimpleResult;
import com.fshows.lifecircle.service.user.openapi.facade.domain.agent.AgentUserAndBelongResult;
import com.fshows.lifecircle.service.user.openapi.facade.domain.agent.UserAgentDTO;
import com.fshows.lifecircle.service.user.openapi.facade.domain.agent.UserAgentParam;
import com.fshows.lifecircle.service.user.openapi.facade.domain.agent.UserAgentResult;
import com.fshows.lifecircle.service.user.openapi.facade.domain.constants.IUserConstants;
import com.fshows.lifecircle.service.user.openapi.facade.domain.result.commons.ErrorCodeEnum;
import com.fshows.lifecircle.service.user.openapi.facade.domain.userbase.BaseInfo;
import com.fshows.lifecircle.service.utils.DateUtil;
import com.fshows.lifecircle.service.utils.KeyGenerator;
import com.fshows.lifecircle.service.utils.KeyTypeEnum;
import com.fshows.lifecircle.service.utils.domain.BizResponse;
import com.fshows.lifecircle.service.utils.domain.PageResult;
import com.fshows.lifecircle.user.service.business.biz.IUserAgentService;
import com.fshows.lifecircle.user.service.business.db.IFbUserAgentService;
import com.fshows.lifecircle.user.service.business.db.IFbUserBaseService;
import com.fshows.lifecircle.user.service.business.db.IFbUserMerchantService;
import com.fshows.lifecircle.user.service.business.db.IFbUserOemService;
import com.fshows.lifecircle.user.service.business.utils.BijectionUtils;
import com.fshows.lifecircle.user.service.dao.FbUserAgentMapper;
import com.fshows.lifecircle.user.service.dao.FbUserMerchantMapper;
import com.fshows.lifecircle.user.service.domain.po.AgentPO;
import com.fshows.lifecircle.user.service.domain.po.FbUserAgent;
import com.fshows.lifecircle.user.service.domain.po.FbUserBase;
import com.fshows.lifecircle.user.service.domain.po.FbUserMerchant;
import com.google.common.collect.Lists;
import com.xiaoleilu.hutool.util.BeanUtil;
import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.joda.time.DateTime;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 项目：lifecircle-service-user
 * 包名：com.fshows.lifecircle.user.service.business.impl.biz
 * 功能：
 * 创建时间：2017-12-15 10:31
 * 更新记录：
 *
 * @author caofy
 */
@Slf4j
@Service
public class UserAgentServiceImpl implements IUserAgentService {

  @Autowired
  private IFbUserBaseService fbUserBaseService;
  @Autowired
  private IFbUserOemService fbUserOemService;
  @Autowired
  private IFbUserAgentService fbUserAgentService;
  @Autowired
  private IFbUserMerchantService fbUserMerchantService;

  @Autowired
  private FbUserAgentMapper fbUserAgentMapper;

  @Autowired
  private FbUserMerchantMapper fbUserMerchantMapper;

  @Override
  public BizResponse<UserAgentResult> addUserAgent(UserAgentParam param) {
    UserAgentResult userAgentResult = new UserAgentResult();
    log.info("1, addUserAgent execute >> para={}", JSON.toJSONString(param));
    try {

      if (param.getBaseId() == null || param.getOemId() == null) {
        log.info("1.1, addUserAgent check fail, baseId or oemId can not be null");
        return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(),
            ErrorCodeEnum.INVALID_PARAMETER.getMsg());
      }

      FbUserAgent obj = new FbUserAgent();
      BeanUtils.copyProperties(param, obj);

      // 生成业务主键id
      long baseId = KeyGenerator.getKeyByType(KeyTypeEnum.AGENT_USER_KEY);
      obj.setAgentId(baseId);

      if (param.getStatus() == null) {
        obj.setStatus(IUserConstants.DEFAULT_STATUS);
      }

      boolean addResult = fbUserAgentService.insert(obj);

      if (addResult) {
        userAgentResult.setAgentId(obj.getAgentId());
        log.info("2, addUserAgent success >> para={}", JSON.toJSONString(param));

        return BizResponse.success(userAgentResult);
      }

      log.info("3, addUserAgent fail >> para={}", JSON.toJSONString(param));

      return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(),
          ErrorCodeEnum.INVALID_PARAMETER.getMsg());

    } catch (Exception e) {
      log.error("方法: addUserAgent 发生异常， 参数: param = {} ,异常: Ex = {}", JSON.toJSONString(param),
          ExceptionUtils.getStackTrace(e));
      boolean deleteResult = fbUserBaseService
          .delete(Condition.create().eq(FbUserBase.BASE_ID, param.getBaseId()));
      log.info("delete userbase deleteResult = {}, baseId = {}", deleteResult, param.getBaseId());
      return BizResponse.fail(ErrorCodeEnum.DATA_OPERATION_FAILURE.getCode(),
          ErrorCodeEnum.DATA_OPERATION_FAILURE.getMsg());
    }

  }


  @Override
  public BizResponse<Boolean> updateUserAgent(UserAgentParam param) {
    log.info("1, updateUserAgent execute >> para={}", JSON.toJSONString(param));

    if (param.getAgentId() == null) {
      return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(),
          ErrorCodeEnum.INVALID_PARAMETER.getMsg());
    }

    long agentId = param.getAgentId();
    try {

      FbUserAgent obj = new FbUserAgent();
      BeanUtils.copyProperties(param, obj);

      boolean updateResult = fbUserAgentService
          .update(obj, Condition.create().eq(FbUserAgent.AGENT_ID, agentId));

      if (updateResult) {
        log.info("2, updateUserAgent success >> para={}", JSON.toJSONString(param));

        return BizResponse.success(Boolean.TRUE);
      }

      log.info("3, updateUserBase fail >> para={}", JSON.toJSONString(param));

      return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(),
          ErrorCodeEnum.INVALID_PARAMETER.getMsg());

    } catch (Exception e) {
      log.error("方法: updateUserAgent 发生异常， 参数: param = {} ,异常: Ex = {}", JSON.toJSONString(param),
          ExceptionUtils.getStackTrace(e));
      return BizResponse.fail(ErrorCodeEnum.DATA_OPERATION_FAILURE.getCode(),
          ErrorCodeEnum.DATA_OPERATION_FAILURE.getMsg());
    }
  }

  @Override
  public BizResponse<Boolean> auditUserAgent(Long agentId, int checkStatus, String remark) {
    log.info("1, auditUserAgent execute >> agentId={}, checkStatus={}", agentId, checkStatus);

    if (agentId == null) {
      return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(),
          ErrorCodeEnum.INVALID_PARAMETER.getMsg());
    }

    try {

      FbUserAgent obj = new FbUserAgent();
      obj.setAgentId(agentId);
      obj.setCheckStatus(checkStatus);
      obj.setCheckTime(System.currentTimeMillis());
      obj.setRemark(remark);

      fbUserAgentService.update(obj, Condition.create().eq(FbUserAgent.AGENT_ID, agentId));
      log.info("2, auditUserAgent succrss >> agentId={}, checkStatus={}", agentId, checkStatus);

      return BizResponse.success(Boolean.TRUE);

    } catch (Exception e) {
      log.error("方法: auditUserAgent 发生异常， 参数: agentId = {} ,异常: Ex = {}", agentId,
          ExceptionUtils.getStackTrace(e));
      return BizResponse.fail(ErrorCodeEnum.DATA_OPERATION_FAILURE.getCode(),
          ErrorCodeEnum.DATA_OPERATION_FAILURE.getMsg());
    }
  }

  @Override
  public BizResponse<AgentDetailDTO> queryUserAgentByAgentId(Long agentId) {

    /*agentDetailDTO  包含用户基础信息  和 代理商信息*/
    AgentDetailDTO agentDetailDTO = new AgentDetailDTO();

    FbUserAgent fbUserAgent = fbUserAgentService
        .selectOne(Condition.create().eq(FbUserAgent.AGENT_ID, agentId));

    AgentInfo agentInfo = BijectionUtils.invert(fbUserAgent, AgentInfo.class);

    if (agentInfo != null) {
      FbUserBase fbUserBase = fbUserBaseService
          .selectOne(Condition.create().eq(FbUserBase.BASE_ID, agentInfo.getBaseId()));
      BaseInfo baseInfo = BijectionUtils.invert(fbUserBase, BaseInfo.class);
      agentDetailDTO.setBaseInfo(baseInfo);
    }

    agentDetailDTO.setAgentInfo(agentInfo);

    return BizResponse.success(agentDetailDTO);
  }

  @Override
  public BizResponse<PageResult<UserAgentDTO>> queryUserAgentPage(AgentQuery agentQuery) {

    Map map = BeanUtil.beanToMap(agentQuery);

    int startNum = (agentQuery.getPage() - 1) * agentQuery.getPageSize();

    if (agentQuery.isExpireFlag()) {
      map.put("expire", agentQuery.isExpire() ? IUserConstants.YES : IUserConstants.NO);
    } else {
      map.put("expire", "all");
    }

    map.put("startNum", startNum);

    //代理商 grant设置no
    map.put("grant", IUserConstants.NO);

    List<AgentPO> list = fbUserAgentMapper.queryUserAgentPage(map);

    int totalNum = fbUserAgentMapper.countUserAgentByCondition(map);

    List<UserAgentDTO> targetList = BijectionUtils.invertList(list, UserAgentDTO.class);

    PageResult<UserAgentDTO> pageResult = new PageResult(totalNum, targetList);

    return BizResponse.success(pageResult);
  }

  @Override
  public BizResponse<PageResult<UserAgentDTO>> queryUserAgentGrantPage(AgentQuery agentQuery) {
    Map map = BeanUtil.beanToMap(agentQuery);

    int startNum = (agentQuery.getPage() - 1) * agentQuery.getPageSize();

    if (agentQuery.isExpireFlag()) {
      map.put("expire", agentQuery.isExpire() ? IUserConstants.YES : IUserConstants.NO);
    } else {
      map.put("expire", "all");
    }

    map.put("startNum", startNum);

    //授理商
    map.put("grant", IUserConstants.YES);

    List<AgentPO> list = fbUserAgentMapper.queryUserAgentPage(map);

    int totalNum = fbUserAgentMapper.countUserAgentByCondition(map);

    List<UserAgentDTO> targetList = BijectionUtils.invertList(list, UserAgentDTO.class);

    //设置 belongName

    for (UserAgentDTO grant : targetList) {
      if (grant.getBelong() != null && grant.getBelong() > 0) {
        FbUserAgent fbUserAgent = fbUserAgentService
            .selectOne(Condition.create().eq(FbUserAgent.AGENT_ID, grant.getBelong()));
        grant.setBelongAgentName(fbUserAgent.getCompany());
      } else {
        grant.setBelongAgentName("");
      }

      int yesterdayMerchantTotalNum = fbUserMerchantMapper
          .selectVaildMerchantCount(grant.getAgentId());

      Map yesterdayMap = new HashMap();
      yesterdayMap.put("agentId", grant.getAgentId());
      yesterdayMap.put("startTime", DateTime.now().plusDays(-1).toString("yyyy-MM-dd"));
      yesterdayMap.put("endTime", DateTime.now().toString("yyyy-MM-dd"));

      int merchantTotalNum = fbUserMerchantMapper.countYesterdayMerchant(yesterdayMap);

      grant.setYesterdayMerchantTotalNum(yesterdayMerchantTotalNum);
      grant.setMerchantTotalNum(merchantTotalNum);
    }

    PageResult<UserAgentDTO> pageResult = new PageResult(totalNum, targetList);

    return BizResponse.success(pageResult);
  }

  @Override
  public BizResponse<AgentUserAndBelongResult> queryUserAgentNameAndBelongAgengName(Long agentId) {

    Map<String, Object> map = fbUserAgentMapper.queryUserAgentNameAndBelongAgengName(agentId);
    AgentUserAndBelongResult result = BijectionUtils
        .mapToObject(map, AgentUserAndBelongResult.class);

    return BizResponse.success(result);
  }

  @Override
  public BizResponse<Boolean> existByCompany(String company) {
    int count = fbUserAgentService.selectCount(Condition.create().eq(FbUserAgent.COMPANY, company));
    if (count > 0) {
      return BizResponse.success(Boolean.TRUE);
    }
    return BizResponse.success(Boolean.FALSE);
  }

  @Override
  public BizResponse<List<AgentSimpleResult>> getAllAgentListByOemId(Long oemId) {
    // 获得 oem 下面的数据
    List<FbUserAgent> sourceList = fbUserAgentService
        .selectList(Condition.create().eq(FbUserAgent.OEM_ID, oemId)
            .eq(FbUserAgent.STATUS, 1).gt(FbUserAgent.VIP_TIME, System.currentTimeMillis()));

    List<AgentSimpleResult> targetList = BijectionUtils
        .invertList(sourceList, AgentSimpleResult.class);

    return BizResponse.success(targetList);
  }


  @Override
  public BizResponse<AgentAdminIndexResult> statisticData(Long agentId) {

//        toDayAccountNum	int	今日开户数
//        toYearAccountNum	int	本月开户数
//        yesterdayPayNum	int	昨日交易总额
//        toYearPayNum	string	本月交易总额
//        sumMerchantNum	string	总商户数
//        putawayShopNum	string	上架门店总数

    Date date = new Date();
    int toDayAccountNum = fbUserMerchantService.selectCount(new EntityWrapper<FbUserMerchant>()
        .and(FbUserMerchant.AGENT_ID, agentId)
        .between(FbUserMerchant.CREATE_TIME, DateUtil.beginOfToday(), date));

    int toYearAccountNum = fbUserMerchantService.selectCount(new EntityWrapper<FbUserMerchant>()
        .and(FbUserMerchant.AGENT_ID, agentId)
        .between(FbUserMerchant.CREATE_TIME, DateUtil.beginOfToday(), date));

    int sumMerchantNum = fbUserMerchantService.selectCount(new EntityWrapper<FbUserMerchant>()
        .and(FbUserMerchant.AGENT_ID, agentId));

    int putawayShopNum = fbUserMerchantMapper.selectVaildMerchantCount(agentId);

    AgentAdminIndexResult indexResult = AgentAdminIndexResult.builder()
        .toDayAccountNum(toDayAccountNum)
        .toYearAccountNum(toYearAccountNum)
        .sumMerchantNum(sumMerchantNum)
        .putawayShopNum(putawayShopNum)
        .yesterdayPayNum(BigDecimal.ZERO) //fixme  since trade
        .toYearPayNum(BigDecimal.ZERO) // fixme  since trade
        .build();

    return BizResponse.success(indexResult);


  }

  /**
   * 根据 代理商 获得下面所有受理商
   */
  @Override
  public List<AgentSimpleResult> getAllGrantListByAgentId(Long agentId) {

    List<FbUserAgent> fbUserAgents = fbUserAgentService
        .selectList(new EntityWrapper<FbUserAgent>().eq(FbUserAgent.BELONG, agentId));

    if (fbUserAgents != null) {

      return fbUserAgents.stream().map(
          agent -> AgentSimpleResult.builder().agentId(agent.getAgentId())
              .company(agent.getCompany()).build()).collect(
          Collectors.toList());
    }
    return Lists.newArrayList();
  }
}
