/**
 * Copyright (c) 2017, buh@fshows.com(BuHao) All Rights Reserved.
 */
package com.fshows.lifecircle.user.service.business.impl.biz;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.fshows.lifecircle.service.user.openapi.facade.domain.constants.UserType;
import com.fshows.lifecircle.service.user.openapi.facade.domain.params.AppLoginInfoParam;
import com.fshows.lifecircle.service.user.openapi.facade.domain.params.LoginConfigParam;
import com.fshows.lifecircle.service.user.openapi.facade.domain.params.LoginParam;
import com.fshows.lifecircle.service.user.openapi.facade.domain.result.AppLoginInfoResult;
import com.fshows.lifecircle.service.user.openapi.facade.domain.result.LoginConfigResult;
import com.fshows.lifecircle.service.user.openapi.facade.domain.result.LoginResult;
import com.fshows.lifecircle.service.user.openapi.facade.domain.result.commons.ErrorCodeEnum;
import com.fshows.lifecircle.service.utils.domain.BizResponse;
import com.fshows.lifecircle.user.service.business.biz.ILoginService;
import com.fshows.lifecircle.user.service.business.db.IFbAppLoginLogService;
import com.fshows.lifecircle.user.service.business.db.IFbOemConfigService;
import com.fshows.lifecircle.user.service.business.db.IFbOemRoleService;
import com.fshows.lifecircle.user.service.business.db.IFbUserAgentService;
import com.fshows.lifecircle.user.service.business.db.IFbUserBaseService;
import com.fshows.lifecircle.user.service.business.db.IFbUserLoginLogService;
import com.fshows.lifecircle.user.service.business.db.IFbUserMerchantChildrenRoleService;
import com.fshows.lifecircle.user.service.business.db.IFbUserMerchantChildrenService;
import com.fshows.lifecircle.user.service.business.db.IFbUserMerchantService;
import com.fshows.lifecircle.user.service.business.db.IFbUserOemChildrenService;
import com.fshows.lifecircle.user.service.business.db.IFbUserOemService;
import com.fshows.lifecircle.user.service.business.utils.LoginPlatformTypeEnum;
import com.fshows.lifecircle.user.service.business.utils.UserTypeEnum;
import com.fshows.lifecircle.user.service.dao.FbUserBaseMapper;
import com.fshows.lifecircle.user.service.domain.po.FbAppLoginLog;
import com.fshows.lifecircle.user.service.domain.po.FbOemConfig;
import com.fshows.lifecircle.user.service.domain.po.FbOemRole;
import com.fshows.lifecircle.user.service.domain.po.FbUserAgent;
import com.fshows.lifecircle.user.service.domain.po.FbUserBase;
import com.fshows.lifecircle.user.service.domain.po.FbUserLoginLog;
import com.fshows.lifecircle.user.service.domain.po.FbUserMerchant;
import com.fshows.lifecircle.user.service.domain.po.FbUserMerchantChildren;
import com.fshows.lifecircle.user.service.domain.po.FbUserMerchantChildrenRole;
import com.fshows.lifecircle.user.service.domain.po.FbUserOem;
import com.fshows.lifecircle.user.service.domain.po.FbUserOemChildren;
import com.xiaoleilu.hutool.crypto.SecureUtil;
import com.xiaoleilu.hutool.util.NetUtil;
import java.util.Date;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 项目：lifecircle-service-user
 * 包名：com.fshows.lifecircle.user.service.business.impl.biz
 * 功能：
 * 创建时间：2017-12-12 16:31
 * 更新记录：
 *
 * @author BuHao
 */
@Slf4j
@Service
public class LoginServiceImpl implements ILoginService {

  @Autowired
  private IFbUserBaseService fbUserBaseService;
  @Autowired
  private IFbUserOemService fbUserOemService;
  @Autowired
  private IFbUserAgentService fbUserAgentService;
  @Autowired
  private IFbUserMerchantService fbUserMerchantService;
  @Autowired
  private IFbOemConfigService fbOemConfigService;
  @Autowired
  private IFbUserOemChildrenService fbUserOemChildrenService;
  @Autowired
  private IFbUserLoginLogService fbUserLoginLogService;
  @Autowired
  private IFbAppLoginLogService fbAppLoginLogService;
  @Autowired
  private IFbUserMerchantChildrenService fbUserMerchantChildrenService;
  @Autowired
  private IFbUserMerchantChildrenRoleService fbUserMerchantChildrenRoleService;
  @Autowired
  private IFbOemRoleService fbOemRoleService;
  @Autowired
  private FbUserBaseMapper fbUserBaseMapper;

  /**
   * 登录接口
   */
  @Override
  public BizResponse<LoginResult> login(LoginParam loginParam) {

    // 查询用户信息
    FbUserBase userBase = fbUserBaseService.selectOne(
        new EntityWrapper<FbUserBase>().eq(FbUserBase.USERNAME, loginParam.getUsername())
            .eq(FbUserBase.IS_DEL, 0));

    // 检查登录是否可以进行
    LoginResult loginResult = null;
    BizResponse<LoginResult> response = checkLogin(loginParam, userBase);
    if (response.isSuccess()) {

      loginResult = response.getData();
    } else {

      return response;
    }

    // 比较密码是否正确
    // 获得请求密码加盐的 md5值
    String requestPwdByMd5 = SecureUtil.md5(loginParam.getPassword() + userBase.getSalt());
    if (!requestPwdByMd5.equals(userBase.getUserpwd())) {

      // 登录失败控制
      loginErrorController(userBase);

      return BizResponse
          .fail(ErrorCodeEnum.PASSWORD_ERROR.getCode(), ErrorCodeEnum.PASSWORD_ERROR.getMsg());

      // 清空异常登录次数
    } else {

      fbUserBaseService.update(new FbUserBase().setLoginLockTime(0L).setLoginErrorCount(0),
          new EntityWrapper<FbUserBase>().eq(FbUserBase.BASE_ID, userBase.getBaseId()));
    }

    // 如果是 app还要生成 accessToken
    if (loginParam.getIsApp()) {

      String accessToken = SecureUtil.md5(
          loginResult.getOemId() + "" + loginResult.getUserId() + "" + System.currentTimeMillis());
      loginResult.setAccessToken(accessToken);
    }

    // 保存日志
    saveLoginLog(loginParam, loginResult, getLoginPlatformType(userBase.getUserType()));

    return BizResponse.success(loginResult);
  }

  /**
   * 登录失败控制
   */
  private void loginErrorController(FbUserBase userBase) {

    Integer loginErrorCount = userBase.getLoginErrorCount();
    loginErrorCount += 1;

    // 获得最大失败次数
    Integer maxLoginCount = 3;
    // 获得锁定时间
    Long lockTimeMinutes = 2 * 60L;

    // 已达到最大登录次数，锁定
    long loginLockTime = 0;
    if (loginErrorCount >= maxLoginCount) {

      loginLockTime = System.currentTimeMillis() + (lockTimeMinutes * 60 * 1000);

    }

    fbUserBaseService.update(
        new FbUserBase().setLoginErrorCount(loginErrorCount).setLoginLockTime(loginLockTime),
        new EntityWrapper<FbUserBase>().eq(FbUserBase.BASE_ID, userBase.getBaseId()));
  }

  /**
   * 保存登录日志
   */
  private void saveLoginLog(LoginParam loginParam, LoginResult loginResult,
      LoginPlatformTypeEnum loginPlatformTypeEnum) {
    // 登录成功
    // 修改最后登录信息
    try {

      // 获得整型 ip
      long lastLoginIp = NetUtil.ipv4ToLong(loginParam.getLoginIp());

      // 更新用户表最后登录
      fbUserBaseService.update(
          new FbUserBase().setLastLoginIp(lastLoginIp)
              .setLastLoginTime(new Date()),
          new EntityWrapper<FbUserBase>().eq(FbUserBase.USERNAME, loginParam.getUsername()));

      // 更新登录日志表
      Long userId = loginResult.getUserId();
      fbUserLoginLogService.insert(
          new FbUserLoginLog().setLoginIp(lastLoginIp).setUserId(userId)
              .setPlatformType(loginPlatformTypeEnum.getVal()));

      // 更新 app 登录日志表
      if (LoginPlatformTypeEnum.APP.equals(loginPlatformTypeEnum)) {

        FbAppLoginLog appLoginLog = fbAppLoginLogService
            .selectOne(new EntityWrapper<FbAppLoginLog>().eq(FbAppLoginLog.USER_ID, userId));

        // 新增
        if (appLoginLog == null) {

          fbAppLoginLogService
              .insert(new FbAppLoginLog().setAccessToken(loginResult.getAccessToken())
                  .setUserId(userId));

          // 修改
        } else {

          // 放入旧的 token
          loginResult.setLastAccessToken(appLoginLog.getAccessToken());

          fbAppLoginLogService
              .update(new FbAppLoginLog().setAccessToken(loginResult.getAccessToken()),
                  new EntityWrapper<FbAppLoginLog>().eq(FbAppLoginLog.USER_ID, userId));
        }
      }

    } catch (Exception e) {

      log.error("方法: login 发生异常， 参数: loginParam = {} ,异常: Ex = {}", JSON.toJSONString(loginParam),
          ExceptionUtils.getStackTrace(e));
    }
  }

  /**
   * 获得登录配置接口（web）
   */
  @Override
  public BizResponse<LoginConfigResult> getLoginConf(LoginConfigParam loginConfigParam) {

    // 配置类型
    Integer configType = loginConfigParam.getConfigType();
    // oem的标志
    String oemShortName = loginConfigParam.getOemShortName();

    // 获得配置
    FbOemConfig oemConfig = fbOemConfigService
        .selectOne(new EntityWrapper<FbOemConfig>().eq(FbOemConfig.SUB_DOMAIN, oemShortName));

    // 没有配置返回错误
    if (oemConfig == null) {

      return BizResponse.fail(ErrorCodeEnum.NO_HAVE_LOGIN_CONFIG.getCode(),
          ErrorCodeEnum.NO_HAVE_LOGIN_CONFIG.getMsg());
    }

    LoginConfigResult loginConfigResult = new LoginConfigResult();
    loginConfigResult.setLogoutUrl(oemConfig.getSubDomain());
    loginConfigResult.setOemId(oemConfig.getOemId());
    loginConfigResult.setWebLogo(oemConfig.getWebLogo());

    // 如果是 Ome
    if (configType.equals(UserTypeEnum.OEM.getValue())) {

      loginConfigResult.setLoginBg(oemConfig.getAdminLoginBg());
      loginConfigResult.setLoginLogo(oemConfig.getAdminLoginLogo());
      loginConfigResult.setTitle(oemConfig.getAdminLoginTitle());
      loginConfigResult.setLogo(oemConfig.getAdminLogo());

      // 授理商或代理商
    } else if (configType.equals(UserTypeEnum.AGENT.getValue()) || configType
        .equals(UserTypeEnum.GRANT.getValue())) {

      // 背景
      loginConfigResult.setLoginBg(oemConfig.getAgentLoginBg());
      // 登录 logo
      loginConfigResult.setLoginLogo(oemConfig.getAgentLoginLogo());
      // 系统 logo
      loginConfigResult.setLogo(oemConfig.getAgentLogo());
      // 后台名称
      loginConfigResult.setTitle(oemConfig.getAgentLoginTitle());

      // 商户后台
    } else if (configType.equals(UserTypeEnum.MERCHANT.getValue())) {

      loginConfigResult.setLoginBg(oemConfig.getUserLoginBg());
      loginConfigResult.setLoginLogo(oemConfig.getUserLoginLogo());
      loginConfigResult.setLogo(oemConfig.getUserLogo());
      loginConfigResult.setTitle(oemConfig.getUserLoginTitle());
    }

    return BizResponse.success(loginConfigResult);
  }

  /**
   * 获得 app 登录信息
   */
  @Override
  public AppLoginInfoResult getAppLoginInfo(AppLoginInfoParam param) {

    // 获得 accessToken
    String accessToken = param.getAccessToken();

    // 获得 apploginLog
    FbAppLoginLog appLoginLog = fbAppLoginLogService
        .selectOne(new EntityWrapper<FbAppLoginLog>().eq(FbAppLoginLog.ACCESS_TOKEN, accessToken));

    if (appLoginLog != null) {

      // 查询用户基础信息
      FbUserBase fbUserBase = fbUserBaseService.selectOne(
          new EntityWrapper<FbUserBase>().eq(FbUserOem.BASE_ID, appLoginLog.getUserId())
              .eq(FbUserBase.IS_DEL, 0));
      if (fbUserBase != null) {

        AppLoginInfoResult result = new AppLoginInfoResult();
        result.setAccessToken(param.getAccessToken());
        result.setUsername(fbUserBase.getUsername());
        result.setUserId(fbUserBase.getBaseId());

        Integer isChild = fbUserBase.getIsChild();
        Long userId = fbUserBase.getBaseId();

        // 如果是子账号
        if (1 == isChild) {

          // 获得商户子账号
          FbUserMerchantChildren merchantChildren = fbUserMerchantChildrenService.selectOne(
              new EntityWrapper<FbUserMerchantChildren>().eq(FbUserMerchantChildren.IS_DEL, 0)
                  .eq(FbUserMerchantChildren.BASE_ID, userId));
          if (merchantChildren != null) {

            // todo:下期开发
            result.setIsPush(1);
            result.setStoreName("todo");

            result.setMid(merchantChildren.getMid());
            result.setStoreId(merchantChildren.getStoreId());

            // 获得权限
            Long roleId = merchantChildren.getRoleId();
            FbUserMerchantChildrenRole merchantChildrenRole = fbUserMerchantChildrenRoleService
                .selectOne(new EntityWrapper<FbUserMerchantChildrenRole>()
                    .eq(FbUserMerchantChildrenRole.IS_DEL, 0)
                    .eq(FbUserMerchantChildrenRole.ROLE_ID, roleId));

            result.setRoleType(merchantChildrenRole.getRoleType());

            // 获得商户主账号
            FbUserMerchant userMerchant = fbUserMerchantService
                .selectOne(new EntityWrapper<FbUserMerchant>().eq(FbUserMerchant.MID,
                    merchantChildren.getMid()));
            if (userMerchant != null) {

              // 用户过期
              Long vipTime = userMerchant.getVipTime();
              if (System.currentTimeMillis() >= vipTime || userMerchant.getStatus() != 1) {

                return null;
              }

              result.setOemId(userMerchant.getOemId());
              result.setAgentId(userMerchant.getAgentId());
              FbUserBase parentUser = fbUserBaseService.selectOne(
                  new EntityWrapper<FbUserBase>().eq(FbUserBase.IS_DEL, 0)
                      .eq(FbUserBase.BASE_ID, userMerchant.getBaseId()));
              if (parentUser != null) {

                result.setMUsername(parentUser.getUsername());
              } else {

                result.setMUsername("");
              }
            }

            return result;

          }
          // 主账号
        } else {

          // 获得商户账号
          FbUserMerchant userMerchant = fbUserMerchantService
              .selectOne(new EntityWrapper<FbUserMerchant>().eq(FbUserMerchant.BASE_ID,
                  userId));

          if (userMerchant != null) {

            // 用户过期
            Long vipTime = userMerchant.getVipTime();
            if (System.currentTimeMillis() >= vipTime || userMerchant.getStatus() != 1) {

              return null;
            }

            // todo:下期开发
            result.setIsPush(1);
            result.setStoreName("todo");

            result.setMid(userMerchant.getMid());
            result.setStoreId(0L);
            result.setRoleType(0);
            result.setOemId(userMerchant.getOemId());
            result.setAgentId(userMerchant.getAgentId());
            result.setMUsername(fbUserBase.getUsername());

            return result;

          }

        }

      }
    }

    return null;
  }

  /**
   * app 退出登录
   */
  @Override
  public Boolean appLoginOut(String accessToken) {

    return fbAppLoginLogService.update(new FbAppLoginLog().setAccessToken(""),
        new EntityWrapper<FbAppLoginLog>().eq(FbAppLoginLog.ACCESS_TOKEN, accessToken));
  }


  /**
   * 登录检查
   */
  private BizResponse<LoginResult> checkLogin(LoginParam loginParam, FbUserBase userBase) {

    //---------------------------------公共逻辑检测-------------------------------------
    // 判断是否存在
    if (userBase == null) {

      return BizResponse.fail(ErrorCodeEnum.USERNAME_NOT_EXIST.getCode(),
          ErrorCodeEnum.USERNAME_NOT_EXIST.getMsg());
    }

    // 判断账号状态
    if (userBase.getStatus() != 1) {

      return BizResponse
          .fail(ErrorCodeEnum.USER_DISABLE.getCode(), ErrorCodeEnum.USER_DISABLE.getMsg());
    }

    // 设置是否为子账号
    boolean isChild = userBase.getIsChild() == 1;
    // 获得用户类型
    Integer userType = loginParam.getUserType();

    // 登录锁定判断
    if (userBase.getLoginLockTime() != 0) {

      if (userBase.getLoginLockTime() > System.currentTimeMillis()) {

        ErrorCodeEnum loginLock = ErrorCodeEnum.LOGIN_LOCK;
        return BizResponse.fail(loginLock.getCode(), loginLock.getMsg());
      } else {

        // 初始化登录锁定锁定
        fbUserBaseService.update(
            new FbUserBase().setLoginErrorCount(0).setLoginLockTime(0L),
            new EntityWrapper<FbUserBase>().eq(FbUserBase.BASE_ID, userBase.getBaseId()));
      }
    }

    // 比较用户类型是否正确
    if (userType != null && !userBase.getUserType().equals(userType)) {

      if (userBase.getUserType().intValue() == UserType.GRANT.getValue()) {
      } else {
        return BizResponse
            .fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());
      }

    }

    //--------------------------------------平台逻辑检测---------------------------------------------
    // 获得用户类型
    UserTypeEnum userTypeEnum = UserTypeEnum.valueOf(userType);

    return checkByUserType(userTypeEnum, userBase, isChild, loginParam);
  }

  /**
   * oem 类型用户检测
   */
  private BizResponse<LoginResult> checkOem(FbUserBase userBase, Boolean isChild,
      LoginParam param) {

    LoginResult loginResult = new LoginResult();
    FbUserOem userOem = null;

    // 如果是子账号
    if (isChild) {

      FbUserOemChildren fbUserOemChildren = fbUserOemChildrenService.selectOne(
          new EntityWrapper<FbUserOemChildren>()
              .eq(FbUserOemChildren.BASE_ID, userBase.getBaseId()));
      if (fbUserOemChildren == null) {

        return BizResponse
            .fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());
      } else {

        Long belognOemId = fbUserOemChildren.getOemId();
        // 查询 oem 帐户信息
        userOem = fbUserOemService
            .selectOne(new EntityWrapper<FbUserOem>().eq(FbUserOem.OEM_ID, belognOemId));

        // 判断所属角色是否关闭
        Long roleId = fbUserOemChildren.getRoleId();
        int count = fbOemRoleService.selectCount(
            new EntityWrapper<FbOemRole>().eq(FbOemRole.IS_DEL, 0).eq(FbOemRole.STATUS, 1)
                .eq(FbOemRole.ROLE_ID, roleId));
        if (count == 0) {

          return BizResponse
              .fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(),
                  ErrorCodeEnum.USERTYPE_ERROR.getMsg());
        }
      }
    } else {

      // 查询 oem 帐户信息
      userOem = fbUserOemService
          .selectOne(new EntityWrapper<FbUserOem>().eq(FbUserOem.BASE_ID, userBase.getBaseId()));
    }

    if (userOem == null) {

      return BizResponse
          .fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());

    } else {

      Long oemId = userOem.getOemId();
      if (!oemId.equals(param.getOemId())) {

        return BizResponse
            .fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());
      }

      // 设置是否是付呗自身
      loginResult.setSelf(userOem.getIsSelfSupport() == 1);
      loginResult.setOemId(userOem.getOemId());

    }

    return BizResponse.success(loginResult);
  }


  /**
   * 代理商/授理商用户检测
   */
  private BizResponse<LoginResult> checkByAgent(FbUserBase userBase) {

    LoginResult loginResult = new LoginResult();

    // 查询代理商信息
    FbUserAgent userAgent = fbUserAgentService
        .selectOne(new EntityWrapper<FbUserAgent>().eq(FbUserOem.BASE_ID, userBase.getBaseId()));
    if (userAgent == null) {

      return BizResponse
          .fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());

      // 是否过期
    } else if (System.currentTimeMillis() >= userAgent.getVipTime()) {

      ErrorCodeEnum userOverdue = ErrorCodeEnum.USER_OVERDUE;
      return BizResponse.fail(userOverdue.getCode(), userOverdue.getMsg());
    } else {

      if (userBase.getUserType().intValue() == UserType.AGENT.getValue()) {
        //代理商
        loginResult.setUserType(userBase.getUserType());
        loginResult.setAgentId(userAgent.getAgentId());
      } else {
        //授理商
        loginResult.setUserType(userBase.getUserType());
        loginResult.setGrantId(userAgent.getAgentId());
        //设置该授理商 所属代理商id
        loginResult.setAgentId(userAgent.getBelong());
      }

    }

    return BizResponse.success(loginResult);
  }


  /**
   * 商户检查
   */
  private BizResponse<LoginResult> checkByMerchant(FbUserBase userBase, Boolean isChild) {

    LoginResult loginResult = new LoginResult();

    // 查询商户信息
    FbUserMerchant userMerchant = null;

    if (isChild) {

      // 查询子账号
      FbUserMerchantChildren merchantChildren = fbUserMerchantChildrenService.selectOne(
          new EntityWrapper<FbUserMerchantChildren>().eq(FbUserMerchantChildren.IS_DEL, 0)
              .eq(FbUserMerchant.BASE_ID, userBase.getBaseId()));

      if (merchantChildren != null) {

        Long mid = merchantChildren.getMid();

        userMerchant = fbUserMerchantService.selectOne(
            new EntityWrapper<FbUserMerchant>().eq(FbUserMerchant.MID, mid));
      }

    } else {

      userMerchant = fbUserMerchantService.selectOne(
          new EntityWrapper<FbUserMerchant>().eq(FbUserMerchant.BASE_ID, userBase.getBaseId()));
    }

    if (userMerchant == null) {

      return BizResponse
          .fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());

    } else {

      // 用户过期
      Long vipTime = userMerchant.getVipTime();
      if (System.currentTimeMillis() >= vipTime) {

        ErrorCodeEnum userOverdue = ErrorCodeEnum.USER_OVERDUE;
        return BizResponse.fail(userOverdue.getCode(), userOverdue.getMsg());
      }

      loginResult.setOemId(userMerchant.getOemId());

      // 用户的授理商或代理商 id
      Long agentId = userMerchant.getAgentId();
      // 查询代理商 id
      FbUserAgent userAgent = fbUserAgentService
          .selectOne(
              new EntityWrapper<FbUserAgent>().eq(FbUserOem.BASE_ID, agentId));
      if (userAgent == null) {

        loginResult.setAgentId(0L);
        loginResult.setGrantId(0L);

      } else {

        // 授理商
        Long belong = userAgent.getBelong();
        if (belong != 0) {

          loginResult.setAgentId(belong);
          loginResult.setGrantId(agentId);
          // 代理商
        } else {

          loginResult.setAgentId(agentId);
        }
      }
    }

    return BizResponse.success(loginResult);
  }

  /**
   * 根据用户类型获得登录平台类型
   */
  private LoginPlatformTypeEnum getLoginPlatformType(Integer userType) {

    UserTypeEnum userTypeEnum = UserTypeEnum.valueOf(userType);

    switch (userTypeEnum) {

      case OEM:
        return LoginPlatformTypeEnum.OEM_SYSTEM;
      case AGENT:
        return LoginPlatformTypeEnum.AGENT_SYSTEM;
      case GRANT:
        return LoginPlatformTypeEnum.AGENT_SYSTEM;
      case MERCHANT:
        return LoginPlatformTypeEnum.APP;
      default:
        return LoginPlatformTypeEnum.APP;
    }
  }

  /**
   * 按用户类型检查登录
   */
  private BizResponse checkByUserType(UserTypeEnum userTypeEnum, FbUserBase userBase,
      Boolean isChild,
      LoginParam loginParam) {

    // 创建登录返回对象
    LoginResult loginResult = null;

    //----------------------------------- 获得不同类型帐户信息 -------------------------------------
    // oem 账号
    if (UserTypeEnum.OEM.equals(userTypeEnum)) {

      BizResponse<LoginResult> response = checkOem(userBase, isChild, loginParam);
      if (response.isSuccess()) {

        loginResult = response.getData();
      } else {

        return response;
      }

      // 代理商 授理商前端是同一个页面 userType=2
    } else if (UserTypeEnum.AGENT.equals(userTypeEnum)) {

      BizResponse<LoginResult> response = checkByAgent(userBase);
      if (response.isSuccess()) {

        loginResult = response.getData();
      } else {

        return response;
      }

      // 商户
    } else if (UserTypeEnum.MERCHANT.equals(userTypeEnum)) {

      BizResponse<LoginResult> response = checkByMerchant(userBase, isChild);
      if (response.isSuccess()) {

        loginResult = response.getData();
      } else {

        return response;
      }
      // 没有匹配的用户类型
    } else {

      return BizResponse
          .fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());
    }

    //--------------------------------------- 封装公共返回值------------------------------------
    // 设置用户 id
    loginResult.setUserId(userBase.getBaseId());
    // 设置 oem id
    if (loginParam.getOemId() != null) {

      loginResult.setOemId(loginParam.getOemId());
    }
    loginResult.setChild(isChild);
    // 设置账号类型Is
    loginResult.setUserType(userBase.getUserType());
    // 设置用户名   
    loginResult.setUsername(userBase.getUsername());

    return BizResponse.success(loginResult);
  }
}
