/**
 * Copyright (c) 2017, buh@fshows.com(BuHao) All Rights Reserved.
 */
package com.fshows.lifecircle.user.service.business.impl.biz;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.fshows.lifecircle.service.user.openapi.facade.domain.params.LoginConfigParam;
import com.fshows.lifecircle.service.user.openapi.facade.domain.params.LoginParam;
import com.fshows.lifecircle.service.user.openapi.facade.domain.result.LoginConfigResult;
import com.fshows.lifecircle.service.user.openapi.facade.domain.result.LoginResult;
import com.fshows.lifecircle.service.user.openapi.facade.domain.result.commons.ErrorCodeEnum;
import com.fshows.lifecircle.service.utils.domain.BizResponse;
import com.fshows.lifecircle.user.service.business.biz.ILoginService;
import com.fshows.lifecircle.user.service.business.db.*;
import com.fshows.lifecircle.user.service.business.utils.UserTypeEnum;
import com.fshows.lifecircle.user.service.domain.po.*;
import com.xiaoleilu.hutool.crypto.SecureUtil;
import com.xiaoleilu.hutool.util.NetUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;

/**
 * 项目：lifecircle-service-user
 * 包名：com.fshows.lifecircle.user.service.business.impl.biz
 * 功能：
 * 创建时间：2017-12-12 16:31
 * 更新记录：
 *
 * @author BuHao
 */
@Slf4j
@Service
public class LoginServiceImpl implements ILoginService {

  @Autowired
  private IFbUserBaseService fbUserBaseService;
  @Autowired
  private IFbUserOemService fbUserOemService;
  @Autowired
  private IFbUserAgentService fbUserAgentService;
  @Autowired
  private IFbUserMerchantService fbUserMerchantService;
  @Autowired
  private IFbOemConfigService fbOemConfigService;

  /**
   * 登录接口
   */
  @Override
  public BizResponse<LoginResult> login(LoginParam loginParam) {

    // 查询用户信息
    FbUserBase userBase = fbUserBaseService.selectOne(
        new EntityWrapper<FbUserBase>().eq(FbUserBase.USERNAME, loginParam.getUsername())
            .eq(FbUserBase.IS_DEL, 0));

    // 判断是否存在
    if (userBase == null) {

      return BizResponse.fail(ErrorCodeEnum.USERNAME_NOT_EXIST.getCode(), ErrorCodeEnum.USERNAME_NOT_EXIST.getMsg());
    }

    // 判断账号状态
    if (userBase.getStatus() != 1) {

      return BizResponse.fail(ErrorCodeEnum.USER_DISABLE.getCode(), ErrorCodeEnum.USER_DISABLE.getMsg());
    }

    // 比较用户类型是否正确
    Integer userType = loginParam.getUserType();
    if (!userBase.getUserType().equals(userType)) {

      return BizResponse.fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());
    }

    // 封装公共数据
    LoginResult loginResult = new LoginResult();
    // 设置用户 id
    loginResult.setUserId(userBase.getBaseId());
    // 设置 oem id
    loginResult.setOemId(loginParam.getOemId());
    // 设置是否为子账号
    loginResult.setChild(userBase.getIsChild() == 1);
    // 设置账号类型Is
    loginResult.setUserType(userType);

    UserTypeEnum userTypeEnum = UserTypeEnum.valueOf(userType);
    // 获得不同类型帐户信息
    // oem 账号
    if (UserTypeEnum.OEM.equals(userTypeEnum)) {

      // 查询 oem 帐户信息
      FbUserOem userOem = fbUserOemService
          .selectOne(new EntityWrapper<FbUserOem>().eq(FbUserOem.BASE_ID, userBase.getBaseId()));
      if (userOem == null) {

        return BizResponse.fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());

      } else {

        // 设置是否是付呗自身
        loginResult.setSelf(userOem.getIsSelfSupport() == 1);
      }
      // 代理商
    } else if (UserTypeEnum.AGENT.equals(userTypeEnum)) {

      // 查询代理商信息
      FbUserAgent userAgent = fbUserAgentService
          .selectOne(new EntityWrapper<FbUserAgent>().eq(FbUserOem.BASE_ID, userBase.getBaseId()));
      if (userAgent == null) {

        return BizResponse.fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());

      }
      // 受理商
    } else if (UserTypeEnum.GRANT.equals(userTypeEnum)) {

      // 查询授理商信息
      FbUserAgent userAgent = fbUserAgentService
          .selectOne(new EntityWrapper<FbUserAgent>().eq(FbUserOem.BASE_ID, userBase.getBaseId()));
      if (userAgent == null) {

        return BizResponse.fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());

      } else {

        loginResult.setAgentId(userAgent.getBelong());
      }
      // 商户
    } else if (UserTypeEnum.MERCHANT.equals(userTypeEnum)) {

      FbUserMerchant userMerchant = fbUserMerchantService.selectOne(
          new EntityWrapper<FbUserMerchant>().eq(FbUserOem.BASE_ID, userBase.getBaseId()));
      if (userMerchant == null) {

        return BizResponse.fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());

      } else {

        // 用户的授理商或代理商 id
        Long agentId = userMerchant.getAgentId();
        // 查询代理商 id
        FbUserAgent userAgent = fbUserAgentService
            .selectOne(
                new EntityWrapper<FbUserAgent>().eq(FbUserOem.BASE_ID, agentId));
        if (userAgent == null) {

          return BizResponse.fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());

        } else {

          // 授理商
          Long belong = userAgent.getBelong();
          if (belong != 0) {

            loginResult.setAgentId(belong);
            loginResult.setGrantId(agentId);
            // 代理商
          } else {

            loginResult.setAgentId(agentId);
          }
        }
      }

      // 没有匹配的用户类型
    } else {

      return BizResponse.fail(ErrorCodeEnum.USERTYPE_ERROR.getCode(), ErrorCodeEnum.USERTYPE_ERROR.getMsg());
    }

    // 比较密码是否正确
    // 获得请求密码加盐的 md5值
    String requestPwdByMd5 = SecureUtil.md5(loginParam.getPassword() + userBase.getSalt());
    if (!requestPwdByMd5.equals(userBase.getUserpwd())) {

      return BizResponse.fail(ErrorCodeEnum.PASSWORD_ERROR.getCode(), ErrorCodeEnum.PASSWORD_ERROR.getMsg());
    }

    // 登录成功
    // 修改最后登录信息
    boolean update = false;
    try {
      update = fbUserBaseService.update(
          new FbUserBase().setLastLoginIp(NetUtil.ipv4ToLong(loginParam.getLoginIp()))
              .setLastLoginTime(new Date()),
          new EntityWrapper<FbUserBase>().eq(FbUserBase.USERNAME, loginParam.getUsername()));

    } catch (Exception e) {

      log.error("方法: login 发生异常， 参数: loginParam = {} ,异常: Ex = {}", JSON.toJSONString(loginParam),
          ExceptionUtils.getStackTrace(e));
    }

    return BizResponse.success(loginResult);
  }

  /**
   * 获得登录配置接口（web）
   */
  @Override
  public BizResponse<LoginConfigResult> getLoginConf(LoginConfigParam loginConfigParam) {

    // 配置类型
    Integer configType = loginConfigParam.getConfigType();
    // oem的标志
    String oemShortName = loginConfigParam.getOemShortName();

    // 获得配置
    FbOemConfig oemConfig = fbOemConfigService
        .selectOne(new EntityWrapper<FbOemConfig>().eq(FbOemConfig.SUB_DOMAIN, oemShortName));

    // 没有配置返回错误
    if (oemConfig == null) {

      return BizResponse.fail(ErrorCodeEnum.NO_HAVE_LOGIN_CONFIG.getCode(), ErrorCodeEnum.NO_HAVE_LOGIN_CONFIG.getMsg());
    }

    LoginConfigResult loginConfigResult = new LoginConfigResult();
    loginConfigResult.setLogoutUrl(oemConfig.getSubDomain());

    // 如果是 Ome
    if (configType.equals(UserTypeEnum.OEM.getValue())) {

      loginConfigResult.setLoginBg(oemConfig.getAdminLoginBg());
      loginConfigResult.setLoginLogo(oemConfig.getAdminLoginLogo());
      loginConfigResult.setTitle(oemConfig.getAdminLoginTitle());
      loginConfigResult.setLogo(oemConfig.getAdminLogo());

      // 授理商或代理商
    } else if (configType.equals(UserTypeEnum.AGENT.getValue()) || configType
        .equals(UserTypeEnum.GRANT.getValue())) {

      // 背景
      loginConfigResult.setLoginBg(oemConfig.getAgentLoginBg());
      // 登录 logo
      loginConfigResult.setLoginLogo(oemConfig.getAgentLoginLogo());
      // 系统 logo
      loginConfigResult.setLogo(oemConfig.getAgentLogo());
      // 后台名称
      loginConfigResult.setTitle(oemConfig.getAgentLoginTitle());

      // 商户后台
    } else if (configType.equals(UserTypeEnum.MERCHANT.getValue())) {

      loginConfigResult.setLoginBg(oemConfig.getUserLoginBg());
      loginConfigResult.setLoginLogo(oemConfig.getUserLoginLogo());
      loginConfigResult.setLogo(oemConfig.getUserLogo());
      loginConfigResult.setTitle(oemConfig.getUserLoginTitle());
    }

    return BizResponse.success(loginConfigResult);
  }
}
