/*
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.fshows.lifecircle.user.service.business.impl.biz;

import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.plugins.Page;
import com.fshows.lifecircle.service.user.openapi.facade.domain.oem.*;
import com.fshows.lifecircle.service.utils.domain.BizResponse;
import com.fshows.lifecircle.service.user.openapi.facade.domain.result.commons.ErrorCodeEnum;
import com.fshows.lifecircle.service.user.openapi.facade.domain.result.oem.*;
import com.fshows.lifecircle.service.utils.domain.PageResult;
import com.fshows.lifecircle.user.service.business.biz.IOemService;
import com.fshows.lifecircle.user.service.business.db.*;
import com.fshows.lifecircle.user.service.business.utils.BijectionUtils;
import com.fshows.lifecircle.user.service.business.utils.KeyGenerator;
import com.fshows.lifecircle.user.service.business.utils.KeyTypeEnum;
import com.fshows.lifecircle.user.service.domain.po.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;


/**
 * package: com.fshows.lifecircle.user.service.business.impl.biz
 *
 * @author:spafka
 * @description:
 * @date: created in 2017-12-15 11:21:54
 */
@Slf4j
@Service
public class OemServiceImpl implements IOemService {


    @Autowired
    IFbUserOemService oemService;
    @Autowired
    IFbUserBaseService baseService;
    @Autowired
    IFbOemConfigService configService;
    @Autowired
    IFbOemNodeService nodeService;
    @Autowired
    IFbOemNodeRightService nodeRightService;
    @Autowired
    IFbOemRightConfigService oemRightConfigService;
    @Autowired
    IFbOemRightControlService oemRightControlService;


    /**
     * 添加 OEM
     *
     * @param param
     * @return
     */
    @Override
    public BizResponse<UserOemDTO> addOem(AddOemParam param) {

        // base id 非空      
        if (null == param.getBaseId()) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), ErrorCodeEnum.INVALID_PARAMETER.getMsg());
        }
        // 如果没有base消息
        if (null == baseService.selectOne(new EntityWrapper<FbUserBase>().eq(FbUserBase.BASE_ID, param.getBaseId()))) {
            return BizResponse.fail(ErrorCodeEnum.USER_NOT_EXIST.getCode(), ErrorCodeEnum.USER_NOT_EXIST.getMsg());
        }

        // oem 已有 baseID
        if (null != oemService.selectOne(new EntityWrapper<FbUserOem>().eq(FbUserOem.BASE_ID, param.getBaseId()))) {
            return BizResponse.fail(ErrorCodeEnum.USER_HAS_EXIST.getCode(), ErrorCodeEnum.USER_HAS_EXIST.getMsg());
        }

        FbUserOem fbUserOem = new FbUserOem();
        BeanUtils.copyProperties(param, fbUserOem);

        fbUserOem.setOemId(KeyGenerator.getKeyByType(KeyTypeEnum.OEM_USER_KEY));

        boolean inserted = oemService.insert(fbUserOem);

        if (inserted) {

            UserOemDTO oemDTO = BijectionUtils.invert(fbUserOem, UserOemDTO.class);
            return BizResponse.success(oemDTO);
        } else {
            return BizResponse.fail(ErrorCodeEnum.DATA_OPERATION_FAILURE.getCode(), ErrorCodeEnum.DATA_OPERATION_FAILURE.getMsg());
        }

    }

    /**
     * 更新 Oem 基础信息
     *
     * @param param
     * @return
     */
    @Override
    public BizResponse updateOem(UpdateOemParam param) {

        FbUserOem oem = null;

        if (null != param.getOemId()) {
            oem = oemService.
                    selectOne(new EntityWrapper<FbUserOem>()
                            .eq(FbUserOem.OEM_ID, param.getOemId()));
        }

        if (null == oem) {
            return BizResponse.fail(ErrorCodeEnum.USER_NOT_EXIST.getCode(), ErrorCodeEnum.USER_NOT_EXIST.getMsg());
        }

        BijectionUtils.invert(param, oem);
        oem.setUpdateTime(null);

        boolean updateEd = oemService.updateById(oem);

        if (updateEd) {
            return BizResponse.success(true);
        } else {
            return BizResponse.fail(ErrorCodeEnum.SERVER_ERROR.getCode(), ErrorCodeEnum.SERVER_ERROR.getMsg());
        }
    }


    /**
     * 添加 Oem 配置表
     *
     * @param param
     * @return
     */
    @Override
    public BizResponse addOemConfig(AddOemConfigParma param) {


        if (null == param.getOemId()) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), ErrorCodeEnum.INVALID_PARAMETER.getMsg());
        }

        //  数据库中是longtext ，没有设定，则设置默认值
        if (null == param.getProtocolContent()) {
            param.setProtocolContent(" ");
        }
        // copy properties
        FbOemConfig oemConfig = BijectionUtils.invert(param, FbOemConfig.class);

        boolean inserted = configService.insert(oemConfig);

        if (inserted) {
            return BizResponse.success(true);
        } else {
            return BizResponse.fail(ErrorCodeEnum.DATA_OPERATION_FAILURE.getCode(), ErrorCodeEnum.DATA_OPERATION_FAILURE.getMsg());
        }

    }


    /**
     * 更新 oem 配置表
     *
     * @param param
     * @return
     */
    @Override
    public BizResponse updateOemConfig(UpdateOemConfigParam param) {


        // oemId 非空校验
        if (null == param.getOemId()) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), ErrorCodeEnum.INVALID_PARAMETER.getMsg());
        }


        FbOemConfig oemConfig = BijectionUtils.invert(param, FbOemConfig.class);

        boolean inserted = configService.update(oemConfig,
                new EntityWrapper<FbOemConfig>().
                        eq(FbOemConfig.OEM_ID, param.getOemId()));

        if (inserted) {
            return BizResponse.success(true);
        } else {
            return BizResponse.fail(ErrorCodeEnum.DATA_OPERATION_FAILURE.getCode(), ErrorCodeEnum.DATA_OPERATION_FAILURE.getMsg());
        }

    }


    /**
     * 查询（分页_模糊搜索）接口
     *
     * @param param
     * @return
     */
    @Override
    public BizResponse<PageResult<SearchOemDTO>> searchOem(SearchOemParam param) {

        if (null == param.getPage()) {
            param.setPage(1);
        }
        if (null == param.getPageSize()) {

            param.setPageSize(20);
        }


        Page<FbUserOem> oemPagez = oemService.selectPage(
                new Page<FbUserOem>(param.getPage(), param.getPageSize()),
                new EntityWrapper<FbUserOem>()
                        .ge(null != param.getCreateTime(), FbUserOem.CREATE_TIME, param.getCreateTime())
                        .like(null != param.getOemName(), FbUserOem.BRAND_NAME, param.getOemName())
                        .orderBy(null != param.getIsAsc(), FbUserOem.CREATE_TIME, param.getIsAsc())
        );


        List<SearchOemDTO> records = BijectionUtils.invertList(oemPagez.getRecords(), SearchOemDTO.class);

        int total = oemPagez.getTotal();

        PageResult<SearchOemDTO> pageResult = new PageResult(total, records);

        return BizResponse.success(pageResult);
    }

    @Override
    public BizResponse<OemResult> getOemByOemId(Long oemID) {

        if (1 == oemID) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), ErrorCodeEnum.INVALID_PARAMETER.getMsg());
        }

        FbUserOem userOem = oemService.selectOne(
                new EntityWrapper<FbUserOem>()
                        .eq(FbUserOem.OEM_ID, oemID)
        );

        FbUserBase userBase = baseService.selectOne(
                new EntityWrapper<FbUserBase>()
                        .eq(FbUserBase.BASE_ID,
                                userOem.getBaseId()));

        FbOemConfig oemConfig = configService.selectOne(
                new EntityWrapper<FbOemConfig>()
                        .eq(FbOemConfig.OEM_ID, oemID)
        );


        OemResult oemResult = new OemResult();

        BijectionUtils.invert(userOem, oemResult);
        BijectionUtils.invert(oemConfig, oemResult);
        BijectionUtils.invert(userBase, oemResult);

        return BizResponse.success(oemResult);

    }





    @Override
    public BizResponse setOemNodeRight(AddOemNodeRightParam param) {


        if (null == param.getNodeId() ||
                null == param.getOemId()) {

            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), ErrorCodeEnum.INVALID_PARAMETER.getMsg());
        }

        FbOemNodeRight hasOne = nodeRightService.selectOne(
                new EntityWrapper<FbOemNodeRight>().eq(FbOemNodeRight.NODE_ID, param.getNodeId())
                        .eq(FbOemNodeRight.OEM_ID, param.getOemId())
        );


        if (null != hasOne) {

            return BizResponse.fail(ErrorCodeEnum.NODE_HAS_EXIST.getCode(), ErrorCodeEnum.NODE_HAS_EXIST.getMsg());
        }


        FbOemNodeRight nodeRight = BijectionUtils.invert(param, FbOemNodeRight.class);

        boolean insert = nodeRightService.insert(nodeRight);

        if (insert) {
            return BizResponse.success(true);
        } else {
            return BizResponse.fail(ErrorCodeEnum.SERVER_ERROR.getCode(), ErrorCodeEnum.SERVER_ERROR.getMsg());
        }
    }


    @Override
    public BizResponse<List<OemNodeListDTO>> getNodez() {

        List<FbOemNode> nodeList = nodeService.selectList(new EntityWrapper<FbOemNode>()
                .orderBy(FbOemNode.LEVEL, true)
                .orderBy(FbOemNode.SORT, true)
                .orderBy(FbOemNode.PID, true)
        );


        List<OemNodeListDTO> oemNodeLists = BijectionUtils.invertList(nodeList, OemNodeListDTO.class);

        return BizResponse.success(oemNodeLists);

    }


    @Override
    public BizResponse<List<OemNodeRightDTO>> getOemNodeRightz(long oemId) {


        List<Map<String, Object>> maps = nodeRightService.selectOemRights(oemId);
        List<OemNodeRightDTO> oemNodeRightDTOS = BijectionUtils.mapsToObjects(maps, OemNodeRightDTO.class);


        return BizResponse.success(oemNodeRightDTOS);

    }


    @Override
    public BizResponse<List<OemRightConfigDTO>> getAllOemRight() {


        List<FbOemRightConfig> fbOemRightConfigs = oemRightConfigService.selectList(
                new EntityWrapper<FbOemRightConfig>()
                        .eq(FbOemRightConfig.STATUS, 1)
        );

        List<OemRightConfigDTO> rightDTOS = BijectionUtils.invertList(fbOemRightConfigs, OemRightConfigDTO.class);

        return BizResponse.success(rightDTOS);


    }


}
