package com.fshows.lifecircle.service.commons.service.pushservices;

import cn.hutool.core.exceptions.ExceptionUtil;
import com.alibaba.fastjson.JSON;
import com.fshows.lifecircle.service.commons.dao.FbAppMessageMapperExt;
import com.fshows.lifecircle.service.commons.manager.formModels.ApiPushModel;
import com.fshows.lifecircle.service.commons.openapi.facade.domain.result.ErrorCode;
import com.fshows.lifecircle.service.commons.service.contants.PushConstant;
import com.fshows.lifecircle.service.utils.domain.BizResponse;
import com.google.common.collect.Maps;
import com.xiaomi.xmpush.server.Constants;
import com.xiaomi.xmpush.server.Message;
import com.xiaomi.xmpush.server.Result;
import com.xiaomi.xmpush.server.Sender;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Map;

/**
 * @user：52331
 * @packageName：com.fshows.lifecircle.service.service.pushServices
 * @description：描述：小米推送Client
 * @updateDesription：更新记录描述
 * @date：2017/12/11
 * @author：Wendy
 */
@Service
@Slf4j
public class MiPushService {
    @Autowired
    private FbAppMessageMapperExt fbAppMessageMapperExt;
    private final static int RETRIES = 0;

    /**
     * CLient Start
     * @param apiPushModel
     * @return
     */
    public BizResponse miPush(ApiPushModel apiPushModel) {
        Long messageId = apiPushModel.getMessageId();
        String title = apiPushModel.getTitle();
        String description = apiPushModel.getDescription();
        Integer passThrough = apiPushModel.getPassThrough();
        Integer notifyType = apiPushModel.getNotifyType();
        Integer actionType = apiPushModel.getActionType();
        String actionUrl = apiPushModel.getActionUrl();
        String extendContent = apiPushModel.getExtendContent();
        Integer timeOut = apiPushModel.getTimeOut();

        if (StringUtils.isBlank(title) || title.length() > 16) {
            return BizResponse.fail(ErrorCode.PUSH_CODE_ERROR.getCode(), "小米推送标题限制16字符以内");
        }

        if (StringUtils.isBlank(description)) {
            return BizResponse.fail(ErrorCode.PUSH_CODE_ERROR.getCode(), "小米推送描述不可以为空");
        }
        String messagePayload = apiPushModel.getBody();
        String regId = apiPushModel.getTargetValue();
        Result result = null;

        try {
            result = sendMessageToRegId(messageId, title, description, messagePayload, passThrough, actionType, actionUrl, notifyType, extendContent, regId, timeOut);
        } catch (Exception e) {
            log.error("MiPush -- >> 推送失败 messageId = {}, 原因：e = {}", messageId, ExceptionUtil.getMessage(e));
        }

        if (result == null) {
            return BizResponse.fail(ErrorCode.PUSH_CODE_ERROR.getCode(), ErrorCode.PUSH_CODE_ERROR.getMsg());
        }

        /**
         * 发送消息返回的错误码，如果返回ErrorCode.Success表示发送成功，其他表示发送失败。
         */
        com.xiaomi.push.sdk.ErrorCode errorMessage = result.getErrorCode();
        /**
         * 如果消息发送成功，服务器返回消息的ID；如果发送失败，返回null。
         */
        String platformMessageId = result.getMessageId();

        /**
         * 	如果消息发送失败，服务器返回错误原因的文字描述；如果发送成功，返回null。
         */
        String reason = result.getReason();
        /**
         * 定义一个返回值 MAP
         */
        /**
         * 文字描述null&&返回消息id -- 表示真正成功！
         */
        if (StringUtils.isBlank(reason) && StringUtils.isNotBlank(platformMessageId)) {
            fbAppMessageMapperExt.updateMessageByStatus(apiPushModel.getMessageId(), 1);
            Map<String, Object> data = Maps.newHashMap();
            data.put("messageId", messageId);
            data.put("result", result);
            return BizResponse.success(data);
        } else if (StringUtils.isBlank(platformMessageId)) {
            return BizResponse.fail(ErrorCode.PUSH_CODE_ERROR.getCode(), JSON.toJSONString(result));
        } else {
            return BizResponse.fail(ErrorCode.PUSH_CODE_ERROR.getCode(), JSON.toJSONString(result));
        }
    }

    /**
     * 根据regid发送一条短信
     * releaseClient: Constants.useOfficial();
     * testClient: Constants.useSandbox();
     *
     * @param messageId
     * @param title
     * @param description
     * @param messagePayload
     * @param passThrough
     * @param actionType
     * @param actionUrl
     * @param notifyType
     * @param extendContent
     * @param regId
     * @param timeOut
     * @throws Exception
     */
    private Result sendMessageToRegId(Long messageId, String title, String description,
                                      String messagePayload, Integer passThrough, Integer actionType,
                                      String actionUrl, Integer notifyType, String extendContent, String regId,
                                      Integer timeOut) {
        Constants.useOfficial();
        Sender sender = new Sender(PushConstant.MI_APPSECRET);
        Message message = buildMessageForAndroid(title, description, messagePayload, passThrough, actionType, actionUrl, notifyType, extendContent, timeOut);
        if (message == null || sender == null) {
            log.error("MiPush ---- >> 小米推送失败 原因：参数拼装失败 messageId = {}, message = null, sender = null", messageId);
        }
        Result result = null;
        try {
            result = sender.send(message, regId, RETRIES);
        } catch (Exception e) {
            log.error("MiPush ---- >> 小米推送失败 原因：推送失败 messageId = {}, e = {}", messageId, ExceptionUtil.getMessage(e));
        }
        return result;
    }

    /**
     * 构建android推送信息
     *
     * @param title
     * @param description
     * @param messagePayload
     * @param passThrough
     * @param actionType
     * @param actionUrl
     * @param notifyType
     * @param extendContent
     * @param timeOut
     * @return
     */
    private Message buildMessageForAndroid(String title, String description, String messagePayload,
                                           Integer passThrough, Integer actionType, String actionUrl,
                                           Integer notifyType, String extendContent, Integer timeOut) {

        Message message;
        Message.Builder builder = new Message.Builder()
                .title(title)
                .description(description)
                .payload(messagePayload)
                .restrictedPackageName(PushConstant.MI_ANDROID_PACKAGE_NAME)
                .passThrough(passThrough)
                .notifyType(notifyType)
                .timeToLive(timeOut * 1000)
                .extra("extend_content", extendContent);

        if (Integer.valueOf(1).equals(actionType)) {
            builder.extra(Constants.EXTRA_PARAM_NOTIFY_EFFECT, Constants.NOTIFY_ACTIVITY)
                    .extra(Constants.EXTRA_PARAM_INTENT_URI, actionUrl);
        } else if (Integer.valueOf(2).equals(actionType)) {
            builder.extra(Constants.EXTRA_PARAM_NOTIFY_EFFECT, Constants.NOTIFY_WEB)
                    .extra(Constants.EXTRA_PARAM_WEB_URI, actionUrl);
        } else {
            builder.extra(Constants.EXTRA_PARAM_NOTIFY_EFFECT, Constants.NOTIFY_LAUNCHER_ACTIVITY);
        }
        message = builder.build();
        return message;
    }
}
