package com.fshows.lifecircle.service.commons.service.pushservices;

import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.fshows.lifecircle.service.commons.dao.FbAppMessageMapperExt;
import com.fshows.lifecircle.service.commons.manager.formModels.ApiPushModel;
import com.fshows.lifecircle.service.commons.openapi.facade.domain.result.ErrorCode;
import com.fshows.lifecircle.service.commons.service.contants.PushConstant;
import com.fshows.lifecircle.service.utils.domain.BizResponse;
import com.fshows.lifecircle.service.utils.domain.ErrorCodeEnum;
import com.gexin.fastjson.JSON;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import java.net.URLEncoder;
import java.text.MessageFormat;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @user：52331
 * @packageName：com.fshows.lifecircle.service.service.pushServices
 * @description：描述
 * @updateDesription：更新记录描述
 * @date：2017/12/11
 * @author：Wendy
 */
@Service
@Slf4j
public class HvPushService {
    /**
     * 用户在华为开发者联盟申请的appId和appSecret（会员中心->应用管理，点击应用名称的链接）
     */
    private static final String appId = PushConstant.HV_APPID;

    /**
     * 应用级消息下发API
     */
    private static final String apiUrl = PushConstant.HV_API_URL;

    private static final String appSecret = PushConstant.HV_APPSECRET;

    /**
     * 获取认证Token的URL
     */
    private static final String tokenUrl = PushConstant.HV_TOKEN_URL;
    private static final String androidPageName = PushConstant.HV_ANDROID_PACKAGE_NAME;
    private static final String logoUrl = PushConstant.HV_LOGO_URL;
    private static final String redisKey = "SERVICE_HV_ACCESS_TOKEN";
    private static final String tokenTimeKey = "SERVICE_TOKEN_EXPIRED_TIME";

    /**
     * 超时时间
     */
    Long redisTimeOut = 604800L;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;
    @Autowired
    private FbAppMessageMapperExt fbAppMessageMapperExt;

    /**
     * Clien Start
     * @param apiPushModel
     * @return
     */
    public BizResponse hvPush(ApiPushModel apiPushModel) {
        /**
         * 下发通知消息的认证Token
         */
        Map<String, String> data = getToken();
        String accessToken = data.get(redisKey);
        if (StringUtils.isBlank(accessToken)) {
            return BizResponse.fail(ErrorCode.PUSH_CODE_ERROR.getCode(), ErrorCode.PUSH_CODE_ERROR.getMsg());
        }

        String deviceToken = apiPushModel.getTargetValue();
        String title = apiPushModel.getTitle();
        String text = apiPushModel.getBody();
        String actionUrl = apiPushModel.getActionUrl();
        String extendContent = apiPushModel.getExtendContent();
        Integer timeOut = apiPushModel.getTimeOut();
        Long messageId = apiPushModel.getMessageId();

        /**
         * 类型3为打开APP，其他行为请参考接口文档设置
         */
        Integer actionType = apiPushModel.getActionType();
        /**
         * 目标设备
         */
        JSONArray deviceTokens = new JSONArray();
        deviceTokens.add(deviceToken);
        /**
         * 3: 1 透传异步消息 3 系统通知栏异步消息（推荐）
         */
        Integer msgType = apiPushModel.getPassThrough();
        if (0 == msgType) {
            msgType = 3;
        }
        /**
         * 获取封装好的payLoad
         */
        JSONObject payload = encapsulationPayload(title, text, extendContent, actionType, actionUrl, msgType);

        String postBody;
        Map<String, Object> req = Maps.newHashMap();
        try {
            postBody = MessageFormat.format(
                    "access_token={0}&nsp_svc={1}&nsp_ts={2}&device_token_list={3}&payload={4}&expire_time{5}",
                    URLEncoder.encode(accessToken, "UTF-8"),
                    URLEncoder.encode("openpush.message.api.send", "UTF-8"),
                    URLEncoder.encode(String.valueOf(System.currentTimeMillis() / 1000), "UTF-8"),
                    URLEncoder.encode(deviceTokens.toString(), "UTF-8"),
                    URLEncoder.encode(payload.toString(), "UTF-8"),
                    URLEncoder.encode(String.valueOf(System.currentTimeMillis() / 1000 + timeOut), "UTF-8"));


            String postUrl = apiUrl + "?nsp_ctx=" + URLEncoder.encode("{\"ver\":\"1\", \"appId\":\"" + appId + "\"}", "UTF-8");

            String request = HttpUtil.post(postUrl, postBody);

            req = JSON.parseObject(request);

        } catch (Exception e) {
            log.error("HvPush -- 华为推送失败！messageId = {}, 原因：e = {}", messageId, ExceptionUtil.getMessage(e));
        }

        String CODE = "code";
        String SUCCESSCODE = "80000000";
        String MSG = "msg";
        String SUCCESS = "SUCCESS";

        if (req.get(CODE).toString().equals(SUCCESSCODE) && req.get(MSG).toString().toUpperCase().equals(SUCCESS)) {
            fbAppMessageMapperExt.updateMessageByStatus(apiPushModel.getMessageId(), 1);
            return BizResponse.success(req);
        } else {
            return BizResponse.fail(ErrorCodeEnum.DATA_OPERATION_FAILURE.getCode(), ErrorCodeEnum.DATA_OPERATION_FAILURE.getMsg());
        }
    }

    /**
     * 封装payLoad
     *
     * @param title
     * @param text
     * @param extendContent
     * @param actionUrl
     * @param actionType
     * @param msgType
     * @return
     */
    public JSONObject encapsulationPayload(String title, String text, String extendContent, Integer actionType, String actionUrl,
                                           Integer msgType) {
        /**
         * 仅通知栏消息需要设置标题和内容，透传消息key和value为用户自定义
         * 消息标题
         * 消息内容体
         */
        JSONObject body = new JSONObject();
        body.put("title", title);
        body.put("content", text);
        body.put("extend_content", extendContent);

        /**
         * 定义需要打开的appPkgName
         */
        JSONObject param1 = new JSONObject();
        param1.put("appPkgName", actionUrl);
        /**
         * 定义需要打开的URL
         */
        JSONObject param2 = new JSONObject();
        param2.put("url", actionUrl);
        /**
         * 自定义行为
         */
        JSONObject param3 = new JSONObject();
        param3.put("intent", actionUrl);


        /**
         * 消息点击动作参数
         */
        JSONObject action = new JSONObject();

        switch (actionType) {
            case 1:
                action.put("type", 3);
                action.put("param", param1);
                break;
            case 2:
                action.put("type", 2);
                action.put("param", param2);
                break;
            default:
                action.put("type", 1);
                action.put("param", param3);

        }


        JSONObject msg = new JSONObject();
        msg.put("type", msgType);
        msg.put("action", action);//消息点击动作
        msg.put("body", body);//通知栏消息body内容

        JSONObject ext = new JSONObject();//扩展信息，含BI消息统计，特定展示风格，消息折叠。
        ext.put("biTag", "Trump");//设置消息标签，如果带了这个标签，会在回执中推送给CP用于检测某种类型消息的到达率和状态
        ext.put("icon", logoUrl);//自定义推送消息在通知栏的图标,value为一个公网可以访问的URL


        JSONObject hps = new JSONObject();//华为PUSH消息总结构体
        hps.put("msg", msg);
        hps.put("ext", ext);

        JSONObject payload = new JSONObject();

        payload.put("hps", hps);

        return payload;
    }

    /**
     * 获取缓存中的token，不存在重新获取
     */
    public Map<String, String> getToken() {
        String value = stringRedisTemplate.opsForValue().get(redisKey);
        Map<String, String> data = (Map<String, String>) JSON.parse(value);
        if (StringUtils.isBlank(value) || StringUtils.isBlank(data.get(redisKey)) || StringUtils.isBlank(data.get(tokenTimeKey))) {
            data = refreshToken();
            String timeOut = data.get(tokenTimeKey);
            redisTimeOut = Long.valueOf(timeOut) - 60 * 5;//扣除5分钟时间防止与华为服务差值
            log.info("redis -- set - method:getToken 信息收集开始  参数 ：redisKey = {}, data = {}, redisTimeOut = {}, type = {}", redisKey, JSON.toJSONString(data), redisTimeOut, TimeUnit.SECONDS);
            stringRedisTemplate.opsForValue().set(redisKey, JSON.toJSONString(data), redisTimeOut, TimeUnit.SECONDS);
            log.info("redis -- set - method:getToken 信息收集结束  参数 ：redisKey = {}, data = {}, redisTimeOut = {}, type = {}", redisKey, JSON.toJSONString(data), redisTimeOut, TimeUnit.SECONDS);
        }
        return data;
    }

    /**
     * 获取下发通知消息的认证Token
     */
    public Map<String, String> refreshToken() {
        String accessToken;//下发通知消息的认证Token
        long tokenExpiredTime;  //accessToken的过期时间
        Map<String, String> data = Maps.newHashMap();
        try {
            String msgBody = MessageFormat.format(
                    "grant_type=client_credentials&client_secret={0}&client_id={1}",
                    URLEncoder.encode(appSecret, "UTF-8"), appId);
            String response = HttpUtil.post(tokenUrl, msgBody);
            JSONObject obj = JSONObject.parseObject(response);
            accessToken = obj.getString("access_token");
            tokenExpiredTime = obj.getLong("expires_in");

            data.put(redisKey, accessToken);
            data.put(tokenTimeKey, tokenExpiredTime + "");
        } catch (Exception e) {
            log.error("HvPush -- >> 认证Token获取失败！原因：e = {}", ExceptionUtil.getMessage(e));
        }
        return data;
    }
}
