package com.fshows.lifecircle.service.commons.service;

import cn.hutool.core.exceptions.ExceptionUtil;
import com.alibaba.fastjson.JSON;
import com.fshows.lifecircle.service.commons.dao.FbAppMessageMapperExt;
import com.fshows.lifecircle.service.commons.domain.FbAppMessageExt;
import com.fshows.lifecircle.service.commons.manager.formModels.ApiPushModel;
import com.fshows.lifecircle.service.commons.service.pushservices.AliPushService;
import com.fshows.lifecircle.service.commons.service.pushservices.GePushService;
import com.fshows.lifecircle.service.commons.service.pushservices.MiPushService;
import com.fshows.lifecircle.service.commons.service.utils.ValidateResult;
import com.fshows.lifecircle.service.commons.service.pushservices.HvPushService;
import com.fshows.lifecircle.service.commons.service.utils.JsonUtil;
import com.fshows.lifecircle.service.commons.service.utils.ValidateUtils;
import com.fshows.lifecircle.service.utils.domain.BizResponse;
import com.fshows.lifecircle.service.utils.domain.ErrorCodeEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Map;

/**
 * @user：52331
 * @packageName：com.fshows.lifecircle.service.service
 * @description：描述：聚合推送Service
 * @updateDesription：更新记录描述
 * @date：2017/12/11
 * @author：Wendy
 */
@Service
@Slf4j
public class ApiPushService {

    @Autowired
    private FbAppMessageMapperExt fbAppMessageMapperExt;
    @Autowired
    private AliPushService aliPushService;
    @Autowired
    private GePushService gePushService;
    @Autowired
    private HvPushService hvPushService;
    @Autowired
    private MiPushService miPushService;

    /**
     * @author: Wendy
     * @description:推送路由
     * @date: 12:12 2017/12/11
     */
    public BizResponse<Boolean> pushRoute(String content) {
        //转换参数
        ApiPushModel apiPushModel = null;
        //参数校验
        try {
            apiPushModel = JsonUtil.jsonHumpToObj(content, ApiPushModel.class);
        } catch (Exception e) {
            log.error("aggregationPush -- >> 聚合推送接口错误：e = {}", ExceptionUtil.getMessage(e));
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), ErrorCodeEnum.INVALID_PARAMETER.getMsg());
        }

        //参数校验
        BizResponse<Boolean> bizResponse = CheckValidate(apiPushModel);
        if (!bizResponse.getData()) {
            return bizResponse;
        }

        //解析参数
        Map<String, Object> extentContent = JSON.parseObject(apiPushModel.getExtendContent());
        Map<String, Object> onMessage = (Map<String, Object>) extentContent.get("on_message");
        Map<String, Object> ext = (Map<String, Object>) onMessage.get("ext");

        if (StringUtils.isBlank(apiPushModel.getExtendContent()) || extentContent == null || onMessage == null
                || ext == null) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), "透传字段不可以为空");
        }
        log.info("aggregationPush -- >> 聚合推送接口参数记录：AggregationPushForm = {}", JSON.toJSONString(apiPushModel));

        String orderSn = onMessage.get("order_sn").toString();
        String messageType = ext.get("message_type").toString();
        if (StringUtils.isBlank(orderSn)) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), "无效订单信息");
        }

        if (StringUtils.isBlank(messageType)) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), "无效消息类型");
        }
        /**
         * 推送消息记录入库
         */
        FbAppMessageExt record = new FbAppMessageExt();
        record.setMessage(apiPushModel.getExtendContent());
        record.setRequestId(apiPushModel.getRequestId());
        record.setMid(apiPushModel.getMid());
        record.setMessage(apiPushModel.getExtendContent());
        record.setReceiverTime(new Date());
        record.setSystemType(apiPushModel.getDeviceType());
        record.setStatus(3);
        record.setOrderSn(orderSn);
        record.setMsgType(Integer.valueOf(messageType));


        try {
            //插入推送记录表
            fbAppMessageMapperExt.insertSelectiveForKey(record);
            apiPushModel.setMessageId(record.getId());

            final ApiPushModel finalForm = apiPushModel;
            //提交推送信息
            pushService(finalForm);
            return BizResponse.success(true);
        } catch (Exception e) {
            log.error("aggregationPush -- >> 推送失败：AggregationPushForm = {}, record = {}, e = {}",
                    JSON.toJSONString(apiPushModel), JSON.toJSONString(record), ExceptionUtil.getMessage(e));
            return BizResponse.fail(ErrorCodeEnum.SERVER_ERROR.getCode(), ErrorCodeEnum.SERVER_ERROR.getMsg());
        }
    }

    /**
     * @author: Wendy
     * @description:检查参数是否合规
     * @date: 16:22 2017/12/11
     */
    public BizResponse<Boolean> CheckValidate(ApiPushModel apiPushModel) {
        //参数验证
        ValidateResult validate = ValidateUtils.validate(apiPushModel);
        if (!validate.isSuccess()) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), ErrorCodeEnum.INVALID_PARAMETER.getMsg());
        }
        //无意义参数拦截
        if (apiPushModel.getNotifyType() == null) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), "NotifyType 无效");
        }

        if (apiPushModel.getActionType() == null) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), "ActionType 无效");
        }

        if (apiPushModel.getPassThrough() == null) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), "PassThrough 无效");
        }


        //检查推送信息是否合规
        switch (apiPushModel.getChennelId()) {
            case 0:
                break;
            case 1:
                if (1 != apiPushModel.getDeviceType()) {
                    return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), "不适用IOS设备");
                }
                break;
            case 2:
                if (1 != apiPushModel.getDeviceType()) {
                    return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), "不适用IOS设备");
                }
                break;
            case 3:
                break;
            default:
                return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), "未知推送渠道");
        }
        return null;
    }

    /**
     * 推送线程开始执行任务 wendy 2017年12月8日16:58:28
     */
    @Async
    public void pushService(ApiPushModel apiPushModel) {
        BizResponse bizResponse;
        switch (apiPushModel.getChennelId()) {
            case 0:
                bizResponse = aliPushService.aliPush(apiPushModel);
                break;
            case 1:
                bizResponse = miPushService.miPush(apiPushModel);
                break;
            case 2:
                bizResponse = hvPushService.hvPush(apiPushModel);
                break;
            case 3:
                bizResponse = gePushService.gtPush(apiPushModel);
                break;
            default:
                bizResponse = null;
                break;
        }
        log.info("pushService --- >> 推送结果记录日志 messageId = {}, apiPushModel = {}, bizResponse = {}",
                apiPushModel.getMessageId().toString(),
                JSON.toJSONString(apiPushModel),
                JSON.toJSONString(bizResponse));
    }
}
