package com.fshows.lifecircle.service.commons.service;

import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.io.resource.ClassPathResource;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.http.MethodType;
import com.aliyuncs.http.ProtocolType;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import com.aliyuncs.sts.model.v20150401.AssumeRoleRequest;
import com.aliyuncs.sts.model.v20150401.AssumeRoleResponse;
import com.fshows.lifecircle.service.commons.openapi.facade.domain.result.AppModelResult;
import com.fshows.lifecircle.service.commons.service.contants.AliConstants;
import com.fshows.lifecircle.service.commons.service.utils.AliOSSUtil;
import com.fshows.lifecircle.service.utils.domain.BizResponse;
import com.fshows.lifecircle.service.utils.domain.ErrorCodeEnum;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.io.*;


/**
 * @user：52331
 * @packageName：com.fshows.lifecircle.service.service
 * @description：描述:阿里云oss存储
 * @updateDesription：更新记录描述
 * @date：2017/12/14
 * @author：Wendy
 */
@Service
@Slf4j
public class AliOssService {

    /**
     * 上传
     *
     * @param bucketName
     * @param ossKey
     * @param fileName
     * @returnttt
     */
    public BizResponse<Boolean> uploadFile(String bucketName, String ossKey, String fileName) {
        if (AliOSSUtil.uploadFile(bucketName, ossKey, fileName)) {
            return BizResponse.success(true);
        }
        return BizResponse.fail(ErrorCodeEnum.REQUEST_METHOD_ERROR.getCode(), ErrorCodeEnum.REQUEST_METHOD_ERROR.getMsg());
    }

    /**
     * 下载文件
     *
     * @param bucketName
     * @param key
     * @param filename
     * @return
     */
    public BizResponse<Boolean> downloadFile(String bucketName, String key, String filename) {
        AliOSSUtil.downloadFile(bucketName, key, filename);
        return BizResponse.success(true);
    }


    /**
     * 下载文件Url返回
     *
     * @param bucketName
     * @param key
     * @return
     */
    public BizResponse<String> fileUrl(String bucketName, String key) {
        String url = AliOSSUtil.fileUrl(bucketName, key);
        if (StringUtils.isNotBlank(url)) {
            String[] newStrs = url.split("://");
            String str = newStrs[1];
            url = "https://" + str;
        }
        return BizResponse.success(url);
    }

    /**
     * 验签上传
     *
     * @param imageType
     * @return
     */
    public BizResponse<String> getToken(Integer imageType) {
        //图片类型1:logo 2:头像 3:商户信息 4:二维码
        String dirName = null;
        if (imageType == 1) {
            dirName = AliConstants.ALI_CLOUD_LOGO_DIR;
        }

        if (imageType == 2) {
            dirName = AliConstants.ALI_CLOUD_HEAD_DIR;
        }

        if (imageType == 3) {
            dirName = AliConstants.ALI_CLOUD_MERCHANTINFO_DIR;
        }

        if (imageType == 4) {
            dirName = AliConstants.ALI_CLOUD_QRCODE_DIR;
        }

        if (StringUtils.isBlank(dirName)) {
            return BizResponse.fail(ErrorCodeEnum.INVALID_PARAMETER.getCode(), "无效的照片参数");
        }
        return BizResponse.success(AliOSSUtil.getToken(dirName));
    }

    /**
     * APP-OSS服务
     * @return
     */
    public BizResponse<AppModelResult> getTokenForApp() {
        String data = ReadJson(new ClassPathResource("config.json").getAbsolutePath());
        JSONObject jsonObj  = JSONObject.fromObject(data);
        // 只有 RAM用户（子账号）才能调用 AssumeRole 接口
        // 阿里云主账号的AccessKeys不能用于发起AssumeRole请求
        // 请首先在RAM控制台创建一个RAM用户，并为这个用户创建AccessKeys
        String accessKeyId = jsonObj.getString("AccessKeyID");
        String accessKeySecret = jsonObj.getString("AccessKeySecret");

        // RoleArn 需要在 RAM 控制台上获取
        String roleArn = jsonObj.getString("RoleArn");
        long durationSeconds = jsonObj.getLong("TokenExpireTime");
        String policy = ReadJson(new ClassPathResource(jsonObj.getString("PolicyFile")).getAbsolutePath());
        // RoleSessionName 是临时Token的会话名称，自己指定用于标识你的用户，主要用于审计，或者用于区分Token颁发给谁
        // 但是注意RoleSessionName的长度和规则，不要有空格，只能有'-' '_' 字母和数字等字符
        // 具体规则请参考API文档中的格式要求
        String roleSessionName = "alice-001";

        // 此处必须为 HTTPS
        ProtocolType protocolType = ProtocolType.HTTPS;

        try {
            final AssumeRoleResponse stsResponse = assumeRole(accessKeyId, accessKeySecret, roleArn, roleSessionName,
                    policy, protocolType, durationSeconds);
            AppModelResult result = new AppModelResult();
            result.setAk(stsResponse.getCredentials().getAccessKeyId());
            result.setSk(stsResponse.getCredentials().getAccessKeySecret());
            result.setToken(stsResponse.getCredentials().getSecurityToken());
            result.setExpiration(stsResponse.getCredentials().getExpiration());
            return BizResponse.success(result);
        } catch (ClientException e) {
            e.printStackTrace();
            log.error("APP - OSS 连接异常： 原因：e = {}", ExceptionUtil.stacktraceToString(e));
            return BizResponse.fail(ErrorCodeEnum.SERVER_ERROR.getCode(), ErrorCodeEnum.SERVER_ERROR.getMsg());
        }
    }

    private AssumeRoleResponse assumeRole(String accessKeyId, String accessKeySecret, String roleArn, String roleSessionName, String policy, ProtocolType protocolType, long durationSeconds) throws ClientException {
        // 目前只有"cn-hangzhou"这个region可用, 不要使用填写其他region的值
        String REGION_CN_HANGZHOU = "cn-hangzhou";
        String STS_API_VERSION = "2015-04-01";
        try {
            // 创建一个 Aliyun Acs Client, 用于发起 OpenAPI 请求
            IClientProfile profile = DefaultProfile.getProfile(REGION_CN_HANGZHOU, accessKeyId, accessKeySecret);
            DefaultAcsClient client = new DefaultAcsClient(profile);

            // 创建一个 AssumeRoleRequest 并设置请求参数
            final AssumeRoleRequest request = new AssumeRoleRequest();
            request.setVersion(STS_API_VERSION);
            request.setMethod(MethodType.POST);
            request.setProtocol(protocolType);

            request.setRoleArn(roleArn);
            request.setRoleSessionName(roleSessionName);
            request.setPolicy(policy);
            request.setDurationSeconds(durationSeconds);

            // 发起请求，并得到response
            final AssumeRoleResponse response = client.getAcsResponse(request);

            return response;
        } catch (ClientException e) {
            throw e;
        }

    }

    public static String ReadJson(String path){
        //从给定位置获取文件
        File file = new File(path);
        BufferedReader reader = null;
        //返回值,使用StringBuffer
        StringBuffer data = new StringBuffer();
        //
        try {
            reader = new BufferedReader(new FileReader(file));
            //每次读取文件的缓存
            String temp = null;
            while((temp = reader.readLine()) != null){
                data.append(temp);
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }finally {
            //关闭文件流
            if (reader != null){
                try {
                    reader.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return data.toString();
    }


}
