package com.fshows.lifecircle.service.service;

import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONObject;
import com.alibaba.fastjson.JSON;
import com.fshows.lifecircle.service.openapi.facade.domain.result.BankFourFactorResult;
import com.fshows.lifecircle.service.openapi.facade.domain.result.ErrorCode;
import com.fshows.lifecircle.service.openapi.facade.domain.result.IdCardResult;
import com.fshows.lifecircle.service.service.contants.AliConstants;
import com.fshows.lifecircle.service.service.utils.HttpUtils;
import com.fshows.lifecircle.service.utils.domain.BizResponse;
import com.fshows.lifecircle.service.utils.domain.ErrorCodeEnum;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * @user：52331
 * @packageName：com.fshows.lifecircle.service.service
 * @description：描述:阿里云安全认证模块
 * @updateDesription：更新记录描述
 * @date：2018/1/10
 * @author：Wendy
 */
@Service
@Slf4j
public class AliSecurityAuthService {

    @Autowired
    private AliOssService aliOssService;

    /**
     * 实名认证
     */
    public BizResponse<Boolean> checkIdCard(String name, String cardId) {
        String host = "http://aliyunverifyidcard.haoservice.com";
        String path = "/idcard/VerifyIdcardv2";
        String method = "GET";
        String appcode = AliConstants.ALI_AUTH_APP_CODE;
        Map<String, String> headers = new HashMap<String, String>();
        //最后在header中的格式(中间是英文空格)为Authorization:APPCODE 83359fd73fe94948385f570e3c139105
        headers.put("Authorization", "APPCODE " + appcode);
        Map<String, String> querys = new HashMap<>();
        querys.put("cardNo", cardId);
        querys.put("realName", name);

        try {

            HttpResponse response = HttpUtils.doGet(host, path, method, headers, querys);
            //获取response的body
            String result = EntityUtils.toString(response.getEntity());

            log.info("checkIdCard -- >> 实名认证：参数：name = {}, cardId = {}, result = {}", name, cardId, result);
            if (StringUtils.isBlank(result)) {
                return BizResponse.success(false);
            }
            Map<String, Object> map = (Map<String, Object>) JSON.parse(result);
            if (map.get("reason").toString().toUpperCase().equals("SUCCESS") || map.get("reason").equals("成功")) {
                Map<String, Object> rs = (Map<String, Object>) map.get("result");
                if ((boolean) rs.get("isok")) {
                    return BizResponse.success(true);
                } else {
                    return BizResponse.fail(ErrorCode.AUTH_ERROR.getCode(), ErrorCode.AUTH_ERROR.getMsg());
                }
            } else {
                return BizResponse.fail(ErrorCode.AUTH_ERROR.getCode(), ErrorCode.AUTH_ERROR.getMsg());
            }
        } catch (Exception e) {
            log.error("checkIdCard ---- >> 实名认证异常，请检查ali认证接口bug, e = {}", ExceptionUtil.getMessage(e));
            return BizResponse.fail(ErrorCode.AUTH_ERROR.getCode(), ErrorCode.AUTH_ERROR.getMsg());
        }
    }

    /**
     * 身份证认证
     */
    public BizResponse<IdCardResult> checkIdCardPicture(String bucketName, String key) {
        //正面照片传输格式
        String type = "{\\\"side\\\":\\\"face\\\"}";
        String fileName = AliConstants.IDCARD_PHOTO_DIR + key + ".jpg";
        aliOssService.downloadFile(bucketName, key, fileName);
        File file = new File(fileName);

        // 对图像进行base64编码
        String imgBase64 = "";
        try {
            byte[] content = new byte[(int) file.length()];
            FileInputStream finputstream = new FileInputStream(file);
            finputstream.read(content);
            finputstream.close();
            imgBase64 = new String(Base64.encodeBase64(content));
        } catch (IOException e) {
            e.printStackTrace();
            log.info("checkIdCardPicture -- >> 身份证验证失败 ： e = {}", ExceptionUtil.stacktraceToString(e));
            return BizResponse.fail(ErrorCode.AUTH_ERROR.getCode(), ErrorCode.AUTH_ERROR.getMsg());
        }

        //发送
        IdCardResult result = sendCheckIdCard(imgBase64, type);
        //照片使用完删除掉
        file.delete();
        return BizResponse.success(result);
    }


    /**
     * 银行卡鉴权四要素认证
     */
    public BizResponse<BankFourFactorResult> checkBankFourFactor(String name, String acctPan, String certId, String phoneNum) {
        String host = "http://ali-bankcard4.showapi.com";
        String path = "/bank4";
        String method = "GET";
        String appcode = AliConstants.ALI_AUTH_APP_CODE;
        Map<String, String> headers = new HashMap<String, String>();
        //最后在header中的格式(中间是英文空格)为Authorization:APPCODE 83359fd73fe94948385f570e3c139105
        headers.put("Authorization", "APPCODE " + appcode);
        Map<String, String> querys = new HashMap<String, String>();
        querys.put("acct_name", name);
        querys.put("acct_pan", acctPan);
        querys.put("cert_id", certId);
        querys.put("cert_type", "01");
        querys.put("needBelongArea", "true");
        querys.put("phone_num", phoneNum);
        BankFourFactorResult result = new BankFourFactorResult();
        try {
            /**
             * 重要提示如下:
             * HttpUtils请从
             * https://github.com/aliyun/api-gateway-demo-sign-java/blob/master/src/main/java/com/aliyun/api/gateway/demo/util/HttpUtils.java
             * 下载
             *
             * 相应的依赖请参照
             * https://github.com/aliyun/api-gateway-demo-sign-java/blob/master/pom.xml
             */
            HttpResponse response = HttpUtils.doGet(host, path, method, headers, querys);
            //获取response的body
            String body = EntityUtils.toString(response.getEntity());

            Map<String, Object> bodyMap= (Map<String, Object>) JSON.parse(body);

            log.info("checkBankFourFactor -- >> 银行四要素校验结果 ：name = {}, acctPan = {}, certId = {}, phoneNum = {}, bodyMap = {}", name, acctPan, certId, phoneNum, JSON.toJSONString(bodyMap));
            if (bodyMap == null) {
                result.setIsTrue(false);
                result.setMsg("认证未通过！");
                return BizResponse.success(result);
            }
            Map<String, Object> showapiResBody = (Map<String, Object>) bodyMap.get("showapi_res_body");

            if (showapiResBody == null) {
                result.setIsTrue(false);
                result.setMsg("认证未通过！");
                return BizResponse.success(result);
            }

            if (!showapiResBody.get("code").toString().equals("0")) {
                result.setIsTrue(false);
                result.setMsg("认证未通过！" + showapiResBody.get("msg").toString());
                return BizResponse.success(result);
            }

            if (!showapiResBody.get("ret_code").toString().equals("0")) {
                result.setIsTrue(false);
                result.setMsg("认证未通过！" + showapiResBody.get("msg").toString());
                return BizResponse.success(result);
            }

            result.setMsg(showapiResBody.get("msg").toString());
            result.setBelong(JSON.toJSONString(showapiResBody.get("belong")));
            result.setIsTrue(true);
            return BizResponse.success(result);
        } catch (Exception e) {
            e.printStackTrace();
            log.info("checkBankFourFactor -- >> 银行卡四要素校验失败！错误原因：e = {}", ExceptionUtil.getMessage(e));
            result.setIsTrue(false);
            result.setMsg("认证未通过！");
            return BizResponse.success(result);
        }


    }


    /**
     * 身份证验证
     *
     * @param imgBase64
     * @param type
     * @return
     */
    public IdCardResult sendCheckIdCard(String imgBase64, String type) {

        String host = "https://dm-51.data.aliyun.com";
        String path = "/rest/160601/ocr/ocr_idcard.json";
        String method = "POST";
        String appcode = AliConstants.ALI_AUTH_APP_CODE;
        Map<String, String> headers = new HashMap<String, String>();
        //最后在header中的格式(中间是英文空格)为Authorization:APPCODE 83359fd73fe94948385f570e3c139105
        headers.put("Authorization", "APPCODE " + appcode);
        //根据API的要求，定义相对应的Content-Type
        headers.put("Content-Type", "application/json; charset=UTF-8");
        Map<String, String> querys = Maps.newHashMap();
        try {
            String bodys = "{\"inputs\": [{" +
                    "\"image\": {" +
                    "\"dataType\": 50," +
                    "\"dataValue\": \"" + imgBase64 + "\"}," +
                    "\"configure\": {" +
                    "\"dataType\": 50," +
                    "\"dataValue\": \"" + type + "\"}" +
                    "}]}"; // #身份证正反面类型:face/back

            HttpResponse response = HttpUtils.doPost(host, path, method, headers, querys, bodys);
            //获取response的body
            String body = EntityUtils.toString(response.getEntity());


            if (StringUtils.isBlank(body)) {
                return null;
            }
            JSONObject resultObj = new JSONObject(body);
            JSONArray outputArray = resultObj.getJSONArray("outputs");
            JSONObject output = outputArray.getJSONObject(0).getJSONObject("outputValue");
            String jsonMessage = (String) output.get("dataValue");
            Map<String, Object> map = (Map<String, Object>) JSON.parse(jsonMessage);

            if (!(boolean) map.get("success")) {
                return null;
            }
            String name = (String) map.get("name");
            String address = (String) map.get("address");
            String num = (String) map.get("num");

            IdCardResult result = new IdCardResult();
            result.setName(name);
            result.setNum(num);
            result.setAddress(address);
            return result;
        } catch (Exception e) {
            log.error("checkIdCardPicture 发送异常 e = {}", ExceptionUtil.getMessage(e));
            return null;
        }
    }
}
