/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.lakala.client.impl;

import cn.hutool.core.util.RandomUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.lakala.client.base.ApiClientConfig;
import com.fshows.lakala.client.base.IHttpRequest;
import com.fshows.lakala.client.base.ILakalaApiDefinition;
import com.fshows.lakala.client.base.ISigner;
import com.fshows.lakala.client.base.SignParam;
import com.fshows.lakala.constant.LakalaConstant;
import com.fshows.lakala.exception.LakalaApiException;
import com.fshows.lakala.request.base.LakalaBaseMerchantRequest;
import com.fshows.lakala.request.base.LakalaBizRequest;
import com.fshows.lakala.response.base.LakalaBaseMerchantResponse;
import com.fshows.lakala.response.base.LakalaBizResponse;
import com.fshows.lakala.util.ReqIdUtil;

/**
 * 拉卡拉商户进价相关客户端
 *
 * @author liluqing
 * @version MerchantLakalaApiClientImpl.java, v 0.1 2022-03-04 15:02
 */
public class MerchantLakalaApiClientImpl extends AbstractLakalaApiClientImpl<LakalaBaseMerchantRequest, LakalaBaseMerchantResponse> {

    private static SerializeConfig serializeConfig = new SerializeConfig();

    static {
        serializeConfig.propertyNamingStrategy = PropertyNamingStrategy.CamelCase;
    }

    public MerchantLakalaApiClientImpl(ApiClientConfig apiClientConfig) throws LakalaApiException {
        super(apiClientConfig);
    }

    public MerchantLakalaApiClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest, ISigner signer) throws LakalaApiException {
        super(apiClientConfig, httpRequest, signer);
    }

    @Override
    public <T extends LakalaBizResponse, R extends ILakalaApiDefinition> LakalaBaseMerchantResponse<T> execute(LakalaBizRequest<T, R> request, R merchantApiDefinitionEnum) throws LakalaApiException {
        return super.doExecute(request, merchantApiDefinitionEnum);
    }

    /**
     * 添加公共参数构建完整的请求
     *
     * @param bizRequest                业务请求参数
     * @param merchantApiDefinitionEnum
     * @return
     */
    @Override
    protected LakalaBaseMerchantRequest buildLakalaBizRequest(LakalaBizRequest bizRequest, ILakalaApiDefinition merchantApiDefinitionEnum) {
        LakalaBaseMerchantRequest request = new LakalaBaseMerchantRequest();
        request.setTimestamp(System.currentTimeMillis());
        request.setRnd(RandomUtil.randomString(12));
        request.setVer(merchantApiDefinitionEnum.getVersion());
        request.setReqId(ReqIdUtil.getId());
        request.setReqData(bizRequest);
        return request;
    }

    @Override
    protected SignParam buildSignParam(LakalaBaseMerchantRequest lakalaBaseRequest, String bodyStr) {
        SignParam signParam = new SignParam();
        signParam.setAppid(apiClientConfig.getAppId());
        signParam.setSerialNo(apiClientConfig.getSerialNo());
        signParam.setBody(bodyStr);
        signParam.setNonceStr(lakalaBaseRequest.getRnd());
        signParam.setTimeStamp(lakalaBaseRequest.getTimestamp() / 1000L);
        return signParam;
    }

    @Override
    protected String serializableRequest(LakalaBaseMerchantRequest lakalaBaseRequest, ILakalaApiDefinition merchantApiDefinitionEnum) {
        return JSONObject.toJSONString(lakalaBaseRequest, serializeConfig);
    }

    @Override
    protected LakalaBaseMerchantResponse parseResponse(String resBody, ILakalaApiDefinition merchantApiDefinitionEnum) {
        JSONObject resJson = JSONObject.parseObject(resBody);

        // 业务响应结果
        LakalaBizResponse bizResData = (LakalaBizResponse) resJson.getObject(LakalaConstant.LAKALA_RESPONSE_BIZ_KEY, merchantApiDefinitionEnum.getResponseClass());
        resJson.remove(LakalaConstant.LAKALA_RESPONSE_BIZ_KEY);

        // 完整响应结果
        LakalaBaseMerchantResponse response = resJson.toJavaObject(LakalaBaseMerchantResponse.class);
        response.setRespData(bizResData);
        return response;
    }
}