/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows;

import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fshows.apienum.JlpayApiDefinitionEnum;
import com.fshows.request.JlBizRequest;
import com.fshows.response.JlBizResponse;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import com.fshows.sdk.core.util.ReqIdUtil;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;

import com.jlpay.open.jlpay.sdk.java.exception.HttpExecutionException;
import com.jlpay.open.jlpay.sdk.java.exception.HttpStatusCodeException;
import com.jlpay.open.jlpay.sdk.java.http.HttpMethod;
import com.jlpay.open.jlpay.sdk.java.http.HttpRequest;
import com.jlpay.open.jlpay.sdk.java.http.HttpResponse;
import com.jlpay.open.jlpay.sdk.java.http.JlpayHttpHeaders;
import com.jlpay.open.jlpay.sdk.java.sign.SignVerifier;
import com.jlpay.open.jlpay.sdk.java.sign.SignVerifierManager;
import com.jlpay.open.jlpay.sdk.java.utils.HttpUtils;
import com.jlpay.open.jlpay.sdk.java.utils.NonceUtils;
import lombok.extern.slf4j.Slf4j;
import okhttp3.Headers;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * 嘉联通道客户端
 *
 * @author
 * @version JlPayApiClient.java, v 0.1 2025-01-08 17:37:48
 */
@Slf4j
public class JlPayApiClient extends AbstractApiClient<JlBizRequest, JlBizResponse, JlpayApiDefinitionEnum> {

    private final SignVerifierManager signVerifierManager;

    private static final ObjectMapper INSTANCE;

    static {
        INSTANCE = new ObjectMapper();
        // 配置 ObjectMapper
        INSTANCE.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        INSTANCE.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
        INSTANCE.configure(DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_AS_NULL, true);
        INSTANCE.configure(DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_USING_DEFAULT_VALUE, true);
        INSTANCE.enable(DeserializationFeature.FAIL_ON_NUMBERS_FOR_ENUMS);
        INSTANCE.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        INSTANCE.setPropertyNamingStrategy(com.fasterxml.jackson.databind.PropertyNamingStrategy.SNAKE_CASE);
        // 设置日期格式
        INSTANCE.setDateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));
        // 设置时区
        INSTANCE.setTimeZone(TimeZone.getTimeZone("GMT+8"));
    }

    public JlPayApiClient(DefaultClientConfigModel apiClientConfig) throws FsApiException {
        super(apiClientConfig);
        try {
            SignVerifier signVerifier = new SignVerifier(apiClientConfig.getFubeiPrivateKey(), apiClientConfig.getPayCompanyPublicKey());
            signVerifierManager = new SignVerifierManager(signVerifier);
        } catch (Exception e) {
            LogUtil.error(log, "JlPayApiClient >> 嘉联通道公私钥加载失败! ", e);
            throw new FsApiException("加载公私钥失败", e);
        }
    }

    @Override
    public JlBizResponse execute(JlBizRequest request, JlpayApiDefinitionEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public JlBizResponse execute(JlBizRequest request, JlpayApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

    /**
     * 执行请求
     *
     * @param request
     * @param iApiDefinition
     * @param payConfigModel
     * @return
     * @throws FsApiException
     */
    @Override
    protected JlBizResponse doExecute(JlBizRequest request, JlpayApiDefinitionEnum iApiDefinition, DefaultClientConfigModel payConfigModel) throws FsApiException {
        long beginTime = System.currentTimeMillis();
        LogUtil.info(log, "{} >> 请求开始 >> iApiDefinition={}, request={}",
                getClientInfo().getClientDesc(), iApiDefinition, request);
        // 构建请求上下文
        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, payConfigModel);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;
        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);
            // 请求开始时间
            long reqBeginTime = System.currentTimeMillis();
            // 执行post请求
            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            // 请求结束时间
            long reqEndTime = System.currentTimeMillis();
            Object requestBody = ObjectUtils.defaultIfNull(apiRequestModel.getRequestBody(), apiRequestModel.getRequestForm());
            LogUtil.info(log, "{} >> 请求结束 >> url={}, method={}, request={}, response={}, cost={}ms, reqCost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), iApiDefinition, requestBody,
                    apiResponseModel.getResponseBody(), System.currentTimeMillis() - beginTime, reqEndTime - reqBeginTime);

            // 反序列化构建响应结果
            return buildApiResponse(apiResponseModel, apiRequestModel, requestContext);
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求异常 >> url={}, method={}, request={}, cost={}ms", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel == null ? "" : apiRequestModel.getApiURL(),
                    requestContext.getIApiDefinition(),
                    JSONObject.toJSONString(apiRequestModel == null ? "" : apiRequestModel.getRequest()),
                    System.currentTimeMillis() - beginTime);
            throw e;
        } catch (Exception e) {
            LogUtil.info(log, "{} >> 请求未知异常 >> url={}, method={}, request={}, cost={}ms", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel == null ? "" : apiRequestModel.getApiURL(),
                    requestContext.getIApiDefinition(),
                    JSONObject.toJSONString(apiRequestModel == null ? "" : apiRequestModel.getRequest()),
                    System.currentTimeMillis() - beginTime);
            throw new FsApiException(e.getMessage(), e);
        }
    }

    /**
     * 构建请求上下文
     *
     * @param tradeApiDefinitionEnum
     * @param request
     * @return
     */
    @Override
    protected DefaultRequestContext buildRequestContext(JlpayApiDefinitionEnum tradeApiDefinitionEnum, JlBizRequest request, DefaultClientConfigModel payConfig) {

        DefaultRequestContext context = new DefaultRequestContext();
        context.setRequestId(ReqIdUtil.getId());
        context.setIApiDefinition(tradeApiDefinitionEnum);
        // 复制默认参数
        DefaultClientConfigModel config = BeanUtil.copyProperties(this.apiClientConfig, DefaultClientConfigModel.class);
        if (payConfig != null) {
            config.setAgentId(StringUtils.isEmpty(payConfig.getAgentId()) ? apiClientConfig.getAgentId() : payConfig.getAgentId());
            config.setAppId(StringUtils.isEmpty(payConfig.getAppId()) ? apiClientConfig.getAppId() : payConfig.getAppId());
            // 私钥
            config.setFubeiPrivateKey(StringUtils.isEmpty(payConfig.getFubeiPrivateKey()) ? apiClientConfig.getFubeiPrivateKey() : payConfig.getFubeiPrivateKey());
            // 公钥
            config.setPayCompanyPublicKey(StringUtils.isEmpty(payConfig.getPayCompanyPublicKey()) ? apiClientConfig.getPayCompanyPublicKey() : payConfig.getPayCompanyPublicKey());
        }
        context.setApiClientConfig(config);
        context.setClientInfoModel(getClientInfo());
        return context;
    }

    /**
     * 构建请求参数
     *
     * @param request
     * @param context
     * @return
     */
    @Override
    protected ApiRequestModel buildApiRequestModel(JlBizRequest request, DefaultRequestContext context) {
        // sdk客户端配置
        DefaultClientConfigModel configModel = context.getApiClientConfig();

        ApiRequestModel apiRequestModel = new ApiRequestModel();
        // 设置网关地址
        apiRequestModel.setApiURL(configModel.getApiParentURL() + context.getIApiDefinition().getApiURI());

        // 请求头信息
        Map<String, String> headerMap = new HashMap<>();
        headerMap.put(JlpayHttpHeaders.CONTENT_TYPE, JlpayHttpHeaders.MediaType.APPLICATION_JSON_UTF8);
        headerMap.put(JlpayHttpHeaders.ACCEPT, JlpayHttpHeaders.MediaType.APPLICATION_JSON_UTF8);
        headerMap.put(JlpayHttpHeaders.V5.APP_ID, configModel.getAppId());

        String randomStr = NonceUtils.createNonce(JlpayHttpHeaders.V5.NONCE_LENGTH);
        String timestamp = String.valueOf(Instant.now().getEpochSecond());
        headerMap.put(JlpayHttpHeaders.V5.NONCE, randomStr);
        headerMap.put(JlpayHttpHeaders.V5.TIMESTAMP, timestamp);
        headerMap.put(JlpayHttpHeaders.V5.SIGN_ALG, JlpayHttpHeaders.V5.ALG_SM3_WITH_SM2_WITH_DER);

        // 设置实际请求的http请求体
        apiRequestModel.setHeadMap(headerMap);
        apiRequestModel.setContentType(JlpayHttpHeaders.MediaType.APPLICATION_JSON_UTF8);

        String requestBody;
        try {
            requestBody = INSTANCE.writeValueAsString(request);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException(e);
        }
        apiRequestModel.setRequestBody(requestBody);
        return apiRequestModel;
    }

    /**
     * 处理客户端信息
     *
     * @return
     */
    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientName("嘉联");
        clientInfoModel.setClientCode("jlpay-sdk");
        return clientInfoModel;
    }

    @Override
    protected JlBizResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        // 响应结果
        JSONObject resJson = JSONObject.parseObject(apiResponseModel.getResponseBody());
        return resJson.toJavaObject(requestContext.getIApiDefinition().getResponseClass());
    }

    @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
        HttpRequest postRequest = HttpRequest.builder()
                .url(HttpUtils.parseUrl(apiRequestModel.getApiURL()))
                .headers(apiRequestModel.getHeadMap())
                .body(apiRequestModel.getRequestBody())
                .method(HttpMethod.POST)
                .addHeader(JlpayHttpHeaders.V5.TIMESTAMP, String.valueOf(Instant.now().getEpochSecond()))
                .addHeader(JlpayHttpHeaders.V5.NONCE, NonceUtils.createNonce(JlpayHttpHeaders.V5.NONCE_LENGTH))
                .addHeader(JlpayHttpHeaders.V5.SIGN_ALG, JlpayHttpHeaders.V5.ALG_SM3_WITH_SM2_WITH_DER)
                .build();

        String sign = signVerifierManager.sign(postRequest);
        postRequest.addHeader(JlpayHttpHeaders.V5.SIGN, sign);
        HttpResponse httpResponse = sendPost(postRequest);
        // 封装请求结果
        ApiResponseModel responseModel = new ApiResponseModel();
        responseModel.setResponseBody(httpResponse.getBody());
        return responseModel;
    }

    private HttpResponse sendPost(HttpRequest httpRequest) {
        RequestBody requestBody = RequestBody.create(MediaType.parse("application/json"), httpRequest.getBody());
        Request request = new Request.Builder()
                .url(httpRequest.getUrl())
                .headers(Headers.of(httpRequest.getHeaders()))
                .post(requestBody)
                .build();

        OkHttpClient client2 = new OkHttpClient();
        try (Response response = client2.newCall(request).execute()) {
            if (!response.isSuccessful()) {
                throw new HttpStatusCodeException(response.code(), getBody(response));
            }
            return buildHttpResponse(response);
        } catch (IOException e) {
            throw new HttpExecutionException(String.format("Http Client execute failed, request: %s", httpRequest), e);
        }
    }

    private String getBody(Response response) throws IOException {
        return response.body() != null ? response.body().string() : null;
    }

    private HttpResponse buildHttpResponse(Response response) throws IOException {
        Map<String, String> headerMap = new HashMap<>();
        Headers headers = response.headers();
        for (String name : headers.names()) {
            headerMap.put(name, headers.get(name));
        }
        return HttpResponse.builder()
                .headers(headerMap)
                .body(getBody(response))
                .statusCode(response.code())
                .build();
    }
}