/**
 * fshows.com
 * Copyright (C) 2013-2024 All Rights Reserved
 */
package com.fshows.steward.handler;

import cn.hutool.core.codec.Base64;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.asymmetric.Sign;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.alibaba.fastjson.serializer.SimplePropertyPreFilter;
import com.fshows.fuiou.util.SignStrUtil;
import com.fshows.sdk.core.client.base.handler.IApiSignHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import com.fshows.sdk.core.util.SignerUtil;
import com.fshows.steward.annotation.NoSign;
import com.fshows.steward.apienum.FuStewardApiDefinitionEnum;
import com.fshows.steward.request.trade.FuStdSubAccountInTransferAsynReq;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.NotBlank;
import java.util.List;
import java.util.Map;

/**
 * @author liubo
 * @version FuStewardSignHandler.java, v 0.1 2024-01-07-1:32 下午 liubo
 */
@Slf4j
public class FuStewardSignHandler implements IApiSignHandler{

    @Override
    public String sign(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws FsApiException {
        String waitSignStr = null;
        try {
            // 默认配置
            DefaultClientConfigModel apiClientConfig = requestContext.getApiClientConfig();
            // 获取验签公钥
            String privateKey = apiClientConfig.getFubeiPrivateKey();
            // 获取待加签字符串
            Map<String, Object> data = apiRequestModel.getParamMap();
            // 获取待加签字符串
            waitSignStr = SignStrUtil.getWaitSignStr(data, requestContext.getNoNeedSignWords());

            if (FuStewardApiDefinitionEnum.SUB_ACCOUNT_IN_TRANSFER_ASYN.equals(requestContext.getIApiDefinition())) {
                FuStdSubAccountInTransferAsynReq req = (FuStdSubAccountInTransferAsynReq) apiRequestModel.getRequest();
                List<String> ignoreList = CollectionUtil.newArrayList("transTraceNo", "fee", "feeTraceNo", "feeType", "feeAccount", "srcProtocol", "transferType", "cardNo");
                SimplePropertyPreFilter filter = new SimplePropertyPreFilter();
                filter.getExcludes().addAll(ignoreList);
                waitSignStr = waitSignStr + "&accountInlist=" + JSONObject.toJSONString(req.getAccountInlist(), filter);
            }

            LogUtil.info(log, "{} >> 富友加签 >> waitSignStr={}",requestContext.getClientInfoModel().getClientDesc(), waitSignStr);
            // 创建加签对象
            Sign sign = new Sign(
                    apiClientConfig.getSignTypeEnum().getAlgorithm(),
                    SecureUtil.decode(privateKey),
                    null);
            // 执行加签操作
            byte[] signed = sign.sign(waitSignStr.getBytes(apiClientConfig.getCharset()));
            return Base64.encode(signed);
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求加签失败 >> apiRequestModel={}, waitSignStr={}", e, requestContext.getClientInfoModel().getClientDesc(), apiRequestModel, waitSignStr);
            throw new FsApiException("请求加签失败", e);
        }
    }

    @Override
    public Boolean verifySign(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws FsApiException {
        String waitSignStr = null;
        try {
            // 默认配置
            DefaultClientConfigModel apiClientConfig = requestContext.getApiClientConfig();
            // 获取验签公钥
            String publicKey = apiClientConfig.getPayCompanyPublicKey();
            // 获取待加签字符串
            Map<String, Object> data = apiResponseModel.getResponseMap();
            // 获取待加签字符串
            waitSignStr = SignStrUtil.getWaitSignStr(data,requestContext.getNoNeedSignWords());
            // 创建加签对象
            Sign sign = new Sign(
                    apiClientConfig.getSignTypeEnum().getAlgorithm(),
                    null,
                    SecureUtil.decode(publicKey));
            // 执行加签操作
            return sign.verify(waitSignStr.getBytes(apiClientConfig.getCharset()), Base64.decode(apiResponseModel.getResponseSign()));
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 响应验签失败 >> apiRequestModel={}, waitSignStr={}", e, requestContext.getClientInfoModel().getClientDesc(), apiResponseModel, waitSignStr);
            throw new FsApiException("请求验签失败", e);
        }
    }


}