/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows.steward;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.http.HttpRequest;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SimplePropertyPreFilter;
import com.fshows.fuiou.util.FuiouRequestUtils;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.handler.IApiSignHandler;
import com.fshows.sdk.core.client.base.handler.IHttpRequestHandler;
import com.fshows.sdk.core.client.base.handler.ISerializableHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.client.component.http.FromHttpRequestHandler;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import com.fshows.steward.apienum.FuStewardApiDefinitionEnum;
import com.fshows.steward.component.FuStewardSerializableHandler;
import com.fshows.steward.handler.FuStewardSignHandler;
import com.fshows.steward.request.FuStewardBaseRequest;
import com.fshows.steward.request.FuStewardBizRequest;
import com.fshows.steward.response.FuStewardBaseResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

/**
 * 嵌套参数客户端模版
 * @author liluqing
 * @version NestTemplateApiClinet.java, v 0.1 2023-12-12 15:40
 */
@Slf4j
public class FuStewardApiClinet extends AbstractApiClient<FuStewardBizRequest, FuStewardBaseResponse, FuStewardApiDefinitionEnum> {

     /**
      * 加签验签处理器
      */
    protected IApiSignHandler iApiSignHandler = new FuStewardSignHandler();

     /**
      * 序列化处理器
      */
     protected ISerializableHandler paramSerializable = new FuStewardSerializableHandler();

     /**
      * 请求执行器
      */
     protected IHttpRequestHandler httpRequestHandler = new FromHttpRequestHandler();

    /**
     * 日志打印时过滤普通字段
     */
    private SimplePropertyPreFilter LOG_FILTER = new SimplePropertyPreFilter();

    public FuStewardApiClinet(DefaultClientConfigModel apiClientConfig) throws FsApiException {
        super(apiClientConfig);
        // 初始化日志过滤器
        LOG_FILTER.getExcludes().addAll(CollectionUtil.newArrayList(new String[]{"legalImagF", "legalImagB", "busiLicPic"}));
    }

     /**
      * 构建请求参数
      *
      * @param request
      * @param context
      * @return
      */
    @Override
    protected ApiRequestModel buildApiRequestModel(FuStewardBizRequest request, DefaultRequestContext context) {
        // 1. 设置网关地址  2. 请求参数填充  3. 请求加签  4. 请求参数序列化
        // sdk客户端配置
        DefaultClientConfigModel clientConfig = context.getApiClientConfig();

        ApiRequestModel apiRequestModel = new ApiRequestModel();
        // 设置网关地址
        apiRequestModel.setApiURL(context.getApiClientConfig().getApiParentURL() + context.getIApiDefinition().getApiURI());

        // 补充业务参数入参
        request.setMchntCd(clientConfig.getAppId());
        // 设置加签过滤字符串
        context.setNoNeedSignWords(FuiouRequestUtils.getNoNeedSign(request.getClass()));
        // 请求加签处理
        //Map<String, Object> stringObjectMap = toMapObj(request, apiClientConfig.isHump(),false);
        Map<String, Object> stringObjectMap = FuiouRequestUtils.ObjectToMap(request);

        apiRequestModel.setParamMap(new TreeMap<>(stringObjectMap));
        apiRequestModel.setRequest(request);
        // 获取签名
        apiRequestModel.setRequestSign(iApiSignHandler.sign(apiRequestModel, context));
        // 设置签名
        Map<String, Object> paramMap = apiRequestModel.getParamMap();
        paramMap.put("signature", apiRequestModel.getRequestSign());

        // 补充参数构建实际的请求入参
        FuStewardBaseRequest fuStewardBaseRequest = new FuStewardBaseRequest();
        fuStewardBaseRequest.setMchntCd(clientConfig.getAppId());
        fuStewardBaseRequest.setSignature(apiRequestModel.getRequestSign());
        fuStewardBaseRequest.setData(request);
        apiRequestModel.setRequest(fuStewardBaseRequest);

        // 参数序列化
        String requestBody = paramSerializable.serializeObject(apiRequestModel, context);
        apiRequestModel.setRequestBody(requestBody);

        // 富友银行账户信息查询有时会返回gzip压缩的响应结果，导致数据解析异常，所以指定请求不进行任何编码
        // 指定响应结果不要压缩
        Map<String, String> headMap = new HashMap<>();
        headMap.put("Accept-Encoding", "identity");
        apiRequestModel.setHeadMap(headMap);
        return apiRequestModel;
    }

     /**
      * 处理客户端信息
      *
      * @return
      */
    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientCode("fuiou-sdk");
        clientInfoModel.setClientName("富友分账");
        return clientInfoModel;
    }

     @Override
     protected FuStewardBaseResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
         // 1.  2. 请求参数填充  3. 请求加签  4. 参数反序列化
         // 客户端配置
         DefaultClientConfigModel apiClientConfigModel = requestContext.getApiClientConfig();
         apiResponseModel.setResponse(paramSerializable.deserializationResponse(apiResponseModel, apiRequestModel, requestContext));
         // 获取加签需过滤字段
         requestContext.setNoNeedSignWords(FuiouRequestUtils.getNoNeedSign(requestContext.getIApiDefinition().getResponseClass()));
         // 打印请求的明文日志
         printRequestLog(apiResponseModel, apiRequestModel, requestContext);
         // 如果富友返回了签名才验签
         if (apiClientConfigModel.isVrifySignResponse() && StringUtils.isNotBlank(apiResponseModel.getResponseSign())) {
             // 是否验签
             Boolean verifySign = iApiSignHandler.verifySign(apiResponseModel, apiRequestModel,  requestContext);
             if (apiClientConfigModel.isVrifySignResponse() && !Boolean.TRUE.equals(verifySign)) {
                 throw new FsApiException("验签失败");
             }
         }
         return (FuStewardBaseResponse) apiResponseModel.getResponse();
     }

    /**
     * 打印请求的明文日志
     *
     * @param apiResponseModel
     * @param apiRequestModel
     * @param requestContext
     */
     private void printRequestLog(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
         SimplePropertyPreFilter filter = new SimplePropertyPreFilter();
         filter.getExcludes ().add ("name");
         LogUtil.info(log, "{} >> 请求结束【明文】 >> url={}, method={}, request={}, response={}", new Object[]{
                 requestContext.getClientInfoModel().getClientDesc(),
                 apiRequestModel.getApiURL(),
                 requestContext.getIApiDefinition(),
                 JSONObject.toJSONString(apiRequestModel.getParamMap(), LOG_FILTER),
                 JSONObject.toJSONString(apiResponseModel.getResponse(), LOG_FILTER)
         });
     }

     @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
        // 该接口比较特殊，有时会返回乱码，故特殊处理
        if (FuStewardApiDefinitionEnum.QUERY_BOOK_TRADE.equals(requestContext.getIApiDefinition())) {
            DefaultClientConfigModel defaultClientConfigModel = requestContext.getApiClientConfig();
            String result2 = HttpRequest.post(apiRequestModel.getApiURL())
                    .form("req", apiRequestModel.getRequestForm().get("req"))//表单内容
                    .timeout(defaultClientConfigModel.getReadTimeout())//超时，毫秒
                    .setConnectionTimeout(defaultClientConfigModel.getConnectionTimeout())
                    .execute().body();
            ApiResponseModel apiResponseModel = new ApiResponseModel();
            apiResponseModel.setResponseBody(result2);
            return apiResponseModel;
        }
        return httpRequestHandler.httpRequest(apiRequestModel, requestContext);
    }

    @Override
    public FuStewardBaseResponse execute(FuStewardBizRequest request, FuStewardApiDefinitionEnum apiDefinition) throws FsApiException {
        return super.doExecute(request, apiDefinition);
    }

    @Override
    public FuStewardBaseResponse execute(FuStewardBizRequest request, FuStewardApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return super.doExecute(request, apiDefinition, configModel);
    }

}