/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.fuiou.client.impl;

import cn.hutool.core.codec.Base64;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.asymmetric.Sign;
import com.fshows.fuiou.client.base.ApiClientConfig;
import com.fshows.fuiou.client.base.ISigner;
import com.fshows.fuiou.client.base.SignParam;
import com.fshows.fuiou.client.base.VerifySignParam;
import com.fshows.fuiou.exception.FuiouApiException;
import com.fshows.fuiou.request.base.FuiouBizRequest;
import com.fshows.fuiou.util.FuiouRequestUtils;
import com.fshows.fuiou.util.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.Map;

/**
 * 富结算接口友加签实现
 *
 * @author liluqing
 * @version SettlementSignerImpl.java, v 0.1 2022-03-02 18:32
 */
@Slf4j
public class SettlementSignerImpl implements ISigner {

    @Override
    public String sign(SignParam signParam, ApiClientConfig apiClientConfig) throws FuiouApiException {
        return doSign(signParam, apiClientConfig);
    }

    @Override
    public Boolean verifySign(Map<String, String> resMap, VerifySignParam verifySignParam, ApiClientConfig apiClientConfig) throws FuiouApiException {
        try {
            // 获取待加签字符串
            String waitSignStr = getWaitSignStr(resMap);
            LogUtil.info(log, "【fuiou-sdk】待验签字符串 >> waitSignStr={}", waitSignStr);
            // 响应签名
            String resSign = resMap.get("sign");
            // 创建加签对象
            Sign sign = new Sign(
                    apiClientConfig.getSignTypeEnum().getAlgorithm(),
                    SecureUtil.decode(apiClientConfig.getFubeiPrivateKey()),
                    SecureUtil.decode(apiClientConfig.getFuiouPublicKey()));

            // 执行加签操作
            return sign.verify(waitSignStr.getBytes(apiClientConfig.getCharset()), Base64.decode(resSign));
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk】fuiou响应结果验签失败 >> signParam={}", e, verifySignParam);
            throw new FuiouApiException("fuiou响应结果验签失败", e);
        }
    }

    /**
     * 执行方法加签
     *
     * @param signParam
     * @param apiClientConfig
     * @return
     * @throws FuiouApiException
     */
    public String doSign(SignParam signParam, ApiClientConfig apiClientConfig) throws FuiouApiException {
        try {
            FuiouBizRequest request = signParam.getRequest();
            // 获取待加签字符串
            Map<String, String> data = FuiouRequestUtils.toMap(request);
            // 获取待加签字符串
            String waitSignStr = getWaitSignStr(data);
            LogUtil.info(log, "【fuiou-sdk】待加签字符串 >> waitSignStr={}", waitSignStr);
            // 创建加签对象
            Sign sign = new Sign(
                    apiClientConfig.getSignTypeEnum().getAlgorithm(),
                    SecureUtil.decode(apiClientConfig.getFubeiPrivateKey()),
                    null);
            // 执行加签操作
            byte[] signed = sign.sign(waitSignStr.getBytes(apiClientConfig.getCharset()));
            return Base64.encode(signed);
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk】fuiou请求加签失败 >> signParam={}", e, signParam);
            throw new FuiouApiException("请求加签失败", e);
        }
    }

    /**
     * 获取待加签字符串，格式：addn_inf=&curr_type=&goods_des=卡盟测试&goods_detail=asasda&goods_tag=&ins_cd=08A9999999&
     *
     * @return
     */
    public static String getWaitSignStr(Map<String, String> dataMap) {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, String> entry : dataMap.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            if (StringUtils.equals(key, "sign")
                    || StringUtils.startsWith(key, "reserved")
                    || StringUtils.startsWith(key, "version")) {
                continue;
            }
            if (sb.length() > 0) {
                sb.append("&");
            }
            sb.append(key).append("=").append(StringUtils.defaultString(value, StringUtils.EMPTY));
        }
        return sb.toString();
    }
}