/**
 * fshows.com
 * Copyright (C) 2013-2024 All Rights Reserved.
 */
package com.fshows.fsframework.extend.aliyun.oss;

import cn.hutool.core.codec.Base64;
import cn.hutool.core.util.HexUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.digest.DigestUtil;
import cn.hutool.crypto.symmetric.AES;
import com.aliyun.oss.ClientConfiguration;
import com.aliyun.oss.HttpMethod;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.common.comm.Protocol;
import com.aliyun.oss.model.*;
import com.ctrip.framework.apollo.enums.PropertyChangeType;
import com.ctrip.framework.apollo.model.ConfigChange;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfigChangeListener;
import com.fshows.fsframework.common.exception.CommonException;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.extend.util.FsAESUtil;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author wangqilei
 * @version OssLocalClient.java, v 0.1 2024-12-10 3:26 PM wangqilei
 */
@Slf4j
public class FsOssClient implements ApplicationContextAware {

    /**
     * spring上下文
     */
    private ApplicationContext applicationContext;

    /**
     * OSS客户端配置
     */
    @Setter
    private FsOssConfig fsOssConfig;

    /**
     * 路径分隔符
     */
    private static final String BACKSLASH = "/";

    /**
     * 解密后的aksk
     */
    private String decryptAccess;
    private String decryptSecret;

    private volatile OSSClient outOssClient;

    private volatile OSSClient innerOssClient;

    private final ScheduledExecutorService executorService = Executors.newScheduledThreadPool(1);

    @PostConstruct
    public void init() {
        try {
            // 如果外部没有传入配置,则初始化配置
            if (fsOssConfig ==  null) {
                fsOssConfig = new FsOssConfig();
            }
            // 初始化上下文
            fsOssConfig.setApplicationContext(applicationContext);

            decryptAccess = decryptAkAndSk(fsOssConfig.getEncryAccessKey(), fsOssConfig.getAesPassword());
            decryptSecret = decryptAkAndSk(fsOssConfig.getEncrySecretKey(), fsOssConfig.getAesPassword());

            ClientConfiguration config = buildOssConfig();
            if (outOssClient == null) {
                outOssClient = new OSSClient(fsOssConfig.getOssEndPointOut(), decryptAccess, decryptSecret, config);
            }
            if (innerOssClient == null) {
                innerOssClient = new OSSClient(fsOssConfig.getOssEndPointInner(), decryptAccess, decryptSecret, config);
            }

            LogUtil.info(log, "FsOssClient初始化完成");
        } catch (Exception e) {
            LogUtil.error(log, "FsOssClient初始化异常", e);
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient客户端初始化异常，请检查AKSK是否配置正确");
        }
    }

    /**
     * 监听阿波罗OSS配置更新
     * 1.每次变更都会创建新的外网和内网的OSS客户端
     * 2.创建新客户端完成，30秒后关闭旧客户端
     */
    @ApolloConfigChangeListener
    public synchronized void refreshOssClient(ConfigChangeEvent changeEvent) {
        if (!fsOssConfig.getOssClientDynamicUpdate()) {
            LogUtil.info(log, "refreshOssClient >> 阿波罗配置更新，但当前未开启动态更新");
            return;
        }
        ConfigChange accessConfigChange = changeEvent.getChange(fsOssConfig.getApolloKeyForEncryAccessKey());
        ConfigChange secretConfigChange = changeEvent.getChange(fsOssConfig.getApolloKeyForEncrySecretKey());
        LogUtil.info(log, "refreshOssClient >> 接收到AKSK变换通知 >> accessConfigChange={}, secretConfigChange={}", accessConfigChange, secretConfigChange);

        String refreshEncryAccessKey = fsOssConfig.getEncryAccessKey();
        String refreshEncrySecretKey = fsOssConfig.getEncrySecretKey();

        // 未检测到AKSK变化,则直接返回
        if (accessConfigChange == null && secretConfigChange == null) {
            LogUtil.info(log, "refreshOssClient >> 阿波罗配置更新，但未检测到AKSK变化,本次不做处理。");
            return;
        }

        if (accessConfigChange != null
                && !accessConfigChange.getChangeType().equals(PropertyChangeType.DELETED)
                && !accessConfigChange.getOldValue().equals(accessConfigChange.getNewValue())) {
            refreshEncryAccessKey = accessConfigChange.getNewValue();
        }
        if (secretConfigChange != null
                && !secretConfigChange.getChangeType().equals(PropertyChangeType.DELETED)
                && !secretConfigChange.getOldValue().equals(secretConfigChange.getNewValue())) {
            refreshEncrySecretKey = secretConfigChange.getNewValue();
        }
        // 新客户端是否创建成功
        boolean newSuccess = false;
        // 临时客户端，用于关闭，释放资源
        final OSSClient tmpOutOssClient = outOssClient;
        final OSSClient tmpInnerOssClient = innerOssClient;
        try {
            // 对key解密
            decryptAccess = decryptAkAndSk(refreshEncryAccessKey, fsOssConfig.getAesPassword());
            decryptSecret = decryptAkAndSk(refreshEncrySecretKey, fsOssConfig.getAesPassword());

            // 创建新客户端
            ClientConfiguration config = buildOssConfig();
            outOssClient = new OSSClient(fsOssConfig.getOssEndPointOut(), decryptAccess, decryptSecret, config);
            innerOssClient = new OSSClient(fsOssConfig.getOssEndPointInner(), decryptAccess, decryptSecret, config);

            LogUtil.info(log, "refreshOssClient >> OSS动态替换aksk成功，新OSS客户端创建完成");
            newSuccess = true;
        } catch (Exception e) {
            LogUtil.error(log, "refreshOssClient >> OSS动态替换aksk异常", e);
        } finally {
            if (newSuccess) {
                // 可能存在任务未执行完，延迟关闭旧客户端，默认30秒
                executorService.schedule(() -> {
                    // 当新客户端已经创建完成，才能关闭旧客户端
                    if (!Objects.equals(tmpOutOssClient, outOssClient)) {
                        tmpOutOssClient.shutdown();
                        LogUtil.info(log, "refreshOssClient >> 旧客户端(外网)关闭完成");
                    }
                    if (!Objects.equals(tmpInnerOssClient, innerOssClient)) {
                        tmpInnerOssClient.shutdown();
                        LogUtil.info(log, "refreshOssClient >> 旧客户端(内网)关闭完成");
                    }
                }, fsOssConfig.getCloseDelayTime(), TimeUnit.SECONDS);
            }
        }
    }

    /**
     * 获取正式的AK或者sk
     *
     * @param ciphertext
     * @param pwd
     * @return
     */
    public String decryptAkAndSk(String ciphertext, String pwd) {
        // 小于32位的，说明是明文，则直接返回
        if (StringUtils.length(ciphertext) < 32) {
            return ciphertext;
        }
        return FsAESUtil.decryptKey(ciphertext, pwd);
    }

    private ClientConfiguration buildOssConfig() {
        // 创建ClientConfiguration。ClientConfiguration是OSSClient的配置类，可配置代理、连接超时、最大连接数等参数。
        ClientConfiguration conf = new ClientConfiguration();
        // 设置OSSClient允许打开的最大HTTP连接数，默认为1024个。
        conf.setMaxConnections(200);
        // 设置Socket层传输数据的超时时间，默认为50000毫秒。
        conf.setSocketTimeout(10000);
        // 设置建立连接的超时时间，默认为50000毫秒。
        conf.setConnectionTimeout(10000);
        // 设置从连接池中获取连接的超时时间（单位：毫秒），默认不超时。
        conf.setConnectionRequestTimeout(1000);
        // 设置连接空闲超时时间。超时则关闭连接，默认为60000毫秒。
        conf.setIdleConnectionTime(10000);
        // 设置失败请求重试次数，默认为3次。
        conf.setMaxErrorRetry(3);
        // 设置是否支持将自定义域名作为Endpoint，默认支持。
        conf.setSupportCname(true);
        // 设置是否开启二级域名的访问方式，默认不开启。
        conf.setSLDEnabled(false);
        // 设置连接OSS所使用的协议（HTTP/HTTPS），默认为HTTP。
        conf.setProtocol(Protocol.HTTP);
        return conf;
    }

    /**
     * 上传文件（可选择内外网环境）
     *
     * @param bucketName  bucket空间
     * @param key         存放OSS上的文件全路径，包含文件名
     * @param inputStream 文件流
     * @param outOssFlag  是否使用外网
     */
    public PutObjectResult uploadFileWithOutFlag(String bucketName, String key, InputStream inputStream, boolean outOssFlag) {
        if (StringUtils.isEmpty(bucketName)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient bucketName is null");
        }
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }
        if (outOssFlag) {
            return outOssClient.putObject(bucketName, key, inputStream);
        } else {
            return innerOssClient.putObject(bucketName, key, inputStream);
        }
    }

    /**
     * 上传文件（可选择内外网环境）
     *
     * @param bucketName bucket空间
     * @param key        存放OSS上的文件全路径，包含文件名
     * @param file       文件对象
     * @param outOssFlag 是否使用外网
     */
    public PutObjectResult uploadFileWithOutFlag(String bucketName, String key, File file, boolean outOssFlag) {
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }
        if (outOssFlag) {
            return outOssClient.putObject(bucketName, key, file);
        } else {
            return innerOssClient.putObject(bucketName, key, file);
        }
    }

    /**
     * 上传文件（可选择内外网环境）
     *
     * @param bucketName
     * @param key
     * @param filePath
     * @param outOssFlag
     */
    public PutObjectResult uploadFileWithOutFlag(String bucketName, String key, String filePath, boolean outOssFlag) {
        return uploadFileWithOutFlag(bucketName, key, new File(filePath), outOssFlag);
    }

    /**
     * 上传文件
     *
     * @param bucketName bucket空间
     * @param key        存放OSS上的文件全路径，包含文件名
     * @param filePath   文件路径
     */
    public PutObjectResult uploadFile(String bucketName, String key, String filePath) {
        return uploadFileWithOutFlag(bucketName, key, new File(filePath), fsOssConfig.getOssOutClientFlag());
    }

    /**
     * 上传文件
     *
     * @param bucketName bucket空间
     * @param key        存放OSS上的文件全路径，包含文件名
     * @param file       文件对象
     */
    public PutObjectResult uploadFile(String bucketName, String key, File file) {
        return uploadFileWithOutFlag(bucketName, key, file, fsOssConfig.getOssOutClientFlag());
    }

    /**
     * 上传文件
     *
     * @param bucketName  bucket空间
     * @param key         存放OSS上的文件全路径，包含文件名
     * @param inputStream 文件流
     */
    public PutObjectResult uploadFile(String bucketName, String key, InputStream inputStream) {
        return uploadFileWithOutFlag(bucketName, key, inputStream, fsOssConfig.getOssOutClientFlag());
    }

    /**
     * 上传文件
     *
     * @param bucketName bucket空间
     * @param key        存放OSS上的文件全路径，包含文件名
     * @param filePath   文件路径
     * @param metadata   object的元信息ObjectMetadata，若该元信息未包含Content-Length，则采用chunked编码传输请求数据
     */
    public PutObjectResult uploadFile(String bucketName, String key, String filePath, ObjectMetadata metadata) {
        if (StringUtils.isEmpty(bucketName)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient bucketName is null");
        }
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }
        File file = new File(filePath);
        return innerOssClient.putObject(bucketName, key, file, metadata);
    }

    /**
     * 下载文件（可选择内外网环境）
     *
     * @param bucketName bucket空间
     * @param key        存放OSS上的文件全路径，包含文件名
     * @param file       文件对象
     * @param outOssFlag 是否使用外网
     */
    public void downloadFileWithOutFlag(String bucketName, String key, File file, boolean outOssFlag) {
        if (StringUtils.isEmpty(bucketName)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient bucketName is null");
        }
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }
        if (outOssFlag) {
            outOssClient.getObject(new GetObjectRequest(bucketName, key), file);
        } else {
            innerOssClient.getObject(new GetObjectRequest(bucketName, key), file);
        }
    }

    /**
     * 下载文件（可选择内外网环境）
     *
     * @param bucketName bucket空间
     * @param key        存放OSS上的文件全路径，包含文件名
     * @param filePath   文件路径
     * @param outOssFlag 是否使用外网
     */
    public void downloadFileWithOutFlag(String bucketName, String key, String filePath, boolean outOssFlag) {
        this.downloadFileWithOutFlag(bucketName, key, new File(filePath), outOssFlag);
    }

    /**
     * 下载文件
     *
     * @param bucketName bucket空间
     * @param key        存放OSS上的文件全路径，包含文件名
     * @param filePath   文件路径
     */
    public void downloadFile(String bucketName, String key, String filePath) {
        this.downloadFileWithOutFlag(bucketName, key, new File(filePath), fsOssConfig.getOssOutClientFlag());
    }

    /**
     * 下载文件
     *
     * @param bucketName bucket空间
     * @param key        存放OSS上的文件全路径，包含文件名
     * @param file       文件对象
     */
    public void downloadFile(String bucketName, String key, File file) {
        this.downloadFileWithOutFlag(bucketName, key, file, fsOssConfig.getOssOutClientFlag());
    }

    /**
     * 生成下载url
     *
     * @param bucketName bucket空间
     * @param key        存放OSS上的文件全路径，包含文件名
     * @param expireTime 过期时间，单位秒，默认1小时
     */
    public String generateFileUrlWithExpireTime(String bucketName, String key, Long expireTime) {
        if (StringUtils.isEmpty(bucketName)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient bucketName is null");
        }
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }

        Date time = new Date(System.currentTimeMillis() + (Objects.isNull(expireTime) ? fsOssConfig.getUrlExpireTime() : expireTime) * 1000);
        return outOssClient.generatePresignedUrl(bucketName, key, time).toString();
    }

    /**
     * 生成下载url（默认1小时过期）
     *
     * @param bucketName bucket空间
     * @param key        存放OSS上的文件全路径，包含文件名
     */
    public String generateFileUrl(String bucketName, String key) {
        return generateFileUrlWithExpireTime(bucketName, key, null);
    }

    /**
     * 生成链接地址
     *
     * @param request
     * @return
     */
    public URL generatePresignedUrl(GeneratePresignedUrlRequest request) {
        return outOssClient.generatePresignedUrl(request);
    }

    /**
     * 删除文件
     *
     * @param bucketName
     * @param key
     */
    public void deleteFile(String bucketName, String key) {
        this.deleteFileWithOutOssFlag(bucketName, key, fsOssConfig.getOssOutClientFlag());
    }

    /**
     * 删除文件（可选择内外网环境）
     *
     * @param bucketName
     * @param key
     * @param outOssFlag
     */
    public void deleteFileWithOutOssFlag(String bucketName, String key, boolean outOssFlag) {
        if (StringUtils.isEmpty(bucketName)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient bucketName is null");
        }
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }
        if (outOssFlag) {
            outOssClient.deleteObject(bucketName, key);
        } else {
            innerOssClient.deleteObject(bucketName, key);
        }
    }

    /**
     * 外网：生成Post请求的policy表单域。
     *
     * @param expiration policy过期时间。
     * @param conds      policy条件列表。
     * @return policy字符串。
     */
    public String generatePostPolicy(Date expiration, PolicyConditions conds) {
        return outOssClient.generatePostPolicy(expiration, conds);
    }

    /**
     * 外网：根据Access Key Secret和policy计算签名，OSS依据该签名验证Post请求的合法性
     *
     * @param postPolicy generatePostPolicy生成的
     * @return post签名
     */
    public String calculatePostSignature(String postPolicy) {
        return outOssClient.calculatePostSignature(postPolicy);
    }

    /**
     * 获取OSS的accessKey
     */
    public String getAccessKey() {
        return decryptAccess;
    }

    /**
     * 获取外网链接地址.
     *
     * @return the string
     */
    public String getOssEndPointOut() {
        return fsOssConfig.getOssEndPointOut();
    }

    /**
     * 获取内网链接地址.
     *
     * @return the string
     */
    public String getOssEndPointInner() {
        return fsOssConfig.getOssEndPointInner();
    }

    public static void main(String[] args) {
        // 要加密的字符串
//        String before = new String(SecureUtil.generateKey(SymmetricAlgorithm.AES.getValue()).getEncoded(), StandardCharsets.UTF_8);
//        String inputString = Base64.encode(before);
//        System.out.println("password base64: " + inputString);
//
//        System.out.println("password: " + new String(HexUtil.encodeHex(Base64.decode(inputString))));

        // 进行 MD5 加密
        byte[] md5SecretKey = DigestUtil.md5(DigestUtil.md5("406c42688e7f7db4a8f3912772725a5d".getBytes(StandardCharsets.UTF_8)));
        System.out.println("password md5: " + new String(HexUtil.encodeHex(md5SecretKey)));

        AES aes = SecureUtil.aes(md5SecretKey);

        String accessKey = "LTAI5tHpkPpUwR9Pu3xHFUJV";
        String secretKey = "dOj8EqSma1moEQqmKdJb7sb0Pes8OI";

        String encryAccessKey = Base64.encode(aes.encrypt(accessKey));
        String encrySecretKey = Base64.encode(aes.encrypt(secretKey));
        System.out.println("oss accessKey encode: " + new String(HexUtil.encodeHex(Base64.decode(encryAccessKey))));
        System.out.println("oss accessKey encode: " + new String(HexUtil.encodeHex(Base64.decode(encrySecretKey))));

        String decryptAccessKey = new String(aes.decrypt(Base64.decode(encryAccessKey)));
        String decryptSecretKey = new String(aes.decrypt(Base64.decode(encrySecretKey)));

        System.out.println("oss accessKey decode: " + decryptAccessKey);
        System.out.println("oss secretKey decode: " + decryptSecretKey);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext=applicationContext;
    }
}