/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package
        com.fshows.fsframework.extend.lock;

import com.fshows.fsframework.core.constants.StringPool;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.extend.lock.exception.ZookeeperLockException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.locks.InterProcessMutex;

import java.text.MessageFormat;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * 基于zookeeper的分布式锁服务
 *
 * @author Liluqing
 * @version ZookeeperDistributedLock.java, v 0.1 2018-10-29 15:53
 */
@Slf4j
public class ZookeeperDistributedLock implements DistributedLock {
    /**
     * 锁缓存
     */
    private Map<String, InterProcessMutex> lockMap = new ConcurrentHashMap<>(16);

    /**
     * zookeeper客户端
     */
    private CuratorFramework curatorFramework;

    /**
     * 总计父锁目录
     */
    private String lockBasePath;

    public ZookeeperDistributedLock(CuratorFramework curatorFramework, String lockBasePath) {
        this.curatorFramework = curatorFramework;
        this.lockBasePath = createLockBasePath(lockBasePath);
    }

    /**
     * 获取锁，阻塞
     *
     * @param lockKey 锁的key
     */
    @Override
    public void lock(String lockKey) {
        String lockPath = getLockPath(lockKey);
        long currentTime = System.currentTimeMillis();
        try {
            InterProcessMutex lock = getInterProcessMutex(lockPath);
            lock.acquire();
        } catch (Exception e) {
            LogUtil.error(log, "【分布式锁服务】lock-获取分布式锁失败！lockPath = {}", e, lockPath);
            throw new ZookeeperLockException(MessageFormat.format("【分布式锁服务】lock-获取分布式锁失败！lockPath = {0}",lockPath),e);
        }
        LogUtil.info(log, "【分布式锁服务】lock-获取到分布式锁成功！ 耗时 = {} ，lockPath = {}", System.currentTimeMillis() - currentTime,lockPath);
    }

    /**
     * 尝试获取锁，并立即返回结果
     *
     * @param lockKey 锁的key
     * @return 是否获取到锁
     */
    @Override
    public boolean tryLock(String lockKey) {
        return tryLock(lockKey, 0);
    }

    /**
     * 在timeout时限内获取尝试获取锁
     *
     * @param lockKey 锁的key
     * @param timeout 获取锁的超时时间,单位毫秒
     * @return
     * @throws InterruptedException
     */
    @Override
    public boolean tryLock(String lockKey, long timeout)  {
        String lockPath = getLockPath(lockKey);
        long currentTime = System.currentTimeMillis();
        try {
            InterProcessMutex lock = getInterProcessMutex(lockPath);
            lock.acquire(timeout, TimeUnit.MILLISECONDS);
            LogUtil.info(log, "【分布式锁服务】tryLock-获取到分布式锁成功！ 耗时 = {} ，lockPath = {}", System.currentTimeMillis() - currentTime,lockPath);
        } catch (Exception e) {
            LogUtil.error(log, "【分布式锁服务】tryLock-获取分布式锁失败！lockPath = {}", e, lockPath);
            throw new ZookeeperLockException(MessageFormat.format("【分布式锁服务】tryLock-获取分布式锁失败！lockPath = {0}",lockPath),e);
        }
        return true;
    }

    /**
     * 解锁
     *
     * @param lockKey 锁的key
     */
    @Override
    public void unlock(String lockKey)  {
        String lockPath = getLockPath(lockKey);
        try {
            InterProcessMutex lock = getInterProcessMutex(lockPath);
            lock.release();
        } catch (Exception e) {
            LogUtil.error(log, "【分布式锁服务】unlock-分布式锁解锁失败！lockPath = {}", e, lockPath);
            throw new ZookeeperLockException(MessageFormat.format("unlock error, key={0}",lockKey),e);
        }
    }

    /**
     * 获取curator可重入锁对象
     *
     * @param lockKey
     * @return
     */
    private InterProcessMutex getInterProcessMutex(String lockKey) {
        InterProcessMutex interProcessMutex = lockMap.get(lockKey);
        if (interProcessMutex == null) {
            synchronized (lockMap) {
                interProcessMutex = new InterProcessMutex(curatorFramework, lockKey);
            }
        }
        return interProcessMutex;
    }

    /**
     * 获取zk的锁路径
     *
     * @return
     */
    private String getLockPath(String lockKey) {
        if (StringUtils.isBlank(lockKey)) {
            throw new ZookeeperLockException("[zk分布式锁服务]lockKey不能为空！");
        }
        if (lockKey.startsWith(StringPool.SLASH)) {
            lockKey = lockKey.substring(0, lockKey.length() - 1);
        }
        return lockBasePath + lockKey;
    }

    /**
     * 生成锁的总父路径
     *
     * @param lockBasePath
     * @return
     */
    private String createLockBasePath(String lockBasePath){
        if (lockBasePath == null) {
            return StringPool.EMPTY;
        }
        if (!lockBasePath.endsWith(StringPool.SLASH)) {
            lockBasePath = lockBasePath + StringPool.SLASH;
        }
        return lockBasePath;
    }
}