/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package
        com.fshows.fsframework.extend.transaction;

import com.fshows.fsframework.common.exception.BaseException;
import com.fshows.fsframework.core.utils.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import java.util.function.Supplier;

/**
 * 默认事务管理器
 *
 * @author Liluqing
 * @version DefaultTransactionManager.java, v 0.1 2018-10-22 14:10
 */
@Slf4j
public class DefaultFsTransactionManager implements FsTransactionManager {

    private DataSourceTransactionManager transactionManager;

    @Override
    public <T> T execute(Supplier<T> serviceHandle, ServiceRollback serviceRollback) {
        T result;
        TransactionStatus transaction = null;
        try {
            transaction = transactionManager.getTransaction(new DefaultTransactionDefinition());
            // 业务处理
            result = serviceHandle.get();
            transactionManager.commit(transaction);
        } catch (BaseException e) {
            serviceRollback(serviceRollback, transaction, e);
            LogUtil.warn(log, "业务异常-事务发生回滚", e);
            throw e;
        } catch (Exception e) {
            serviceRollback(serviceRollback, transaction, e);
            LogUtil.error(log, "系统未知错误-事务发生回滚", e);
            throw e;
        }
        return result;
    }

    @Override
    public void executeNoReturn(ServiceHandle serviceHandle, ServiceRollback serviceRollback) {
        TransactionStatus transaction = null;
        try {
            transaction = transactionManager.getTransaction(new DefaultTransactionDefinition());
            // 业务处理
            serviceHandle.handle();
            transactionManager.commit(transaction);
        } catch (BaseException e) {
            serviceRollback(serviceRollback, transaction, e);
            LogUtil.warn(log, "业务异常-事务发生回滚", e);
            throw e;
        } catch (Exception e) {
            serviceRollback(serviceRollback, transaction, e);
            LogUtil.error(log, "系统位置错误-事务发生回滚", e);
            throw e;
        }
    }


    /**
     * 执行业务回滚方法
     *
     * @param serviceRollback
     * @param transaction
     */
    private void serviceRollback(ServiceRollback serviceRollback, TransactionStatus transaction, Exception ex) {
        if (transaction == null) {
            return;
        }
        // 是否回滚事务
        boolean isRollback = true;
        try {
            isRollback = serviceRollback.noException(ex);
        } catch (Exception e) {
            LogUtil.error(log, "业务回滚逻辑异常-事务发生回滚", e);
        } finally {
            if (isRollback) {
                transactionManager.rollback(transaction);
            } else {
                transactionManager.commit(transaction);
            }
        }
    }

    /**
     * Setter method for property <tt>transactionManager</tt>.
     *
     * @param transactionManager value to be assigned to property transactionManager
     */
    public void setTransactionManager(DataSourceTransactionManager transactionManager) {
        this.transactionManager = transactionManager;
    }
}