/**
 * fshows.com
 * Copyright (C) 2013-2025 All Rights Reserved.
 */
package com.fshows.fsframework.extend.aliyun.mq.decorator;

import com.aliyun.openservices.ons.api.Consumer;
import com.aliyun.openservices.ons.api.ExpressionType;
import com.aliyun.openservices.ons.api.MessageListener;
import com.aliyun.openservices.ons.api.MessageSelector;
import com.aliyun.openservices.ons.api.ONSFactory;
import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.aliyun.openservices.ons.api.bean.Subscription;
import com.aliyun.openservices.ons.api.bean.SubscriptionExt;
import com.aliyun.openservices.ons.api.exception.ONSClientException;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.extend.aliyun.mq.config.FsMqConfig;
import com.fshows.fsframework.extend.aliyun.mq.core.FsMqInstanceManager;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PreDestroy;
import java.lang.reflect.Method;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

/**
 * 增强的Consumer装饰器
 * 在原有ConsumerBean基础上增加Apollo配置变更时的随机延迟重启功能
 *
 * @author liluqing
 * @version FsConsumerBean.java, v 0.1 2025-01-10 15:30
 */
@Slf4j
public class FsConsumerBean implements Consumer {

    /**
     * 需要注入该字段，指定构造{@code Consumer}实例的属性，具体支持的属性详见{@link PropertyKeyConst}
     */
    private Properties properties;

    /**
     * 通过注入该字段，在启动{@code Consumer}时完成Topic的订阅
     */
    private Map<Subscription, MessageListener> subscriptionTable;

    /**
     * 当前活跃的Consumer实例
     */
    private Consumer consumer;

    /**
     * 实例唯一标识
     */
    private String instanceKey;

    /**
     * MQ实例管理器
     */
    @Autowired
    private FsMqInstanceManager instanceManager;

    /**
     * MQ配置
     */
    @Autowired
    private FsMqConfig fsMqConfig;

    /**
     * 启动该{@code Consumer}实例，建议配置为Bean的init-method
     */
    @Override
    public void start() {
        if (null == this.instanceManager) {
            throw new ONSClientException("当前 FsConsumerBean 对象，未通过 @Bean 注解注册到Spring容器中, 启动失败");
        }
        if (null == this.fsMqConfig) {
            throw new ONSClientException("当前 FsConsumerBean 对象，未通过 @Bean 注解注册到Spring容器中, 启动失败");
        }
        if (null == this.properties) {
            throw new ONSClientException("properties not set");
        }

        if (null == this.subscriptionTable) {
            throw new ONSClientException("subscriptionTable not set");
        }

        // 校验AK/SK不能由外部设置
        validateCredentials();

        // 更新Properties中的AK/SK（如果配置了动态更新）
        Properties updatedProperties = updatePropertiesWithDecryptedKeys();

        this.consumer = ONSFactory.createConsumer(updatedProperties);

        // 订阅Topic
        subscribeTopics();

        this.consumer.start();

        // 生成实例唯一标识并注册到实例管理器
        this.instanceKey = generateInstanceKey();
        if (instanceManager != null && fsMqConfig != null && fsMqConfig.getMqClientDynamicUpdate()) {
            instanceManager.registerConsumer(this.instanceKey, this, this.properties, this.subscriptionTable);
        }

        LogUtil.info(log, "FsConsumerBean启动完成: {}", instanceKey);
    }

    /**
     * 替换Consumer实例
     * 用于配置变更时的重启更新
     *
     * @param newConsumer       新的Consumer实例
     * @param subscriptionTable 订阅表
     */
    public synchronized void replaceConsumerInstance(Consumer newConsumer, Map<Subscription, MessageListener> subscriptionTable) {
        this.consumer = newConsumer;
        this.subscriptionTable = subscriptionTable;
        LogUtil.info(log, "Consumer实例已替换: {}", instanceKey);
    }

    /**
     * 关闭该{@code Consumer}实例，建议配置为Bean的destroy-method
     */
    @Override
    public void shutdown() {
        // 从实例管理器中注销
        if (instanceManager != null && instanceKey != null) {
            instanceManager.unregisterConsumer(instanceKey);
        }

        if (this.consumer != null) {
            this.consumer.shutdown();
        }
        LogUtil.info(log, "FsConsumerBean已关闭: {}", instanceKey);
    }

    /**
     * 预销毁方法
     */
    @PreDestroy
    public void preDestroy() {
        shutdown();
    }

    @Override
    public void updateCredential(Properties credentialProperties) {
        if (this.consumer != null) {
            this.consumer.updateCredential(credentialProperties);
        }
    }

    @Override
    public void subscribe(String topic, String subExpression, MessageListener listener) {
        if (null == this.consumer) {
            throw new ONSClientException("subscribe must be called after consumerBean started");
        }
        this.consumer.subscribe(topic, subExpression, listener);
    }

    @Override
    public void subscribe(final String topic, final MessageSelector selector, final MessageListener listener) {
        if (null == this.consumer) {
            throw new ONSClientException("subscribe must be called after consumerBean started");
        }
        this.consumer.subscribe(topic, selector, listener);
    }

    @Override
    public void unsubscribe(String topic) {
        if (null == this.consumer) {
            throw new ONSClientException("unsubscribe must be called after consumerBean started");
        }
        this.consumer.unsubscribe(topic);
    }

    @Override
    public boolean isStarted() {
        return this.consumer != null && this.consumer.isStarted();
    }

    @Override
    public boolean isClosed() {
        return this.consumer == null || this.consumer.isClosed();
    }

    /**
     * 获取Properties
     */
    public Properties getProperties() {
        return properties;
    }

    /**
     * 设置Properties
     */
    public void setProperties(Properties properties) {
        this.properties = properties;
    }

    /**
     * 获取订阅表
     */
    public Map<Subscription, MessageListener> getSubscriptionTable() {
        return subscriptionTable;
    }

    /**
     * 设置订阅表
     */
    public void setSubscriptionTable(Map<Subscription, MessageListener> subscriptionTable) {
        this.subscriptionTable = subscriptionTable;
    }

    /**
     * 订阅Topic（从原ConsumerBean复制的逻辑）
     */
    private void subscribeTopics() {
        Iterator<Map.Entry<Subscription, MessageListener>> it = this.subscriptionTable.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<Subscription, MessageListener> next = it.next();
            if ("com.aliyun.openservices.ons.api.impl.notify.ConsumerImpl".equals(this.consumer.getClass().getCanonicalName())
                && (next.getKey() instanceof SubscriptionExt)) {
                SubscriptionExt subscription = (SubscriptionExt) next.getKey();
                for (Method method : this.consumer.getClass().getMethods()) {
                    if ("subscribeNotify".equals(method.getName())) {
                        try {
                            method.invoke(consumer, subscription.getTopic(), subscription.getExpression(),
                                    subscription.isPersistence(), next.getValue());
                        } catch (Exception e) {
                            throw new ONSClientException("subscribeNotify invoke exception", e);
                        }
                        break;
                    }
                }
            } else {
                Subscription subscription = next.getKey();
                if (subscription.getType() == null || ExpressionType.TAG.name().equals(subscription.getType())) {
                    this.subscribe(subscription.getTopic(), subscription.getExpression(), next.getValue());
                } else if (ExpressionType.SQL92.name().equals(subscription.getType())) {
                    this.subscribe(subscription.getTopic(), MessageSelector.bySql(subscription.getExpression()), next.getValue());
                } else {
                    throw new ONSClientException(String.format("Expression type %s is unknown!", subscription.getType()));
                }
            }
        }
    }

    /**
     * 更新Properties，解密AK/SK
     *
     * @return 更新后的Properties
     */
    private Properties updatePropertiesWithDecryptedKeys() {
        Properties updatedProperties = new Properties();
        updatedProperties.putAll(this.properties);

        // 如果配置了动态更新，则使用解密后的AK/SK
        if (fsMqConfig != null && fsMqConfig.getMqClientDynamicUpdate()) {
            updatedProperties.setProperty(PropertyKeyConst.AccessKey, fsMqConfig.getDecryptedAccessKey());
            updatedProperties.setProperty(PropertyKeyConst.SecretKey, fsMqConfig.getDecryptedSecretKey());
        }

        return updatedProperties;
    }

    /**
     * 校验AK/SK不能由外部设置
     */
    private void validateCredentials() {
        String accessKey = this.properties.getProperty(PropertyKeyConst.AccessKey);
        String secretKey = this.properties.getProperty(PropertyKeyConst.SecretKey);
        
        if (accessKey != null || secretKey != null) {
            String consumerId = this.properties.getProperty(PropertyKeyConst.ConsumerId);
            if (StringUtils.isBlank(consumerId)) {
                consumerId = properties.getProperty(PropertyKeyConst.GROUP_ID);
            }
            String errorMsg = String.format("%s消费者实例启动失败，阿里云ak和sk设置必须为空，系统默认使用统一配置ak/sk配置", consumerId);
            LogUtil.error(log, errorMsg);
            throw new ONSClientException(errorMsg);
        }
    }

    /**
     * 生成实例唯一标识
     *
     * @return 实例标识
     */
    private String generateInstanceKey() {
        return "consumer_" + System.identityHashCode(this) + "_" + System.currentTimeMillis();
    }
}
