# 阿里云MQ增强组件

## 概述

本组件在阿里云MQ-SDK的基础上，增加了Apollo配置变更时的无损实例更新功能。当Apollo中的AK/SK配置发生变更时，组件会自动更新所有MQ实例，确保应用无损运行。

## 核心特性

### 1. **无损更新策略**
- **Producer**: 创建新实例 → 原子性替换 → 延迟关闭旧实例
- **Consumer**: 随机延迟重启（1-30秒），确保集群可用性

### 2. **配置灵活性**
- 支持自定义Apollo配置key
- 兼容加密/明文AK/SK
- 完全向后兼容原有ProducerBean和ConsumerBean

### 3. **线程安全**
- 使用ConcurrentHashMap管理实例
- 原子性实例替换操作

## 使用方式

### 1. **配置Apollo参数**

默认使用以下Apollo配置key（可自定义）：
```properties
# AES解密密码
fs.aliyun.key.aes.encry.password=your_aes_password

# MQ访问凭证（可以是加密后的）
fs.aliyun.mq.access.key=your_access_key
fs.aliyun.mq.secret.key=your_secret_key

# 可选配置
fs.aliyun.mq.producer.close.delay.time=30
fs.aliyun.mq.consumer.restart.delay.time=30
```

### 2. **Spring Boot自动配置（推荐）**

#### 2.1 自动装配
本组件支持Spring Boot自动配置，只需在`application.properties`中配置必要参数即可自动装配：

```properties
# 必填配置
fs.aliyun.key.aes.encry.password=your_aes_password
fs.aliyun.mq.access.key=your_access_key
fs.aliyun.mq.secret.key=your_secret_key

# 可选配置
fs.aliyun.mq.producer.close.delay.time=30
fs.aliyun.mq.consumer.restart.delay.time=30
```

当classpath中存在阿里云MQ SDK且配置了必填参数时，组件会自动装配：
- `FsMqConfig` - MQ配置类
- `FsMqInstanceManager` - MQ实例管理器  
- `ApolloConfigListener` - Apollo配置监听器

#### 2.2 手动配置（可选）
如果需要自定义配置，可以手动创建配置类：

```java
@Configuration
public class CustomMqConfiguration {
    
    @Bean
    @Primary
    public FsMqConfig customFsMqConfig() {
        FsMqConfig config = new FsMqConfig();
        // 自定义Apollo配置key
        config.setAesPasswordKey("${custom.aes.password.key}");
        config.setAccessKeyKey("${custom.access.key}");
        config.setSecretKeyKey("${custom.secret.key}");
        return config;
    }
}
```

### 3. **配置Producer和Consumer**

#### 3.1 配置Producer
```java
@Configuration
public class ProducerConfiguration {
    
    @Value("${fs.aliyun.mq.nameserver.addr}")
    private String nameServerAddr;
    
    @Bean(initMethod = "start", destroyMethod = "shutdown")
    public FsProducerBean orderProducer() {
        FsProducerBean producer = new FsProducerBean();
        
        Properties properties = new Properties();
        properties.setProperty("ProducerId", "PID_ORDER_PRODUCER");
        properties.setProperty("NAMESRV_ADDR", nameServerAddr);
        // AK/SK会自动从配置中获取并解密
        
        producer.setProperties(properties);
        return producer;
    }
}
```

#### 3.2 配置Consumer
```java
@Configuration
public class ConsumerConfiguration {
    
    @Value("${fs.aliyun.mq.nameserver.addr}")
    private String nameServerAddr;
    
    @Bean(initMethod = "start", destroyMethod = "shutdown")
    public FsConsumerBean orderConsumer() {
        FsConsumerBean consumer = new FsConsumerBean();
        
        // 设置Properties
        Properties properties = new Properties();
        properties.setProperty("ConsumerId", "CID_ORDER_CONSUMER");
        properties.setProperty("NAMESRV_ADDR", nameServerAddr);
        properties.setProperty("ConsumeThreadNums", "20");
        properties.setProperty("MaxReconsumeTimes", "3");
        consumer.setProperties(properties);
        
        // 设置订阅表
        Map<Subscription, MessageListener> subscriptionTable = new HashMap<>();
        
        Subscription subscription = new Subscription();
        subscription.setTopic("ORDER_TOPIC");
        subscription.setExpression("*");
        
        subscriptionTable.put(subscription, orderMessageListener());
        consumer.setSubscriptionTable(subscriptionTable);
        
        return consumer;
    }
    
    @Bean
    public OrderMessageListener orderMessageListener() {
        return new OrderMessageListener();
    }
}
```

#### 3.3 消息监听器
```java
@Component
public class OrderMessageListener implements MessageListener {
    
    @Override
    public Action consume(Message message, ConsumeContext context) {
        try {
            String messageBody = new String(message.getBody());
            String topic = message.getTopic();
            String tag = message.getTag();
            String msgId = message.getMsgID();
            
            // 处理业务逻辑
            processOrderMessage(messageBody, tag);
            
            return Action.CommitMessage;
        } catch (Exception e) {
            // 根据异常类型决定是否重试
            return isRetryableException(e) ? Action.ReconsumeLater : Action.CommitMessage;
        }
    }
    
    private void processOrderMessage(String messageBody, String tag) {
        // 业务处理逻辑
    }
    
    private boolean isRetryableException(Exception exception) {
        // 判断异常是否可重试
        return true;
    }
}
```

### 4. **Java代码使用**

#### 4.1 使用Producer发送消息
```java
@Service
public class OrderService {
    
    @Autowired
    @Qualifier("orderProducer")
    private Producer orderProducer;
    
    public void createOrder(String orderInfo) {
        // 业务逻辑：保存订单
        String orderId = saveOrder(orderInfo);
        
        // 发送订单创建消息
        String messageBody = buildOrderMessage(orderId, orderInfo);
        Message message = new Message(
            "ORDER_TOPIC",           // Topic
            "ORDER_CREATED",         // Tag
            messageBody.getBytes()   // Body
        );
        message.setKey(orderId);     // 设置消息Key
        
        SendResult result = orderProducer.send(message);
        log.info("订单创建消息发送成功，MsgId: {}", result.getMessageId());
    }
    
    private String saveOrder(String orderInfo) {
        // 保存订单逻辑
        return "ORDER_" + System.currentTimeMillis();
    }
    
    private String buildOrderMessage(String orderId, String orderInfo) {
        // 构建消息体
        return String.format("{\"orderId\":\"%s\",\"orderInfo\":\"%s\"}", orderId, orderInfo);
    }
}
```

#### 4.2 Consumer自动消费消息
Consumer通过MessageListener自动消费消息，无需手动调用：

```java
@Component
public class OrderMessageListener implements MessageListener {
    
    @Override
    public Action consume(Message message, ConsumeContext context) {
        try {
            String messageBody = new String(message.getBody());
            String topic = message.getTopic();
            String tag = message.getTag();
            
            log.info("接收到订单消息 - Topic: {}, Tag: {}, Body: {}", topic, tag, messageBody);
            
            // 根据Tag处理不同类型的消息
            switch (tag) {
                case "ORDER_CREATED":
                    handleOrderCreated(messageBody);
                    break;
                case "ORDER_PAID":
                    handleOrderPaid(messageBody);
                    break;
                default:
                    log.warn("未知的消息标签: {}", tag);
                    break;
            }
            
            return Action.CommitMessage;
        } catch (Exception e) {
            log.error("消息处理失败", e);
            return Action.ReconsumeLater;
        }
    }
    
    private void handleOrderCreated(String messageBody) {
        // 处理订单创建消息
    }
    
    private void handleOrderPaid(String messageBody) {
        // 处理订单支付消息
    }
}
```

## 工作原理

### 1. **启动时**
1. FsProducerBean/FsConsumerBean启动时自动注册到FsMqInstanceManager
2. 如果配置了动态更新，会使用解密后的AK/SK创建实例

### 2. **配置变更时**
1. ApolloConfigListener监听到相关配置变更
2. 异步触发FsMqInstanceManager更新所有实例
3. Producer采用无损更新策略，Consumer采用随机延迟重启策略

### 3. **实例管理**
- 使用ConcurrentHashMap线程安全地管理所有实例
- 每个实例都有唯一标识，便于管理和日志追踪

## 自定义配置

### 1. **自定义Apollo配置key**

```java
@Configuration
public class CustomMqConfiguration {
    
    @Bean
    @Primary
    public FsMqConfig customFsMqConfig() {
        FsMqConfig config = new FsMqConfig();
        config.setAesPasswordKey("${your.custom.aes.key}");
        config.setAccessKeyKey("${your.custom.access.key}");
        config.setSecretKeyKey("${your.custom.secret.key}");
        return config;
    }
}
```

### 2. **多Producer配置**

```java
@Configuration
public class MultiProducerConfiguration {
    
    @Value("${fs.aliyun.mq.nameserver.addr}")
    private String nameServerAddr;
    
    @Bean(initMethod = "start", destroyMethod = "shutdown")
    public FsProducerBean orderProducer() {
        FsProducerBean producer = new FsProducerBean();
        Properties properties = new Properties();
        properties.setProperty("ProducerId", "PID_ORDER_PRODUCER");
        properties.setProperty("NAMESRV_ADDR", nameServerAddr);
        producer.setProperties(properties);
        return producer;
    }
    
    @Bean(initMethod = "start", destroyMethod = "shutdown")
    public FsProducerBean paymentProducer() {
        FsProducerBean producer = new FsProducerBean();
        Properties properties = new Properties();
        properties.setProperty("ProducerId", "PID_PAYMENT_PRODUCER");
        properties.setProperty("NAMESRV_ADDR", nameServerAddr);
        producer.setProperties(properties);
        return producer;
    }
}
```

### 3. **多Topic Consumer配置**

```java
@Configuration
public class MultiTopicConsumerConfiguration {
    
    @Value("${fs.aliyun.mq.nameserver.addr}")
    private String nameServerAddr;
    
    @Bean(initMethod = "start", destroyMethod = "shutdown")
    public FsConsumerBean notificationConsumer() {
        FsConsumerBean consumer = new FsConsumerBean();
        
        Properties properties = new Properties();
        properties.setProperty("ConsumerId", "CID_NOTIFICATION_CONSUMER");
        properties.setProperty("NAMESRV_ADDR", nameServerAddr);
        consumer.setProperties(properties);
        
        // 多Topic订阅
        Map<Subscription, MessageListener> subscriptionTable = new HashMap<>();
        
        // 订单通知
        Subscription orderNotificationSub = new Subscription();
        orderNotificationSub.setTopic("ORDER_NOTIFICATION_TOPIC");
        orderNotificationSub.setExpression("*");
        subscriptionTable.put(orderNotificationSub, orderNotificationListener());
        
        // 支付通知
        Subscription paymentNotificationSub = new Subscription();
        paymentNotificationSub.setTopic("PAYMENT_NOTIFICATION_TOPIC");
        paymentNotificationSub.setExpression("*");
        subscriptionTable.put(paymentNotificationSub, paymentNotificationListener());
        
        consumer.setSubscriptionTable(subscriptionTable);
        return consumer;
    }
    
    @Bean
    public OrderNotificationListener orderNotificationListener() {
        return new OrderNotificationListener();
    }
    
    @Bean
    public PaymentNotificationListener paymentNotificationListener() {
        return new PaymentNotificationListener();
    }
}
```

## 注意事项

1. **向后兼容**: 新组件完全兼容原有ProducerBean和ConsumerBean的使用方式
2. **配置要求**: 需要在Apollo中配置相应的AK/SK和AES密码
3. **动态更新**: 只有当Apollo配置key以`${}`格式配置时才会启用动态更新
4. **线程安全**: 所有操作都是线程安全的，可以在多线程环境下使用
5. **日志监控**: 组件会输出详细的日志，便于监控和排查问题

## 依赖要求

- Spring Framework
- Apollo Client
- 阿里云MQ SDK (ons-client)
- fsframework-core (日志工具)
- fsframework-extend-util (配置工具、加密工具)
