/**
 * fshows.com
 * Copyright (C) 2013-2024 All Rights Reserved.
 */
package com.fshows.fsframework.extend.aliyun.oss;

import cn.hutool.core.codec.Base64;
import cn.hutool.core.util.HexUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.digest.DigestUtil;
import cn.hutool.crypto.symmetric.AES;
import cn.hutool.crypto.symmetric.SymmetricAlgorithm;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyun.oss.model.ObjectMetadata;
import com.ctrip.framework.apollo.enums.PropertyChangeType;
import com.ctrip.framework.apollo.model.ConfigChange;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfigChangeListener;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.extend.util.FsAESUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.Date;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author wangqilei
 * @version OssLocalClient.java, v 0.1 2024-12-10 3:26 PM wangqilei
 */
@Slf4j
@Service
public class FsOssClient {
    /**
     * oss 外网地址
     */
    @Value("${fs.aliyun.oss.out.url:https://oss-cn-hangzhou.aliyuncs.com}")
    private String ossEndPointOut;
    /**
     * oss 内网地址
     */
    @Value("${fs.aliyun.oss.inner.url:https://oss-cn-hangzhou-internal.aliyuncs.com}")
    private String ossEndPointInner;

    /**
     * 外网链接过期时间
     */
    @Value("${fs.aliyun.oss.url.expire.time:3600}")
    private Long urlExpireTime;

    @Value("${fs.aliyun.key.aes.encry.password}")
    private String aesPassword;

    /**
     * 阿波罗配置前缀
     */
    private static final String OSS_ACCESS_KEY_PREFIX = "fs.aliyun.oss.access.key";
    private static final String OSS_SECRET_KEY_PREFIX = "fs.aliyun.oss.secret.key";

    @Value("${fs.aliyun.oss.access.key}")
    private String encryAccessKey;

    @Value("${fs.aliyun.oss.secret.key}")
    private String encrySecretKey;

    private volatile OSSClient outOssClient;

    private volatile OSSClient innerOssClient;

    private final ScheduledExecutorService executorService = Executors.newScheduledThreadPool(1);

    @PostConstruct
    public void init() {
        String decryptAccess = FsAESUtil.decryptKey(encryAccessKey, aesPassword);
        String decryptSecret = FsAESUtil.decryptKey(encrySecretKey, aesPassword);

        if (outOssClient == null) {
            outOssClient = new OSSClient(ossEndPointOut, decryptAccess, decryptSecret);
        }
        if (innerOssClient == null) {
            innerOssClient = new OSSClient(ossEndPointInner, decryptAccess, decryptSecret);
        }
    }

    @ApolloConfigChangeListener(interestedKeys = {"fs.aliyun.oss."})
    public synchronized void refreshOssClient(ConfigChangeEvent changeEvent) {
        LogUtil.info(log, "refreshOssClient >> 接收到AKSK变换通知");
        ConfigChange accessConfigChange = changeEvent.getChange(OSS_ACCESS_KEY_PREFIX);
        ConfigChange secretConfigChange = changeEvent.getChange(OSS_SECRET_KEY_PREFIX);
        LogUtil.info(log, "refreshOssClient >> 接收到AKSK变换通知 >> access={} , secret={}", accessConfigChange, secretConfigChange);

        String refreshEncryAccessKey = encryAccessKey;
        if (accessConfigChange != null
                && !accessConfigChange.getChangeType().equals(PropertyChangeType.DELETED)
                && !accessConfigChange.getOldValue().equals(accessConfigChange.getNewValue())) {
            refreshEncryAccessKey = accessConfigChange.getNewValue();
        }

        String refreshEncrySecretKey = encrySecretKey;
        if (secretConfigChange != null
                && !secretConfigChange.getChangeType().equals(PropertyChangeType.DELETED)
                && !secretConfigChange.getOldValue().equals(secretConfigChange.getNewValue())) {
            refreshEncrySecretKey = secretConfigChange.getNewValue();
        }

        if (refreshEncryAccessKey.equals(encryAccessKey) && refreshEncrySecretKey.equals(encrySecretKey)) {
            LogUtil.info(log, "refreshOssClient >> aksk没有发生过变更 >> access={} , secret={}", accessConfigChange, secretConfigChange);
            // aksk没有发生过变更
            return;
        }

        // 临时客户端，用于关闭，释放资源
        OSSClient tmpOutOssClient = outOssClient;
        OSSClient tmpInnerOssClient = innerOssClient;

        // 对key解密
        String accessKey = FsAESUtil.decryptKey(refreshEncryAccessKey, aesPassword);
        String secretKey = FsAESUtil.decryptKey(refreshEncrySecretKey, aesPassword);

        try {
            LogUtil.info(log, "refreshOssClient >> 初始化新客户端 >> accessKey={} , secretKey={}", accessKey, secretKey);
            outOssClient = new OSSClient(ossEndPointOut, accessKey, secretKey);
            innerOssClient = new OSSClient(ossEndPointInner, accessKey, secretKey);
        } catch (Exception e) {
            LogUtil.error(log, "refreshOssClient >> oss动态替换aksk异常", e);
        } finally {
            // 延迟关闭旧客户端，默认30秒
            executorService.schedule(() -> {
                // 当新客户端已经创建完成，才能关闭旧客户端
                if (tmpOutOssClient != null && tmpOutOssClient != outOssClient) {
                    tmpOutOssClient.shutdown();
                }
                if (tmpInnerOssClient != null && tmpInnerOssClient != innerOssClient) {
                    tmpInnerOssClient.shutdown();
                }
                LogUtil.info(log, "refreshOssClient >> 旧客户端关闭完成");
            }, 30, TimeUnit.SECONDS);
        }
    }

    /**
     * 上传文件（可选择内外网环境）
     */
    public void uploadFileWithOutFlag(String bucketName, String key, InputStream inputStream, boolean outOssFlag) {
        if (outOssFlag) {
            outOssClient.putObject(bucketName, key, inputStream);
        } else {
            innerOssClient.putObject(bucketName, key, inputStream);
        }
    }

    /**
     * 上传文件
     */
    public void uploadFile(String bucketName, String key, String filePath) throws IOException {
        File file = new File(filePath);
        InputStream input = Files.newInputStream(file.toPath());
        this.uploadFileWithOutFlag(bucketName, key, input, false);
    }

    /**
     * 上传文件
     */
    public void uploadFile(String bucketName, String key, File file) throws IOException {
        InputStream input = Files.newInputStream(file.toPath());
        this.uploadFileWithOutFlag(bucketName, key, input, false);
    }

    /**
     * 上传文件
     */
    public void uploadFile(String bucketName, String key, InputStream inputStream) {
        this.uploadFileWithOutFlag(bucketName, key, inputStream, false);
    }

    /**
     * 上传文件
     */
    public void uploadFile(String bucketName, String key, String filename, ObjectMetadata metadata) throws IOException {
        File file = new File(filename);
        InputStream input = Files.newInputStream(file.toPath());
        innerOssClient.putObject(bucketName, key, input, metadata);
    }

    /**
     * 下载文件（可选择内外网环境）
     */
    public void downloadFileWithOutFlag(String bucketName, String key, String filePath, boolean outFlag) {
        if (outFlag) {
            outOssClient.getObject(new GetObjectRequest(bucketName, key), new File(filePath));
        } else {
            innerOssClient.getObject(new GetObjectRequest(bucketName, key), new File(filePath));
        }
    }

    /**
     * 下载文件（可选择内外网环境）
     */
    public void downloadFileWithOutFlag(String bucketName, String key, File file, boolean outFlag) {
        if (outFlag) {
            outOssClient.getObject(new GetObjectRequest(bucketName, key), file);
        } else {
            innerOssClient.getObject(new GetObjectRequest(bucketName, key), file);
        }
    }

    /**
     * 下载文件
     */
    public void downloadFile(String bucketName, String key, String filePath) {
        this.downloadFileWithOutFlag(bucketName, key, filePath, false);
    }

    /**
     * 下载文件
     */
    public void downloadFile(String bucketName, String key, File file) {
        this.downloadFileWithOutFlag(bucketName, key, file, false);
    }

    /**
     * 下载文件
     */
    public void downloadFileKeep(String bucketName, String key, File file) {
        //这里key 不能以 / 开头
        if (key.startsWith("/")) {
            key = key.substring(1);
        }
        this.downloadFileWithOutFlag(bucketName, key, file, false);
    }

    /**
     * 生成下载url（可选择内外网环境）
     */
    public String generateFileUrlWithExpireAndOutFlag(String bucketName, String key, boolean outFlag, Long expireTime) {
        Date time = new Date(System.currentTimeMillis() + (Objects.isNull(expireTime) ? urlExpireTime : expireTime));
        if (outFlag) {
            return outOssClient.generatePresignedUrl(bucketName, key, time).toString();
        } else {
            return innerOssClient.generatePresignedUrl(bucketName, key, time).toString();
        }
    }

    /**
     * 生成下载url（可选择内外网环境）
     */
    public String generateFileUrlWithOutFlag(String bucketName, String key, boolean outFlag) {
        return generateFileUrlWithExpireAndOutFlag(bucketName, key, outFlag, null);
    }

    /**
     * 生成下载url
     */
    public String generateFileUrl(String bucketName, String key) {
        return generateFileUrlWithExpireAndOutFlag(bucketName, key, true, null);
    }

    /**
     * 生成下载url
     */
    public String generateFileUrl(String bucketName, String key, Long expireTime) {
        return generateFileUrlWithExpireAndOutFlag(bucketName, key, true, expireTime);
    }

    public static void main(String[] args) {
        // 要加密的字符串
//        String before = new String(SecureUtil.generateKey(SymmetricAlgorithm.AES.getValue()).getEncoded(), StandardCharsets.UTF_8);
//        String inputString = Base64.encode(before);
//        System.out.println("password base64: " + inputString);
//
//        System.out.println("password: " + new String(HexUtil.encodeHex(Base64.decode(inputString))));

        // 进行 MD5 加密
        byte[] md5SecretKey = DigestUtil.md5(DigestUtil.md5("406c42688e7f7db4a8f3912772725a5d".getBytes(StandardCharsets.UTF_8)));
        System.out.println("password md5: " + new String(HexUtil.encodeHex(md5SecretKey)));

        AES aes = SecureUtil.aes(md5SecretKey);

        String accessKey = "LTAI5tMacBHafqzmAfK4DdoK";
        String secretKey = "vNbUdImu3eSX8yKi5jl2nMG8oneFGW";

        String encryAccessKey = Base64.encode(aes.encrypt(accessKey));
        String encrySecretKey = Base64.encode(aes.encrypt(secretKey));
        System.out.println("oss accessKey encode: " + new String(HexUtil.encodeHex(Base64.decode(encryAccessKey))));
        System.out.println("oss accessKey encode: " + new String(HexUtil.encodeHex(Base64.decode(encrySecretKey))));

        String decryptAccessKey = new String(aes.decrypt(Base64.decode(encryAccessKey)));
        String decryptSecretKey = new String(aes.decrypt(Base64.decode(encrySecretKey)));


        System.out.println("oss accessKey decode: " + decryptAccessKey);
        System.out.println("oss secretKey decode: " + decryptSecretKey);
    }
}