/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.extend.dubbo.filter;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.rpc.Filter;
import com.alibaba.dubbo.rpc.Invocation;
import com.alibaba.dubbo.rpc.Invoker;
import com.alibaba.dubbo.rpc.Result;
import com.alibaba.dubbo.rpc.RpcException;
import com.alibaba.fastjson.JSONObject;
import com.fshows.fsframework.core.constants.StringPool;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.extend.dubbo.model.DubboThreadPoolMonitorConfigConvertModel;
import com.fshows.fsframework.extend.dubbo.model.DubboThreadPoolMonitorConfigModel;
import com.fshows.fsframework.extend.util.SpringContextUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.env.Environment;

import java.math.BigDecimal;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * dubbo线程池占用方法监控
 *
 * @author zhaoxumin
 * @version GlobalDubboThreadPoolMonitor.java, v 0.1 2024-08-06 17:52 zhaoxumin
 */
@Slf4j
@Activate(group = {Constants.PROVIDER}, order = -9999)
public class GlobalDubboThreadPoolMonitor implements Filter {

    /**
     * dubbo线程池占用方法监控开关. 0-关闭 1-开启 apollo中配置
     */
    private final static String DUBBO_THREAD_POOL_MONITOR_SWITCH = "dubbo.thread.pool.monitor.switch";

    /**
     * dubbo线程池监控参数配置. apollo中配置
     */
    private final static String DUBBO_THREAD_POOL_MONITOR_CONFIG = "dubbo.thread.pool.monitor.config";

    private static volatile DubboThreadPoolMonitorConfigConvertModel configModel = null;

    /**
     * 调度任务执行器
     */
    private static final ScheduledExecutorService SCHEDULER = Executors.newScheduledThreadPool(1);

    /**
     * 保存dubbo方法的调用计数
     */
    private final Map<String, AtomicInteger> methodAndCountMap = new ConcurrentHashMap<>();

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        Environment environment = SpringContextUtil.getEnvironment();
        if (environment != null) {
            if (StringPool.ONE.equals(environment.getProperty(DUBBO_THREAD_POOL_MONITOR_SWITCH))) {
                DubboThreadPoolMonitorConfigConvertModel threadPoolMonitorConfigModel = getThreadPoolMonitorConfigModel();
                if (threadPoolMonitorConfigModel != null) {
                    String methodName = invoker.getUrl().getPath() + "." + invocation.getMethodName();

                    methodAndCountMap.compute(methodName, (key, count) -> {
                        if (count == null) {
                            return new AtomicInteger(1);
                        } else {
                            count.incrementAndGet();
                            return count;
                        }
                    });
                    try {
                        return invoker.invoke(invocation);
                    } catch (RpcException e) {
                        if (e.getMessage() != null && e.getMessage().toLowerCase().contains("threadpool is exhausted")) {
                            // dubbo线程池拒绝服务异常
                            recordThreadPoolRunningMethod(threadPoolMonitorConfigModel.getApplicationThreadCount(), 0);
                        }
                        throw e;
                    } finally {
                        methodAndCountMap.compute(methodName, (key, count) -> {
                            if (count != null) {
                                if (count.decrementAndGet() == 0) {
                                    // 计数为0时，删除该方法记录
                                    return null;
                                }
                            }
                            return count;
                        });
                    }
                }
            } else {
                configModel = null;
            }
        }
        return invoker.invoke(invocation);
    }

    /**
     * 日志输出线程池占用情况
     */
    private void recordThreadPoolRunningMethod(Integer applicationThreadCount, float usageThreshold) {
        // 排序计数列表
        List<Map.Entry<String, AtomicInteger>> methodAndCountList = methodAndCountMap.entrySet().stream()
                .sorted((e1, e2) -> Integer.compare(e2.getValue().get(), e1.getValue().get()))
                .collect(Collectors.toList());

        if (usageThreshold != 0) {
            BigDecimal threshold = new BigDecimal(applicationThreadCount).multiply(new BigDecimal(usageThreshold)).setScale(2);
            if (threshold.compareTo(new BigDecimal(methodAndCountList.size())) < 0) {
                LogUtil.warn(log, "record thread pool running method >> dubbo线程池占用方法监控，超过设定阈值 >> applicationThreadCount={}, usageCount={}, methodAndCountList={}",
                        applicationThreadCount, methodAndCountList.size(), methodAndCountList);
            }
        }
        LogUtil.warn(log, "record thread pool running method >> dubbo线程池占用方法监控 >> methodAndCountList={}", methodAndCountList);
    }

    /**
     * 获取线程池监控配置
     *
     * @return
     */
    private DubboThreadPoolMonitorConfigConvertModel getThreadPoolMonitorConfigModel() {
        if (configModel == null) {
            synchronized (this) {
                if (configModel == null) {
                    Environment environment = SpringContextUtil.getEnvironment();
                    if (environment == null) {
                        return null;
                    }
                    DubboThreadPoolMonitorConfigConvertModel threadPoolMonitorConfigConvertModel = new DubboThreadPoolMonitorConfigConvertModel();
                    String threadPoolMonitorConfigStr = environment.getProperty(DUBBO_THREAD_POOL_MONITOR_CONFIG);
                    if (StrUtil.isNotBlank(threadPoolMonitorConfigStr)) {
                        DubboThreadPoolMonitorConfigModel config = JSONObject.parseObject(threadPoolMonitorConfigStr, DubboThreadPoolMonitorConfigModel.class);
                        if (config != null) {
                            threadPoolMonitorConfigConvertModel.setApplicationThreadCount(config.getApplicationThreadCount());
                            List<DubboThreadPoolMonitorConfigModel.ThreadPoolMonitorPointConfigModel> threadPoolMonitorPointConfigList = config.getThreadPoolMonitorPointConfigModelList();
                            if (CollUtil.isNotEmpty(threadPoolMonitorPointConfigList)) {
                                List<DubboThreadPoolMonitorConfigConvertModel.ThreadPoolMonitorPointConfigConvertModel> threadPoolMonitorPointConfigConvertModelList = Lists.newArrayList();
                                for (DubboThreadPoolMonitorConfigModel.ThreadPoolMonitorPointConfigModel threadPoolMonitorPointConfigModel : threadPoolMonitorPointConfigList) {
                                    if (StrUtil.isNotBlank(threadPoolMonitorPointConfigModel.getMonitorPeriod())) {
                                        String[] split = threadPoolMonitorPointConfigModel.getMonitorPeriod().split(StringPool.TILDA);
                                        if (split.length == 2) {
                                            DubboThreadPoolMonitorConfigConvertModel.ThreadPoolMonitorPointConfigConvertModel threadPoolMonitorPointConfigConvertModel = new DubboThreadPoolMonitorConfigConvertModel.ThreadPoolMonitorPointConfigConvertModel();
                                            threadPoolMonitorPointConfigConvertModel.setMonitorPeriodStartTime(LocalTime.parse(split[0], DateTimeFormatter.ofPattern("HH:mm:ss")));
                                            threadPoolMonitorPointConfigConvertModel.setMonitorPeriodEndTime(LocalTime.parse(split[1], DateTimeFormatter.ofPattern("HH:mm:ss")));
                                            threadPoolMonitorPointConfigConvertModel.setMonitorFrequency(threadPoolMonitorPointConfigModel.getMonitorFrequency());
                                            threadPoolMonitorPointConfigConvertModel.setUsageThreshold(Float.parseFloat(threadPoolMonitorPointConfigModel.getUsageThreshold()));
                                            threadPoolMonitorPointConfigConvertModelList.add(threadPoolMonitorPointConfigConvertModel);
                                        }
                                    }
                                }
                                threadPoolMonitorConfigConvertModel.setThreadPoolMonitorPointConfigList(threadPoolMonitorPointConfigConvertModelList);
                                // 初始化参数完成，开启独立线程监控
                                startMonitorTask(threadPoolMonitorConfigConvertModel);
                            }
                        }
                    }
                    configModel = threadPoolMonitorConfigConvertModel;
                }
            }
        }
        return configModel;
    }

    /**
     * 开启独立线程监控线程池
     *
     * @param threadPoolMonitorConfigConvertModel
     */
    private void startMonitorTask(DubboThreadPoolMonitorConfigConvertModel threadPoolMonitorConfigConvertModel) {
        if (threadPoolMonitorConfigConvertModel != null && CollUtil.isNotEmpty(threadPoolMonitorConfigConvertModel.getThreadPoolMonitorPointConfigList())) {
            new Thread(() -> {
                for (DubboThreadPoolMonitorConfigConvertModel.ThreadPoolMonitorPointConfigConvertModel configConvertModel : threadPoolMonitorConfigConvertModel.getThreadPoolMonitorPointConfigList()) {
                    LocalTime start = configConvertModel.getMonitorPeriodStartTime();
                    LocalTime end = configConvertModel.getMonitorPeriodEndTime();
                    long monitorFrequency = configConvertModel.getMonitorFrequency();

                    long initialDelay = 0;
                    LocalTime nowTime = LocalTime.now();
                    if (nowTime.isBefore(start)) {
                        initialDelay = nowTime.until(start, ChronoUnit.SECONDS);
                    }

                    // 调度任务
                    SCHEDULER.scheduleAtFixedRate(() -> {
                        LocalTime now = LocalTime.now();
                        if (now.isAfter(start) && now.isBefore(end)) {
                            recordThreadPoolRunningMethod(threadPoolMonitorConfigConvertModel.getApplicationThreadCount(), configConvertModel.getUsageThreshold());
                        }
                    }, initialDelay, monitorFrequency, TimeUnit.SECONDS);
                }
            }).start();
        }
    }
}