/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.extend.dubbo.filter;

import cn.hutool.core.util.StrUtil;
import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.common.utils.CollectionUtils;
import com.alibaba.dubbo.config.spring.ServiceBean;
import com.alibaba.dubbo.rpc.Filter;
import com.alibaba.dubbo.rpc.Invocation;
import com.alibaba.dubbo.rpc.Invoker;
import com.alibaba.dubbo.rpc.Result;
import com.alibaba.dubbo.rpc.RpcContext;
import com.alibaba.dubbo.rpc.RpcException;
import com.alibaba.fastjson.JSONObject;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.core.utils.SystemClock;
import com.fshows.fsframework.extend.dubbo.model.DubboServiceTimeConsumingLevelConfigModel;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * dubbo 全局 慢facade 过滤器
 *
 * @author zhaoxumin
 * @version GlobalSlowFacadeFilter.java, v 0.1 2023-05-31 10:42 zhaoxumin
 */
@Slf4j
@Activate(group = {Constants.PROVIDER, Constants.CONSUMER}, order = -9999)
public class GlobalSlowFacadeFilter implements Filter {

    /**
     * dubbo服务耗时等级配置. apollo中配置
     */
    private final static String DUBBO_SERVICE_TIME_CONSUMING_LEVEL_CONFIG = "dubbo.service.time.consuming.level.config";

    private static List<DubboServiceTimeConsumingLevelConfigModel> timeConsumingLevelConfigModels = Lists.newArrayList();

    static {
        String configStr = ServiceBean.getSpringContext().getEnvironment().getProperty(DUBBO_SERVICE_TIME_CONSUMING_LEVEL_CONFIG);
        if (StrUtil.isNotBlank(configStr)) {
            List<DubboServiceTimeConsumingLevelConfigModel> configModels = JSONObject.parseArray(configStr, DubboServiceTimeConsumingLevelConfigModel.class);
            if (CollectionUtils.isNotEmpty(configModels)) {
                timeConsumingLevelConfigModels = configModels.stream().sorted(Comparator.comparingInt(DubboServiceTimeConsumingLevelConfigModel::getLowerThreshold).reversed())
                        .distinct().collect(Collectors.toList());
            }
        }
    }

    /**
     * do invoke filter.
     * <p>
     * <code>
     * // before filter
     * Result result = invoker.invoke(invocation);
     * // after filter
     * return result;
     * </code>
     *
     * @param invoker    service
     * @param invocation invocation.
     * @return invoke result.
     * @throws RpcException
     * @see Invoker#invoke(Invocation)
     */
    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {

        if (CollectionUtils.isNotEmpty(timeConsumingLevelConfigModels)) {

            // 获得开始时间
            long rpcStartTime = SystemClock.millisClock().now();
            // 调用接口
            Result result = invoker.invoke(invocation);
            // 获得结束时间
            long rpcEndTime = SystemClock.millisClock().now();

            // rpc调用超过3秒，即为慢facade
            long timeConsuming = rpcEndTime - rpcStartTime;

            // 获得 RPC 方法名
            String methodName = invoker.getUrl().getPath() + "." + RpcContext.getContext().getMethodName();
            Object[] arguments = invocation.getArguments();
            for (DubboServiceTimeConsumingLevelConfigModel timeConsumingLevelConfigModel : timeConsumingLevelConfigModels) {
                if (timeConsuming > timeConsumingLevelConfigModel.getLowerThreshold()) {
                    LogUtil.warn(log, "RPC 慢facade接口调用结束，耗时等级:【{}】，methodName = {}, agruments = {}, time = {}ms",
                            timeConsumingLevelConfigModel.getLevel(), methodName, arguments, timeConsuming);
                    break;
                }
            }
            return result;
        }
        return invoker.invoke(invocation);
    }
}