/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.easypay.sdk.client.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.fshows.easypay.sdk.client.EasyPayBaseClient;
import com.fshows.easypay.sdk.constant.CommonConstant;
import com.fshows.easypay.sdk.enums.EasyPayApiEnum;
import com.fshows.easypay.sdk.enums.SignTypeEnum;
import com.fshows.easypay.sdk.exception.EasyPayException;
import com.fshows.easypay.sdk.request.base.EasyPayDataBaseRequest;
import com.fshows.easypay.sdk.request.trade.EasyPayQsTradeDataBaseRequest;
import com.fshows.easypay.sdk.request.trade.EasyPayTradeDataBaseRequest;
import com.fshows.easypay.sdk.response.base.EasyPayDataBaseResponse;
import com.fshows.easypay.sdk.util.*;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

import javax.validation.constraints.NotBlank;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;

/**
 * @author zhaoxumin
 * @version EasyPayBaseClientImpl.java, v 0.1 2023-10-27 13:32
 */
@Slf4j
@Data
public class EasyPayBaseClientImpl implements EasyPayBaseClient {

    /**
     * 易生接口网关根URL
     */
    @NotBlank
    protected String apiParentUrl;

    /**
     * 机构号
     */
    @NotBlank
    private final String orgId;

    /**
     * 签名类型
     */
    @NotBlank
    private final String signType;

    /**
     * 用于加签的付呗私钥
     */
    @NotBlank
    private String fubeiPrivateKey;

    /**
     * 用于验签的易生公钥
     */
    @NotBlank
    private String easyPayPublicKey;


    /**
     * 验签时是否过滤空值
     */
    private Boolean signFilterNull = false;

    /**
     * http连接超时时间
     */
    protected int connectionTimeout = CommonConstant.DEFAULT_CONNECTION_TIMEOUT;

    /**
     * http response读取超时时间
     */
    protected int readTimeout = CommonConstant.DEFAULT_READ_TIMEOUT;

    public EasyPayBaseClientImpl(String apiParentUrl, String orgId, String signType,
                                 String fubeiPrivateKey, String easyPayPublicKey) {
        this.apiParentUrl = apiParentUrl;
        this.orgId = orgId;
        this.signType = signType;
        this.fubeiPrivateKey = fubeiPrivateKey;
        this.easyPayPublicKey = easyPayPublicKey;
    }

    public EasyPayBaseClientImpl(String apiParentUrl, String orgId, String signType,
                                 String fubeiPrivateKey, String easyPayPublicKey,
                                 int connectionTimeout, int readTimeout) {
        this.apiParentUrl = apiParentUrl;
        this.orgId = orgId;
        this.signType = signType;
        this.fubeiPrivateKey = fubeiPrivateKey;
        this.easyPayPublicKey = easyPayPublicKey;
        this.connectionTimeout = connectionTimeout;
        this.readTimeout = readTimeout;
    }

    /**
     * 请求易生接口
     *
     * @param request
     * @param tradeApiDefinitionEnum
     * @return
     * @param <T>
     * @param <R>
     * @throws EasyPayException
     */
    @Override
    public <T extends EasyPayDataBaseResponse, R extends EasyPayApiDefinition> T execute(EasyPayDataBaseRequest<T, R> request, R tradeApiDefinitionEnum) throws EasyPayException {
        return execute(request, tradeApiDefinitionEnum, null);
    }

    @Override
    public <T extends EasyPayDataBaseResponse, R extends EasyPayApiDefinition> T execute(EasyPayDataBaseRequest<T, R> request, R tradeApiDefinitionEnum, String subAppId) throws EasyPayException {
        String method = tradeApiDefinitionEnum.getMethod();
        // 验证参数
        validateParam(request, method);
        // 获取开始时间
        final long startTime = System.currentTimeMillis();
        String url = apiParentUrl + method;
        try {
            //获取请求参数
            String requestData = getRequestData(request, subAppId);
            log.info("【easypay-sdk】接口调用开始 >> url={},request={}", url, requestData);

            HashMap<String, String> headers = new HashMap<>();
            headers.put("Accept", CommonConstant.ACCEPT_TYPE);
            headers.put("Accept-Encoding", CommonConstant.ACCEPT_ENCODING);
            String fsHttpResult = FsHttpUtil.postString(url, requestData, CommonConstant.UTF_8, CommonConstant.JSON_CONTENT_TYPE, connectionTimeout, readTimeout, headers);
            log.info("record bad-liquidation info >> success >> 【easypay-sdk】接口调用结束 >> url={}, request={},cost={}", url, request, System.currentTimeMillis() - startTime);
            log.info("【easypay-sdk】接口调用结束 >> url={},request={},response={},cost={}", url, requestData, fsHttpResult, System.currentTimeMillis() - startTime);
            if (fsHttpResult != null) {
                return (T)parseResponse(fsHttpResult, tradeApiDefinitionEnum);
            }
        } catch (Exception ex) {
            log.error("record bad-liquidation info >> fail >>【easypay-sdk】接口调用失败 >> url={}, request={},ex={},cost={}", url, request, ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            log.error("【easypay-sdk】接口调用失败 >> url={},request={},ex={},cost={}", url, request, ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            throw new EasyPayException(ex.getMessage());
        }
        return null;
    }

    @SuppressWarnings("unchecked")
    private <R extends EasyPayApiDefinition> EasyPayDataBaseResponse parseResponse(String fsHttpResult, R tradeApiDefinitionEnum) throws Exception {
        if (!checkSign(fsHttpResult)) {
            LogUtil.error(log, "【easypay-sdk】响应结果验签失败 >> fsHttpResult={}", fsHttpResult);
            throw new EasyPayException("[easypay-sdk]响应结果验签失败");
        }

        JSONObject resMap = JSONObject.parseObject(fsHttpResult);
        JSONObject responseJson = new JSONObject();
        JSONObject data = resMap.getJSONObject("data");
        JSONObject appendData = resMap.getJSONObject("appendData");

        if (CollUtil.isNotEmpty(data)) {
            responseJson.putAll(data);
        }
        if (CollUtil.isNotEmpty(appendData)) {
            responseJson.put("appendData", appendData);
        }

        try {
            EasyPayDataBaseResponse response = EasyPayRequestUtils.getObjectFromMap(responseJson, tradeApiDefinitionEnum.getResponseClass());
            String sysRetcode = StrUtil.blankToDefault(resMap.getString("sysRetcode"), resMap.getString("sysRetCode"));
            String sysRetmsg = StrUtil.blankToDefault(resMap.getString("sysRetmsg"), resMap.getString("sysRetMsg"));
            if (StrUtil.isBlank(sysRetcode) && StrUtil.isBlank(sysRetmsg)) {
                sysRetcode = StrUtil.blankToDefault(resMap.getString("resultCode"), resMap.getString("resultCode"));
                sysRetmsg = StrUtil.blankToDefault(resMap.getString("resultMsg"), resMap.getString("resultMsg"));
            }
            response.setSysRetcode(sysRetcode);
            response.setSysRetmsg(sysRetmsg);
            response.setAppendRetcode(response.getAppendRetcode());
            response.setAppendRetmsg(response.getAppendRetmsg());
            response.setResMap(resMap);
            return response;
        } catch (Exception e) {
            LogUtil.error(log, "【easypay-sdk】响应结果反序列化异常 >> fsHttpResult={}", e, fsHttpResult);
            throw new EasyPayException("[easypay-sdk]响应结果反序列化异常");
        }
    }

    private boolean checkSign(String fsHttpResult) {
        JSONObject resMap = JSONObject.parseObject(fsHttpResult);
        if ("000000".equals(resMap.getString("sysRetcode")) || "000000".equals(resMap.getString("sysRetCode"))) {
            String sign = resMap.getString("sign");
            if (StringUtils.isEmpty(sign)) {
                return false;
            }
            // 业务参数
            Map<String, Object> bizResult = resMap.getJSONObject("data");
            // 如果空值过滤开关打开,则过滤业务参数中的空值
            if (Boolean.TRUE.equals(signFilterNull) && bizResult != null) {
                bizResult = bizResult.entrySet().stream()
                        .filter(e-> e.getValue() != null)
                        .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
            }
            // 返参验签
            return CollUtil.isNotEmpty(bizResult)
                    && SignUtil.verifySign(bizResult, signType, resMap.get("sign").toString(), easyPayPublicKey);
        } else {
            return true;
        }
    }

    /**
     * 获取签名参数
     *
     * @param request
     * @param appId
     * @return
     * @throws Exception
     */
    private String getRequestData(EasyPayDataBaseRequest request, String appId) throws Exception {
        HashMap<String, Object> requestDataMap = new HashMap<>();
        requestDataMap.put("orgId", StringUtils.defaultIfBlank(appId, this.orgId));
        if (request instanceof EasyPayTradeDataBaseRequest) {
            EasyPayTradeDataBaseRequest tradeDataBaseRequest = (EasyPayTradeDataBaseRequest) request;
            requestDataMap.put("orgMercode", tradeDataBaseRequest.getOrgMercode());
            requestDataMap.put("orgTermno", tradeDataBaseRequest.getOrgTermno());
            requestDataMap.put("orgTrace", StrUtil.blankToDefault(tradeDataBaseRequest.getOrgTrace(),
                    ReqSerialNoUtil.getReqSerialNo()));
        }
        if (request instanceof EasyPayQsTradeDataBaseRequest) {
            EasyPayQsTradeDataBaseRequest tradeDataBaseRequest = (EasyPayQsTradeDataBaseRequest) request;
            requestDataMap.put("orgMercode", tradeDataBaseRequest.getOrgMercode());
            requestDataMap.put("orgTermno", tradeDataBaseRequest.getOrgTermno());

        }

        // 获取参与加签的参数集合
        TreeMap<String, String> bizParamMap = SignUtil.getSignParameterMap(request);
        String sign = SignUtil.generateSign(bizParamMap, SignTypeEnum.getByValue(signType).getAlgorithm(), fubeiPrivateKey);
        requestDataMap.put("sign", sign);
        requestDataMap.put("signType", signType);
        requestDataMap.put("appendData", request.getAppendData());
        requestDataMap.put("data", bizParamMap);
        return JSONObject.toJSONString(requestDataMap);
    }

    /**
     * 参数校验
     *
     * @param request 请求参数
     * @param method  请求方法
     * @param <R>
     */
    private <R> void validateParam(EasyPayDataBaseRequest request, String method) {
        if (request == null) {
            throw new IllegalArgumentException("接口请求参数不能为空");
        }
        if (StringUtils.isBlank(fubeiPrivateKey)) {
            throw new IllegalArgumentException("私钥不能为空");
        }
        if (StringUtils.isBlank(easyPayPublicKey)) {
            throw new IllegalArgumentException("公钥不能为空");
        }
        if (StringUtils.isBlank(signType)) {
            throw new IllegalArgumentException("签名类型不能为空");
        }
        if (null == SignTypeEnum.getByValue(signType)){
            throw new IllegalArgumentException("签名类型不存在");
        }
        if (StringUtils.isBlank(method)) {
            throw new IllegalArgumentException("请求方法不能为空");
        }
        if (null == EasyPayApiEnum.getByValue(method)) {
            throw new IllegalArgumentException("请求方法不存在");
        }
        //注解验证
        ValidateUtil.validateWithThrow(request);
    }


    /**
     * 验签时是否过滤null值
     * @param signFilterNull
     */
    public void setSignFilterNull(Boolean signFilterNull) {
        this.signFilterNull = signFilterNull;
    }
}