/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.easypay.sdk.util;

import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.asymmetric.Sign;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fshows.easypay.sdk.base.SignFiled;
import com.fshows.easypay.sdk.enums.SignTypeEnum;
import com.fshows.easypay.sdk.exception.EasyPayException;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.lang.reflect.Field;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;

/**
 * @author zhaoxumin
 * @version SignUtil.java, v 0.1 2023-08-25 下午11:14 zhaoxumin
 */
@Slf4j
public class SignUtil {

    private static final String SIGN_ALGORITHMS = "SHA1WithRSA";

    private static PrivateKey getPrivateKeyFromPKCS8(String algorithm, InputStream ins) throws Exception {
        if (ins != null && !StringUtils.isEmpty(algorithm)) {
            KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
            byte[] encodedKey = StreamUtil.readText(ins).getBytes();
            encodedKey = Base64.getDecoder().decode(encodedKey);
            return keyFactory.generatePrivate(new PKCS8EncodedKeySpec(encodedKey));
        } else {
            return null;
        }
    }

    private static String getContent(Map<String, String> param) {
        StringBuilder content = new StringBuilder();
        List<String> keys = new ArrayList<>(param.keySet());
        Collections.sort(keys);
        int index = 0;
        for (String key : keys) {
            String value = param.get(key);
            if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(value)) {
                content.append(index == 0 ? "" : "&").append(key).append("=").append(value);
                index++;
            }
        }
        return content.toString();
    }

    private static String getVerifiedContent(Map<String, Object> params) {
        if (params == null) {
            return null;
        }
        params.remove("sign");
        StringBuilder content = new StringBuilder();
        List<String> keys = new ArrayList<>(params.keySet());
        Collections.sort(keys);

        for (int i = 0; i < keys.size(); i++) {
            String key = keys.get(i);
            String value = params.get(key) == null ? "" : params.get(key).toString();
            content.append(i == 0 ? "" : "&").append(key).append("=").append(value);
        }

        return content.toString();
    }

    public static PublicKey getPublicKeyFromX509(String algorithm,
                                                  InputStream ins) throws Exception {
        KeyFactory keyFactory = KeyFactory.getInstance(algorithm);

        StringWriter writer = new StringWriter();
        StreamUtil.io(new InputStreamReader(ins), writer);

        byte[] encodedKey = writer.toString().getBytes();

        encodedKey = Base64.getDecoder().decode(encodedKey);

        return keyFactory.generatePublic(new X509EncodedKeySpec(encodedKey));
    }

    public static String sign(Map<String, String> data, String signType, String privateKey) {
        try {
            PrivateKey priKey = getPrivateKeyFromPKCS8(signType, new ByteArrayInputStream(privateKey.getBytes()));
            Signature signature = Signature.getInstance(SIGN_ALGORITHMS);
            signature.initSign(priKey);
            String content = getContent(data);
            signature.update(content.getBytes(StringPool.UTF_8));
            byte[] signed = signature.sign();
            return new String(Base64.getEncoder().encode(signed));
        } catch (Exception ex) {
            return null;
        }
    }

    public static String sign(String content, String signType, String privateKey) {
        try {
            PrivateKey priKey = getPrivateKeyFromPKCS8(signType, new ByteArrayInputStream(privateKey.getBytes()));
            Signature signature = Signature.getInstance(SIGN_ALGORITHMS);
            signature.initSign(priKey);
            signature.update(content.getBytes(StringPool.UTF_8));
            byte[] signed = signature.sign();
            return new String(Base64.getEncoder().encode(signed));
        } catch (Exception ex) {
            return null;
        }
    }

    /**
     * 返参验签
     *
     * @param dataMap
     * @param signType
     * @param sign
     * @param publicKey
     * @return
     */
    public static boolean verifySign(Map<String, Object> dataMap, String signType, String sign, String publicKey) {
        try {
            Sign signer = new Sign(SignTypeEnum.getByValue(signType).getAlgorithm(), null, SecureUtil.decode(publicKey));

            TreeMap<String, Object> sortedMap = dataMap.entrySet()
                    .stream()
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (v1, v2) -> v2, TreeMap::new));
            String waitSignStr = getWaitSignStr(sortedMap);

            // 执行加签操作
            return signer.verify(waitSignStr.getBytes(StandardCharsets.UTF_8), Base64.getDecoder().decode(sign));
        } catch (Exception ex) {
            return false;
        }
    }

    /**
     * 获取参数集合
     *
     * @param t
     * @return
     * @param <T>
     * @throws EasyPayException
     */
    public static <T> TreeMap<String, String> getParameterMap(T t) throws EasyPayException {
        Field[] fieldList = t.getClass().getDeclaredFields();

        TreeMap<String, String> paramMap = new TreeMap<>();
        for (Field field : fieldList) {
            String fieldName = field.getName();
            if ("serialVersionUID".equalsIgnoreCase(fieldName)){
                continue;
            }
            field.setAccessible(true);
            try {
                Object o = field.get(t);
                if (o == null) {
                    continue;
                }
                paramMap.put(fieldName, JSONObject.toJSONString(o));
            } catch (IllegalAccessException e) {
                throw new EasyPayException("");
            }
        }
        return paramMap;
    }

    /**
     * 获取参与加签的参数集合
     *
     * @param t
     * @return
     * @param <T>
     * @throws EasyPayException
     */
    public static <T> TreeMap<String, String> getNoSignParameterMap(T t) throws EasyPayException {
        Field[] fieldList = t.getClass().getDeclaredFields();

        TreeMap<String, String> paramMap = new TreeMap<>();
        for (Field field : fieldList) {
            String fieldName = field.getName();
            if ("serialVersionUID".equalsIgnoreCase(fieldName)){
                continue;
            }

            SignFiled signFiledAnnotation = field.getAnnotation(SignFiled.class);
            if (signFiledAnnotation == null || signFiledAnnotation.needJoin()) {
                continue;
            }

            field.setAccessible(true);
            try {
                Object o = field.get(t);
                if (o == null) {
                    continue;
                }
                paramMap.put(fieldName, o instanceof String ? o.toString() : JSONObject.toJSONString(o));
            } catch (IllegalAccessException e) {
                throw new EasyPayException("");
            }
        }
        return paramMap;
    }

    /**
     * 获取参与加签的参数集合
     *
     * @param t
     * @return
     * @param <T>
     * @throws EasyPayException
     */
    public static <T> TreeMap<String, String> getSignParameterMap(T t) throws EasyPayException {
        Field[] fieldList = t.getClass().getDeclaredFields();

        TreeMap<String, String> paramMap = new TreeMap<>();
        for (Field field : fieldList) {
            String fieldName = field.getName();
            if ("serialVersionUID".equalsIgnoreCase(fieldName)){
                continue;
            }

            SignFiled signFiledAnnotation = field.getAnnotation(SignFiled.class);
            if (signFiledAnnotation != null && !signFiledAnnotation.needJoin()) {
                continue;
            }

            field.setAccessible(true);
            try {
                Object o = field.get(t);
                if (o == null) {
                    continue;
                }
                paramMap.put(fieldName, o instanceof String ? o.toString() : JSON.toJSONString(o));
            } catch (IllegalAccessException e) {
                throw new EasyPayException("");
            }
        }
        return paramMap;
    }

    public static TreeMap<String, String> getParameterMap(JSONObject jsonObject) {
        Map<String, Object> map = jsonObject.getInnerMap();
        TreeMap<String, String> paramMap = new TreeMap<>();
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            paramMap.put(entry.getKey(), (String)entry.getValue());
        }
        return paramMap;
    }

    /**
     * 获取签名
     *
     * @param params 请求参数
     * @return sign
     */
    public static String generateSign(TreeMap<String, String> params, String signType, String fubeiPrivateKey) {
        List<String> paramKeyBySort = params.keySet().stream().sorted().collect(Collectors.toList());
        // 转成小写
        Map<String, String> resultMap = mapKey(params);
        StringBuilder paramStrBuilder = new StringBuilder();
        // 拼接字符串参数
        for (String paramKey : paramKeyBySort) {
            // 参数值为空，则不参与签名，空格也过滤掉
            if (StrUtil.isNotBlank(resultMap.get(paramKey))) {
                paramStrBuilder.append(paramKey).append("=").append(resultMap.get(paramKey)).append("&");
            }
        }

        String paramStr = StrUtil.removeSuffix(paramStrBuilder, "&");
        // 创建加签对象
        Sign signer = new Sign(signType, SecureUtil.decode(fubeiPrivateKey), null);
        // 执行加签操作
        byte[] signed = signer.sign(paramStr.getBytes(StandardCharsets.UTF_8));
        return new String(Base64.getEncoder().encode(signed));
    }

    /**
     * 获取签名转大写
     *
     * @param params 请求参数
     * @return sign
     */
    public static String generateMd5Sign(TreeMap<String, String> params, String key) {
        List<String> paramKeyBySort = params.keySet().stream().sorted().collect(Collectors.toList());
        // 转成小写
        Map<String, String> resultMap = mapKey(params);
        StringBuilder paramStr = new StringBuilder();
        // 拼接字符串参数
        for (String paramKey : paramKeyBySort) {
            paramStr.append(paramKey).append("=").append(resultMap.get(paramKey)).append("&");
        }
        paramStr.append("key=").append(key);
        log.info("easypay 待加签字符串: {}", paramStr);
        return SecureUtil.md5(paramStr.toString()).toUpperCase();
    }

    /**
     * 获取参数名称集合
     *
     * @param formParam
     * @return
     */
    private static Map<String, String> mapKey(Map<String, String> formParam) {
        Map<String, String> resultMap = Maps.newHashMap();
        for (String key : formParam.keySet()) {
            resultMap.put(key, formParam.get(key));
        }
        return resultMap;
    }

    /**
     * 获取待加签字符串，格式：addn_inf=&curr_type=&goods_des=卡盟测试&goods_detail=asasda&goods_tag=&ins_cd=08A9999999&
     *
     * @return
     */
    public static String getWaitSignStr(Map<String, Object> dataMap) {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, Object> entry : dataMap.entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();
            if (value == null || StrUtil.isBlank(value.toString())){
                continue;
            }
            if (sb.length() > 0) {
                sb.append("&");
            }
            sb.append(key).append("=").append(value.toString().trim());
        }
        return sb.toString();
    }
}