/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.easypay.sdk.client.impl;

import cn.hutool.core.collection.CollUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.easypay.sdk.client.EasyPayRefundClient;
import com.fshows.easypay.sdk.constant.CommonConstant;
import com.fshows.easypay.sdk.enums.EasyPayApiEnum;
import com.fshows.easypay.sdk.enums.SignTypeEnum;
import com.fshows.easypay.sdk.exception.EasyPayException;
import com.fshows.easypay.sdk.request.EasyPayBizRequest;
import com.fshows.easypay.sdk.response.base.EasyPayRefundBizResponse;
import com.fshows.easypay.sdk.util.FsHttpUtil;
import com.fshows.easypay.sdk.util.LogUtil;
import com.fshows.easypay.sdk.util.SignUtil;
import com.fshows.easypay.sdk.util.ValidateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

import javax.validation.constraints.NotBlank;
import java.util.Map;
import java.util.TreeMap;

/**
 * @author zhaoxumin
 * @version EasyPayBaseClientImpl.java, v 0.1 2023-10-27 13:32
 */
@Slf4j
public class EasyPayRefundClientImpl implements EasyPayRefundClient {

    /**
     * 易生接口网关根URL
     */
    @NotBlank
    protected String apiParentUrl;

    /**
     * 机构号
     */
    @NotBlank
    private final String orgId;

    /**
     * 签名类型
     */
    @NotBlank
    private final String signType;

    /**
     * 用于加签的付呗私钥
     */
    @NotBlank
    private String fubeiPrivateKey;

    /**
     * 用于验签的易生公钥
     */
    @NotBlank
    private String easyPayPublicKey;

    /**
     * http连接超时时间
     */
    protected int connectionTimeout = CommonConstant.DEFAULT_CONNECTION_TIMEOUT;

    /**
     * http response读取超时时间
     */
    protected int readTimeout = CommonConstant.DEFAULT_READ_TIMEOUT;

    private static final SerializeConfig SNAKE_CASE_CONFIG = new SerializeConfig();

    static {
        SNAKE_CASE_CONFIG.propertyNamingStrategy = PropertyNamingStrategy.SnakeCase;
    }

    public EasyPayRefundClientImpl(String apiParentUrl, String orgId, String signType, String fubeiPrivateKey, String easyPayPublicKey) {
        this.apiParentUrl = apiParentUrl;
        this.orgId = orgId;
        this.signType = signType;
        this.fubeiPrivateKey = fubeiPrivateKey;
        this.easyPayPublicKey = easyPayPublicKey;
    }

    /**
     * 请求易生接口
     *
     * @param request              请求参数
     * @param easyPayApiDefinition 方法枚举
     * @param url
     * @param <R>
     * @return
     * @throws EasyPayException
     */
    @Override
    public <R, T> EasyPayRefundBizResponse<T> execute(EasyPayBizRequest<R> request, EasyPayApiDefinition easyPayApiDefinition, String url) throws EasyPayException {
        return execute(request, easyPayApiDefinition, url, null);
    }

    @Override
    public <R, T> EasyPayRefundBizResponse<T> execute(EasyPayBizRequest<R> request, EasyPayApiDefinition definition, String postUrl, String subAppId) throws EasyPayException {
        String method = definition.getMethod();
        //验证参数
        validateParam(request, method);
        // 获取开始时间
        final long startTime = System.currentTimeMillis();
        String url = apiParentUrl + method;
        try {
            // 获取请求参数
            final Map<String, Object> requestData = getRequestData(request, subAppId);
            log.info("【easypay-sdk】接口调用开始 >> url={},request={}", url, requestData);
            String fsHttpResult = FsHttpUtil.postString(url, JSONObject.toJSONString(requestData), CommonConstant.UTF_8, CommonConstant.JSON_CONTENT_TYPE, connectionTimeout, readTimeout, null);
            log.info("【easypay-sdk】接口调用结束 >> url={},request={},response={},cost={}", url, requestData, fsHttpResult, System.currentTimeMillis() - startTime);
            return parseResponse(fsHttpResult, request);
        } catch (Exception ex) {
            log.error("【easypay-sdk】接口调用失败 >> url={},request={},ex={},cost={}", url, request, ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            throw new EasyPayException(ex.getMessage());
        }
    }

//    private boolean checkSign(String fsHttpResult) {
//        JSONObject json = JSON.parseObject(fsHttpResult);
//        if ("000000".equals(json.getString("sysRetcode")) || "000000".equals(json.getString("sysRetCode"))) {
//            String sign = json.getString("sign");
//            if (StringUtils.isEmpty(sign)) {
//                return false;
//            }
//            TreeMap<String, Object> jsonData = json.getObject("data", TreeMap.class);
//            return SignUtil.verifySign(jsonData, signType, sign, fubeiPrivateKey);
//        } else {
//            return true;
//        }
//    }

    private boolean checkSign(String fsHttpResult) {
        JSONObject resMap = JSONObject.parseObject(fsHttpResult);
        if ("000000".equals(resMap.getString("sysRetcode")) || "000000".equals(resMap.getString("sysRetCode"))) {
            String sign = resMap.getString("sign");
            if (StringUtils.isEmpty(sign)) {
                return false;
            }
            // 返参验签
            return CollUtil.isNotEmpty(resMap.getJSONObject("bizData"))
                    && SignUtil.verifySign((Map<String, Object>) resMap.get("bizData"), signType, resMap.get("sign").toString(), easyPayPublicKey);
        } else {
            return true;
        }
    }

    /**
     * 结果解析
     *
     * @param result  返回结果
     * @param request 请求参数
     * @param <R>     结果
     * @return UmBaseResponse
     */
    @SuppressWarnings("unchecked")
    private <R, T> EasyPayRefundBizResponse<T> parseResponse(String result, EasyPayBizRequest<R> request) throws Exception {
        if (!checkSign(result)) {
            LogUtil.error(log, "【easypay-sdk】响应结果验签失败 >> fsHttpResult={}", result);
            throw new EasyPayException("[easypay-sdk]响应结果验签失败");
        }
        EasyPayRefundBizResponse bizResponse = JSON.parseObject(result, EasyPayRefundBizResponse.class);
        bizResponse.setBizData(JSON.parseObject(JSON.toJSONString(bizResponse.getBizData()), request.getResponseClass()));
        return bizResponse;
    }

    /**
     * 获取签名参数
     *
     * @param request 请求参数
     * @param <R>     返参
     * @return Map<String, String>
     */
    private <R> Map<String, Object> getRequestData(EasyPayBizRequest request, String appId) throws Exception {
        TreeMap<String, Object> requestDataMap = new TreeMap<>();
        TreeMap<String, String> signParameterMap = SignUtil.getSignParameterMap(request);
        String sign = SignUtil.generateSign(signParameterMap, SignTypeEnum.getByValue(signType).getAlgorithm(), fubeiPrivateKey);
        TreeMap<String, String> parameterMap = SignUtil.getNoSignParameterMap(request);
        requestDataMap.putAll(parameterMap);
        requestDataMap.put("orgId", StringUtils.defaultIfBlank(appId, this.orgId));
        requestDataMap.put("signType", signType);
        requestDataMap.put("sign", sign);
        requestDataMap.put("bizData", signParameterMap);
        return requestDataMap;
    }

    /**
     * 参数校验
     *
     * @param request 请求参数
     * @param method  请求方法
     * @param <R>
     */
    private <R> void validateParam(EasyPayBizRequest request, String method) {
        if (request == null) {
            throw new IllegalArgumentException("接口请求参数不能为空");
        }
        if (StringUtils.isBlank(fubeiPrivateKey)) {
            throw new IllegalArgumentException("私钥不能为空");
        }
        if (StringUtils.isBlank(easyPayPublicKey)) {
            throw new IllegalArgumentException("公钥不能为空");
        }
        if (StringUtils.isBlank(signType)) {
            throw new IllegalArgumentException("签名类型不能为空");
        }
        if (StringUtils.isBlank(method)) {
            throw new IllegalArgumentException("请求方法不能为空");
        }
        if (null == EasyPayApiEnum.getByValue(method)) {
            throw new IllegalArgumentException("请求方法不存在");
        }
        //注解验证
        ValidateUtil.validateWithThrow(request);
    }
}