package com.fshows.ark.spring.boot.starter.extend.mq.configlistener;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.RandomUtil;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsProducerConfigUpdateModel;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.IProducerConfigUpdateHandler;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.IProducerProxyFactory;
import com.fshows.ark.spring.boot.starter.exception.MQConsumerException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * 生产者配置变更监听器
 * 监听Apollo中阿里云AK/SK配置的变更，并动态更新生产者实例
 * 
 * @author liluqing
 * @version ProducerConfigChangeListener.java, v 0.1 2024-06-14
 */
@Slf4j
@Component
public class ProducerConfigChangeListener {

    private IProducerProxyFactory producerProxyFactory;

    public ProducerConfigChangeListener(IProducerProxyFactory producerProxyFactory) {
        this.producerProxyFactory = producerProxyFactory;
    }

    /**
     * 处理配置变更事件
     * 
     * @param configChangeModels 配置变更模型列表
     */
    public void reloadConfig(List<ConfigChangeModel> configChangeModels) {
        if (configChangeModels == null || configChangeModels.isEmpty()) {
            return;
        }

        log.info("ark-spring-boot-starter >> 开始处理生产者配置变更，变更项数量: {}", configChangeModels.size());
        // 检查是否包含阿里云AK/SK配置变更
        Set<IProducerConfigUpdateHandler> changedProxyHandlers = updateProxyConfig(configChangeModels);

        if (CollectionUtil.isEmpty(changedProxyHandlers)) {
            log.debug("ark-spring-boot-starter >> 本次配置变更不涉及阿里云AK/SK，跳过生产者更新");
            return;
        }

        // 异步处理生产者更新，避免阻塞Apollo配置变更流程
        CompletableFuture.runAsync(() -> {
            try {
                // 随机延迟，避免多个实例同时重启
                int delaySeconds = RandomUtil.randomInt(10);
                log.info("ark-spring-boot-starter >> 将在{}秒后开始更新生产者实例", delaySeconds);
                TimeUnit.SECONDS.sleep(delaySeconds);

                updateProducersForAkSkChange(changedProxyHandlers);
            } catch (Exception e) {
                log.error("ark-spring-boot-starter >> 更新生产者实例时发生异常", e);
            }
        });
    }

    /**
     * 更新代理对象中的动态配置
     *
     * @param configChangeModels 配置变更模型列表
     * @return 受影响的代理Handler集合
     */
    private Set<IProducerConfigUpdateHandler> updateProxyConfig(List<ConfigChangeModel> configChangeModels) {
        Set<IProducerConfigUpdateHandler> allAffectedHandlers = ConcurrentHashMap.newKeySet();

        // 动态key和代理对象的映射关系
        Map<String, Set<FsProducerConfigUpdateModel>> producerConfigUpdateModelMap = producerProxyFactory.getFsProducerConfigUpdateModelMap();

        for (ConfigChangeModel config : configChangeModels) {
            Set<FsProducerConfigUpdateModel> updateModels = producerConfigUpdateModelMap.get(config.getKey());
            if (CollectionUtil.isEmpty(updateModels)) {
                continue;
            }
            for (FsProducerConfigUpdateModel model : updateModels) {
                Field f = model.getField();
                String oldValue = getFieldValueStr(model.getTarget(), f);
                if (oldValue != null && oldValue.equals(config.getNewValue())) {
                    continue;
                }
                boolean success = setFieldValueStr(config.getNewValue(), model.getTarget(), f);
                if (success) {
                    allAffectedHandlers.add(model.getConfigUpdateHandler());
                }
            }
        }
        return allAffectedHandlers;
    }

    /**
     * 更新受AK/SK配置变更影响的生产者实例
     * 
     * @param producerConfigUpdateHandlerSet 配置变更模型列表
     */
    private void updateProducersForAkSkChange(Set<IProducerConfigUpdateHandler> producerConfigUpdateHandlerSet) {
        log.info("ark-spring-boot-starter >> 开始更新受AK/SK配置变更影响的生产者实例");

        if (producerConfigUpdateHandlerSet.isEmpty()) {
            log.info("ark-spring-boot-starter >> 未找到受影响的生产者代理Handler");
            return;
        }

        log.info("ark-spring-boot-starter >> 找到{}个受影响的生产者代理Handler，开始更新", producerConfigUpdateHandlerSet.size());

        int successCount = 0;
        int failureCount = 0;

        // 逐个更新代理Handler中的生产者实例
        for (IProducerConfigUpdateHandler handler : producerConfigUpdateHandlerSet) {
            try {
                // 更新代理Handler中的生产者实例
                handler.restart();
                successCount++;
                log.info("ark-spring-boot-starter >> 成功更新生产者实例");
            } catch (Exception e) {
                failureCount++;
                log.error("ark-spring-boot-starter >> 更新生产者实例时发生异常", e);
            }
        }
        // 清理生产者实例构建缓存
        producerProxyFactory.clearFshowsProducerCache();

        log.info("ark-spring-boot-starter >> 生产者实例更新完成，成功: {}, 失败: {}", successCount, failureCount);
    }

    /**
     * 根据反射获取对象值
     *
     * @param target
     * @param field
     * @return
     */
    public boolean setFieldValueStr(String value, Object target, Field field) {
        try {
            field.setAccessible(true);
            field.set(target, value);
            return true;
        } catch (IllegalAccessException e) {
            log.error("ark-spring-boot-starter >> mq消费者配置变更监听器异常 >> 动态加载变更配置异常 >> value={}", value);
        }
        return false;
    }


    /**
     * 根据反射获取对象值
     *
     * @param target
     * @param field
     * @return
     */
    public String getFieldValueStr(Object target, Field field) {
        try {
            field.setAccessible(true);
            Object value = field.get(target);
            return value == null ? null : value.toString();
        } catch (IllegalAccessException e) {
            log.error("ark-spring-boot-starter >> mq消费者配置变更监听器加载异常!", e);
            throw new MQConsumerException("ark-spring-boot-starter >> mq消费者配置变更监听器加载异常！");
        }
    }
}
