/**
 * fshows.com
 * Copyright (C) 2013-2021 All Rights Reserved.
 */
package
        com.fshows.ark.spring.boot.starter.core.mq.rocketmq.producer;

import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.aliyun.openservices.ons.api.bean.ProducerBean;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsProducerModel;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.IFshowsProducer;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.TransactionMessageManage;
import com.fshows.ark.spring.boot.starter.core.mq.rocketmq.interceptor.IProducerInterceptorManagement;
import com.fshows.ark.spring.boot.starter.exception.MQConsumerException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;

import java.lang.reflect.Proxy;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Rocket生产者代理对象工厂
 *
 * @author liluqing
 * @version RocketProducerProxyFactory.java, v 0.1 2021-08-19 15:04
 */
@Slf4j
public class DefaultProducerProxyFactoryImpl extends AbstractProducerProxyFactory {

    /**
     * 拦截器管理器
     */
    private IProducerInterceptorManagement producerInterceptorManagement;

    /**
     * 缓存的阿里云的生产者bean
     */
    private Map<String, ProducerBean> aliyunProducerBeanMap = null;

    /**
     * 构建生产者动态代理处理器,进行消息发送处理
     *
     * @param fshowsProducer
     * @param producerModel
     * @return
     */
    public <T> T buildProducerProxy(IFshowsProducer fshowsProducer, TransactionMessageManage transactionMessageManage, FsProducerModel producerModel, Class<T> producerInterface) {
        RocketProducerProxyHandler producerProxyHandler = new RocketProducerProxyHandler(fshowsProducer, producerModel, transactionMessageManage, this);
        // 创建代理对象
        T instance = (T) Proxy.newProxyInstance(producerInterface.getClassLoader(), new Class[]{producerInterface}, producerProxyHandler);
        return instance;
    }

    /**
     * 创建fs生产者bean对象
     *
     * @param fsProducerModel
     * @return
     */
    public IFshowsProducer createIFshowsProducer(FsProducerModel fsProducerModel, TransactionMessageManage transactionMessageManage, boolean isNewProducer) {
        // 阿里云producerBean缓存是否存在,如不存在则从上下文获取所有用户自定义的阿里云producerBean
        if (aliyunProducerBeanMap == null) {
            aliyunProducerBeanMap = getProducerBeanMapForCtx(applicationContext);
        }
        ProducerBean producerBean = aliyunProducerBeanMap.get(fsProducerModel.getGroupId());
        // 如不存在用户自定义的阿里云producerBean,则自己创建
        if (producerBean == null || isNewProducer) {
            producerBean = createProducerBean(fsProducerModel);
        }
        // 启动生产者实例
        producerBean.start();
        log.info("ark-spring-boot-starter >> createIFshowsProducer >> [实际]生产者bean创建成功 >> groupId={}", fsProducerModel.getGroupId());
        // 构建并缓存本次获取到的Producer
        IFshowsProducer iFshowsProducer = new AliyunFshowsProducerImpl(producerBean, fsProducerModel.getCharsetName(), producerInterceptorManagement.getProducerInterceptorActuator(), transactionMessageManage);
        return iFshowsProducer;
    }

    /**
     * 创建fs生产者bean对象
     *
     * @param fsProducerModel
     * @return
     */
    public IFshowsProducer createSimpleFshowsProducer(FsProducerModel fsProducerModel) {
        // 阿里云producerBean缓存是否存在,如不存在则从上下文获取所有用户自定义的阿里云producerBean
        if (aliyunProducerBeanMap == null) {
            aliyunProducerBeanMap = getProducerBeanMapForCtx(applicationContext);
        }
        ProducerBean producerBean = aliyunProducerBeanMap.get(fsProducerModel.getGroupId());
        // 如不存在用户自定义的阿里云producerBean,则自己创建
        if (producerBean == null) {
            producerBean = createProducerBean(fsProducerModel);
        }
        // 启动生产者实例
        producerBean.start();
        // 构建并缓存本次获取到的Producer
        IFshowsProducer iFshowsProducer = new SimpleFshowsProducerImpl(producerBean, fsProducerModel.getCharsetName());
        return iFshowsProducer;
    }

    /**
     * 创建生成者bean
     *
     * @return
     */
    private ProducerBean createProducerBean(FsProducerModel fsProducerModel) {
        ProducerBean producerBean = new ProducerBean();
        Properties properties = new Properties();
        properties.setProperty(PropertyKeyConst.GROUP_ID, fsProducerModel.getGroupId());
        properties.setProperty(PropertyKeyConst.AccessKey, fsProducerModel.getAccessKey());
        properties.setProperty(PropertyKeyConst.SecretKey, fsProducerModel.getSecretKey());
        properties.setProperty(PropertyKeyConst.NAMESRV_ADDR, fsProducerModel.getNamesrvAddr());
        properties.setProperty(PropertyKeyConst.SendMsgTimeoutMillis, fsProducerModel.getSendMsgTimeoutMillis());
        producerBean.setProperties(properties);
        return producerBean;
    }

    /**
     * 从spring上下文中获取用户自定义的消费者bean
     * key为消费者实例的groupId
     * value为ProducerBean
     *
     * @return
     */
    private Map<String, ProducerBean> getProducerBeanMapForCtx(ApplicationContext ctx) {
        Map<String, ProducerBean> producerBeanMap = ctx.getBeansOfType(ProducerBean.class);
        if (producerBeanMap == null || producerBeanMap.size() == 0) {
            return new ConcurrentHashMap<>();
        }
        Map<String, ProducerBean> groupProducerBeanMap = new ConcurrentHashMap<>(producerBeanMap.size());
        for (Map.Entry<String, ProducerBean> entry : producerBeanMap.entrySet()) {
            Properties properties = entry.getValue().getProperties();
            if (properties == null) {
                log.error("ark-spring-boot-starter >> 自定义producerBean的参数未设置! beanName={}", entry.getKey());
                throw new MQConsumerException("ark-spring-boot-starter >> rocketmq生产者者创建失败！");
            }
            String key = properties.getProperty(PropertyKeyConst.GROUP_ID) == null
                    ? properties.getProperty(PropertyKeyConst.ProducerId) : properties.getProperty(PropertyKeyConst.GROUP_ID);
            groupProducerBeanMap.put(key, entry.getValue());
        }
        return groupProducerBeanMap;
    }

    public void setProducerInterceptorManagement(IProducerInterceptorManagement producerInterceptorManagement) {
        this.producerInterceptorManagement = producerInterceptorManagement;
    }
}