/**
 * fshows.com
 * Copyright (C) 2013-2024 All Rights Reserved.
 */
package
        com.fshows.ark.spring.boot.starter.core.sensitive.encrypt;

import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.digest.DigestAlgorithm;
import cn.hutool.crypto.digest.Digester;
import com.alibaba.fastjson.JSONObject;
import com.fshows.ark.spring.boot.starter.constant.CommonConstant;
import com.fshows.ark.spring.boot.starter.core.sensitive.enums.AlgorithmTypeEnum;
import com.fshows.ark.spring.boot.starter.core.sensitive.model.SecretKeyModel;
import com.fshows.ark.spring.boot.starter.core.sensitive.model.SensitiveConfigItem;
import com.fshows.ark.spring.boot.starter.core.sensitive.model.SensitiveConfigModel;
import com.fshows.ark.spring.boot.starter.exception.DbSensitiveException;
import com.fshows.ark.spring.boot.starter.util.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 *
 * @author liluqing
 * @version DefaultSecretKeyManage.java, v 0.1 2024-05-28 13:34
 */
@Slf4j
@Component
public class DefaultSecretKeyManage implements SecretKeyManage {

    /**
     * 密钥缓存
     * key: keyId+AlgorithmType
     * value
     */
    private final Map<String, SecretKeyModel> KEY_CACHE_MAP = new ConcurrentHashMap<>();

    /**
     * 密钥缓存
     * key: NAME
     * value
     */
    private final Map<String, SecretKeyModel> NAME_KEY_CACHE_MAP = new ConcurrentHashMap<>();

    /**
     * 密钥配置
     * 示例：
     * {
     *     "mainSecretName":"FS_SM4",
     *     "searchKeywordsSecretName": "FS_MD5",
     *     "secretList": [
     *         {
     *             "secretName": "FS_SM4",
     *             "keyId": "1",
     *             "algorithmType": "SM4",
     *             "seed": "123456"
     *         },
     *         {
     *             "secretName": "FS_MD5",
     *             "keyId": "2",
     *             "algorithmType": "MD5"
     *         }
     *     ]
     * }
     */
    @Value("${ark.sensitive.secrekey.config:}")
    private String sensitiveConfig;
    /**
     * 用于加密的主密钥
     */
    private volatile SecretKeyModel mainEncryptSecretKey;

    /**
     * 数据库检索字段主密钥
     */
    private volatile SecretKeyModel searchKeywordsSecret;

    /**
     * 初始化密钥缓存
     */
    @PostConstruct
    public void initMap() {
        if (StrUtil.isBlank(sensitiveConfig)) {
            LogUtil.error(log, "ark-spring-boot-starter >>  数据库敏感字段密钥配置为空 >> 请检查‘ark.sensitive.secrekey.config’配置");
            throw new DbSensitiveException("ark-spring-boot-starter >> 数据库敏感字段密钥配置为空 >> 请检查‘ark.sensitive.secrekey.config’配置");
        }
        SensitiveConfigModel sensitiveConfigModel = JSONObject.parseObject(sensitiveConfig, SensitiveConfigModel.class);
        List<SensitiveConfigItem> configList = sensitiveConfigModel.getSecretList();
        for (SensitiveConfigItem config : configList) {
            AlgorithmTypeEnum algorithmType = AlgorithmTypeEnum.getByName(config.getAlgorithmType());
            if (algorithmType == null) {
                LogUtil.error(log, "ark-spring-boot-starter >>  数据库敏感字段加密配置初始化异常 >> 未识别的algorithmType,请检查‘ark.sensitive.secrekey.config’配置是否正确 >> {}", config.getAlgorithmType());
                throw new DbSensitiveException("ark-spring-boot-starter >> 数据库敏感字段加密配置初始化异常！ >> 未识别的algorithmType");
            }
            if (StrUtil.isBlank(config.getKeyId())) {
                LogUtil.error(log, "ark-spring-boot-starter >>  数据库敏感字段加密配置初始化异常 >> keyId不能为空,请检查‘ark.sensitive.secrekey.config’配置是否正确 >> {}", config.getKeyId());
                throw new DbSensitiveException("ark-spring-boot-starter >> 数据库敏感字段加密配置初始化异常！ >> keyId不能为空");
            }
            int keyId = Integer.parseInt(config.getKeyId());
            if (keyId < 0 || keyId > 65535) {
                LogUtil.error(log, "ark-spring-boot-starter >>  数据库敏感字段加密配置初始化异常 >> 无效的keyId,keyId的取值范围为（0-65535的数字）,请检查‘ark.sensitive.secrekey.config’配置是否正确 >> {}", config.getKeyId());
                throw new DbSensitiveException("ark-spring-boot-starter >> 数据库敏感字段加密配置初始化异常！ >> keyId不能为空");
            }
            if (StrUtil.isBlank(config.getSeed())) {
                LogUtil.error(log, "ark-spring-boot-starter >>  数据库敏感字段加密配置初始化异常 >> seed不能为空,请检查‘ark.sensitive.secrekey.config’配置是否正确 >> {}", config.getSeed());
                throw new DbSensitiveException("ark-spring-boot-starter >> 数据库敏感字段加密配置初始化异常！ >> seed不能为空");
            }

            SecretKeyModel keyModel = new SecretKeyModel();
            keyModel.setSecretName(config.getSecretName());
            keyModel.setKeyId(keyId);
            keyModel.setAlgorithmType(algorithmType);

            // 根据种子生成实际的密钥 MD5(MD5(seed)+seed)
            Digester md5 = new Digester(DigestAlgorithm.MD5);
            String digestHex = md5.digestHex(config.getSeed());
            byte[] privateKey = md5.digest(digestHex+config.getSeed());
            // 设置实际密钥
            keyModel.setPrivateKey(privateKey);
            // 生成缓存key
            String key = StrUtil.format(CommonConstant.SECRETKEY_CACHE_MAP_KEY, config.getKeyId(), algorithmType.getValue());
            KEY_CACHE_MAP.put(key, keyModel);
            if (StrUtil.isNotBlank(keyModel.getSecretName())) {
                NAME_KEY_CACHE_MAP.put(keyModel.getSecretName(), keyModel);
            }
        }
        // 用于加密的主密钥
        mainEncryptSecretKey = NAME_KEY_CACHE_MAP.get(sensitiveConfigModel.getMainSecretName());
        // 用于做字段检索的主密钥
        searchKeywordsSecret = NAME_KEY_CACHE_MAP.get(sensitiveConfigModel.getSearchKeywordsSecretName());
        if (mainEncryptSecretKey == null) {
            LogUtil.error(log, "ark-spring-boot-starter >>  数据库敏感字段加密配置初始化异常 >> 未设置加密主密钥,请检查‘ark.sensitive.secrekey.config’配置中“mainEncryptSecretKey”是否正确 >> {}", sensitiveConfigModel.getMainSecretName());
            throw new DbSensitiveException("ark-spring-boot-starter >> 数据库敏感字段加密配置初始化异常！ >> 未设置加密主密钥");
        }
        if (searchKeywordsSecret == null) {
            LogUtil.error(log, "ark-spring-boot-starter >>  数据库敏感字段加密配置初始化异常 >> 未设置检索查询字段加密主密钥,请检查‘ark.sensitive.secrekey.config’配置中“searchKeywordsSecretName”是否正确 >> {}", sensitiveConfigModel.getSearchKeywordsSecretName());
            throw new DbSensitiveException("ark-spring-boot-starter >> 数据库敏感字段加密配置初始化异常！ >> 未设置检索查询字段加密主密钥");
        }
    }



    @Override
    public SecretKeyModel getSecretKey(String keyId, AlgorithmTypeEnum algorithmType) {
        if (StrUtil.isBlank(sensitiveConfig)) {
            return null;
        }
        // 生成缓存key
        String key = StrUtil.format(CommonConstant.SECRETKEY_CACHE_MAP_KEY, keyId, algorithmType.getValue());
        return KEY_CACHE_MAP.get(key);
    }

    @Override
    public SecretKeyModel getSecretKeyByName(String secretName) {
        if (StrUtil.isBlank(sensitiveConfig)) {
            return null;
        }
        return NAME_KEY_CACHE_MAP.get(secretName);
    }

    @Override
    public SecretKeyModel getMainEncryptSecretKey() {
        return mainEncryptSecretKey;
    }

    @Override
    public SecretKeyModel getSearchKeywordsSecret() {
        if (StrUtil.isBlank(sensitiveConfig)) {
            return null;
        }
        return searchKeywordsSecret;
    }
}