/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.umpay.sdk.request.merchant;

import com.fshows.umpay.sdk.request.UmBizRequest;
import com.fshows.umpay.sdk.response.merchant.UmpayMerchantSignResponse;
import lombok.Data;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.NotBlank;

/**
 * 支付平台签约
 * 针对在<商户入驻>时只入驻了某一支付平台的商户，例如只入驻了微信，可调用本接口帮助商户开通支付宝的支付业务；
 *
 * @author
 * @version UmpayMerchantSignRequest.java, v 0.1 2022-03-16 17:02:33
 */
@Data
public class UmpayMerchantSignRequest extends UmBizRequest<UmpayMerchantSignResponse> {

    private static final long serialVersionUID = -2213838555753606068L;


    /**
     * 聚合收单平台为商户分配的唯一id,商户入驻成功后由平台返回
     * <p>
     * (必填)
     */
    @NotBlank
    private String storeId;

    /**
     * 签约平台类型：1-支付宝；2-微信
     * <p>
     * (特殊必填)
     */
    private Integer allowType;

    /**
     * 是否签约云闪付，默认值为0；0-不签约；1-签约；若allow_type不传，则open_union_pay必须为1-签约；
     * <p>
     * (特殊必填)
     */
    private Integer openUnionPay;

    /**
     * 商户全称若商户类型为1-小微商户，商户名称应填“商户_XXX”（XXX为小微商户身份证上的姓名）示例：商户_张三若商户类型为2-个体户及3-企事业，商户名称应填写商户营业执照上的名称示例：张三的水果店个体工商户营业执照没有商家名称或者名称为“*”或者“***”，则商户名称应填 “个体户XXX”（XXX为营业执照上经营者姓名）示例：个体户张三
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 50, message = "merchantName长度不能超过50")
    private String merchantName;

    /**
     * 经营行业类目，请填写对应的ID
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 128, message = "categoryId长度不能超过128")
    private String categoryId;

    /**
     * 商户简称,在支付时展示给消费者
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 20, message = "merchantShortname长度不能超过20")
    private String merchantShortname;

    /**
     * 客服电话，方便微信、支付宝在必要时能联系上商家，会在支付详情展示给消费者
     * <p>
     * (必填)
     */
    @NotBlank
    @Length(max = 20, message = "servicePhone长度不能超过20")
    private String servicePhone;

    /**
     * 微信账户类型: 0-当面付账户（不传默认0），微信签约时必填
     * <p>
     * (特殊必填)
     */
    private Integer accountType;

    /**
     * 支付类型：3-微信扫码支付、微信刷卡支付、公众号支付，微信签约时必填
     * <p>
     * (特殊必填)
     */
    private Integer payType;

    /**
     * 微信渠道号，商户签约微信且服务商有多个渠道号时必填
     * <p>
     * (特殊必填)
     */
    @Length(max = 32, message = "channelId长度不能超过32")
    private String channelId;

    /**
     * 联系人，方便微信在必要时能联系上商家示例：张三
     * <p>
     * (选填)
     */
    @Length(max = 10, message = "contact长度不能超过10")
    private String contact;

    /**
     * 联系电话，方便微信在必要时能联系上商家
     * <p>
     * (选填)
     */
    @Length(max = 11, message = "contactPhone长度不能超过11")
    private String contactPhone;

    /**
     * 联系邮箱，方便微信在必要时能联系上商家
     * <p>
     * (选填)
     */
    @Length(max = 30, message = "contactEmail长度不能超过30")
    private String contactEmail;

    /**
     * 联系人微信账号类型，如传微信号，值为type_wechatid示例：type_wechatid
     * <p>
     * (选填)
     */
    @Length(max = 32, message = "contactWechatidType长度不能超过32")
    private String contactWechatidType;

    /**
     * 联系人微信账号：打开微信，在"个人信息"中查看到的"微信号"示例：OPENID_012312321
     * <p>
     * (选填)
     */
    @Length(max = 32, message = "contactWechatid长度不能超过32")
    private String contactWechatid;

    /**
     * 支付宝渠道号，商户签约支付宝且服务商有多个渠道号时必填
     * <p>
     * (特殊必填)
     */
    @Length(max = 16, message = "pid长度不能超过16")
    private String pid;

    /**
     * 联系人名字若上传了card_no、 business_license、 business_license_type、contact_type，其中一个参数，则contact_name必传
     * <p>
     * (选填)
     */
    @Length(max = 64, message = "contactName长度不能超过64")
    private String contactName;

    /**
     * 联系人手机号
     * <p>
     * (选填)
     */
    @Length(max = 64, message = "contactMobile长度不能超过64")
    private String contactMobile;


    /**
     * 商户备注，可填写额外信息
     * <p>
     * (选填)
     */
    @Length(max = 512, message = "memo长度不能超过512")
    private String memo;

    /**
     * 法人姓名
     * <p>
     * (选填)
     */
    @Length(max = 64, message = "idCardName长度不能超过64")
    private String idCardName;

    /**
     * 法人身份证号码
     * <p>
     * (选填)
     */
    @Length(max = 18, message = "idCardNum长度不能超过18")
    private String idCardNum;

    /**
     * 商户店铺地址(同营业执照地址)
     * <p>
     * (选填)
     */
    @Length(max = 255, message = "storeAddress长度不能超过255")
    private String storeAddress;

    /**
     * 省级编码，数据字典请联系我方对接人员
     * <p>
     * (选填)
     */
    @Length(max = 20, message = "province长度不能超过20")
    private String province;

    /**
     * 市级编码
     * <p>
     * (选填)
     */
    @Length(max = 20, message = "city长度不能超过20")
    private String city;

    /**
     * 区级编码
     * <p>
     * (选填)
     */
    @Length(max = 20, message = "district长度不能超过20")
    private String district;

    /**
     * 结算银行卡号若上传了card_no、business_license、 business_license_type、contact_type，其中一个参数，则四个参数都为必传
     * <p>
     * (选填)
     */
    @Length(max = 32, message = "cardNo长度不能超过32")
    private String cardNo;

    /**
     * 结算人姓名
     * <p>
     * (选填)
     */
    @Length(max = 64, message = "cardName长度不能超过64")
    private String cardName;

    /**
     * 营业执照编号（企业或者个体工商户提供营业执照编号，事业单位提供事证号）若 上 传 了 card_no、business_license、 business_license_type、contact_type，其中一个参数，则四个参数必传
     * <p>
     * (选填)
     */
    @Length(max = 20, message = "businessLicense长度不能超过20")
    private String businessLicense;

    /**
     * 商户证件类型，取值范围： NATIONAL_LEGAL：(营业执照)； NATIONAL_LEGAL_MERGE：(营业执照(多证合一)；INST_RGST_CTF： (事业单位法人证书)示例：NATIONAL_LEGAL若上传了card_no、business_license、 business_license_type、contact_type，其中一个参数，则四个参数必传
     * <p>
     * (选填)
     */
    @Length(max = 20, message = "businessLicenseType长度不能超过20")
    private String businessLicenseType;

    /**
     * 商户联系人类型，取值范围：LEGAL_PERSON：(法人)；CONTROLLER：(实际控制人)；AGENT：(代理人)；OTHER：(其他 )；示例：LEGAL_PERSON若 上 传 了 card_no， business_license， business_license_type，contact_type，其中一个参数，则四个参数必传
     * <p>
     * (选填)
     */
    @Length(max = 32, message = "contactType长度不能超过32")
    private String contactType;

    /**
     * 开户支行联行号（商户需签约云闪付时必填）
     * <p>
     * (特殊必填)
     */
    @Length(max = 64, message = "unitedBankNo长度不能超过64")
    private String unitedBankNo;

    /**
     * 银联商户经营名称签约银联云闪付时必填
     * <p>
     * (选填)
     */
    @Length(max = 40, message = "unionAliasName长度不能超过40")
    private String unionAliasName;

    /**
     * 微信特殊行业优惠费率：1-公共缴费（0.1%）；2-公益（学校、医院、基金会）（0）；3-保险（0.38%）；4-教培行业（0.6%）；5-K12（0.3%）；
     * <p>
     * (选填)
     */
    private Integer wxSpecialActivity;

    /**
     * 经度，浮点型, 小数点后最多保留6位。如需要录入经纬度，请以高德坐标系为准，录入时请确保经纬度参数准确。高德经纬度查询 ：http://lbs.amap.com/console/show/picker
     * <p>
     * (选填)
     */
    @Length(max = 6, message = "longitude长度不能超过6")
    private String longitude;

    /**
     * 纬度，浮点型,小数点后最多保留 6 位 如需要录入经纬度，请以高德 坐标系为准，录入时请确保经 纬 度 参 数 准 确 。 高 德 经 纬 度 查 询 ：http://lbs.amap.com/console/show/picker
     * <p>
     * (选填)
     */
    @Length(max = 6, message = "latitude长度不能超过6")
    private String latitude;

    @Override
    public Class<UmpayMerchantSignResponse> getResponseClass() {
        return UmpayMerchantSignResponse.class;
    }
}