/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.fuiou.client.impl;

import cn.hutool.crypto.digest.MD5;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.fshows.fuiou.client.base.BankPayApiClientConfig;
import com.fshows.fuiou.client.base.FuiouHttpResult;
import com.fshows.fuiou.client.base.IFuiouBankPayApiDefinition;
import com.fshows.fuiou.exception.FuiouApiException;
import com.fshows.fuiou.request.base.FuiouBankPayBizRequest;
import com.fshows.fuiou.response.base.FuiouBankPayBizResponse;
import com.fshows.fuiou.util.FsHttpUtil;
import com.fshows.fuiou.util.FuiouRequestUtils;
import com.fshows.fuiou.util.LogUtil;
import com.fshows.fuiou.util.ValidateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.net.URLDecoder;
import java.util.HashMap;
import java.util.Map;

/**
 * 银行卡支付富友交易接口客户端
 *
 * @author zhaoxumin
 * @version BankPayTradeFuiouApiClientImpl.java, v 0.1 2023-10-13 14:02
 */
@Slf4j
public class BankPayTradeFuiouApiClientImpl {

    /**
     * 客户端配置
     */
    private BankPayApiClientConfig bankPayApiClientConfig;

    public BankPayTradeFuiouApiClientImpl(BankPayApiClientConfig bankPayApiClientConfig) throws FuiouApiException {
        ValidateUtil.validateWithThrow(bankPayApiClientConfig);
        this.bankPayApiClientConfig = bankPayApiClientConfig;
    }

    /**
     * 执行富友请求
     *
     * @param bizRequest
     * @param tradeApiDefinitionEnum
     * @return
     * @throws FuiouApiException
     */
    protected FuiouBankPayBizResponse doExecute(FuiouBankPayBizRequest bizRequest, IFuiouBankPayApiDefinition tradeApiDefinitionEnum) throws FuiouApiException {
        try {
            // 参数校验
            checkParam(bizRequest, tradeApiDefinitionEnum);
            // 补充全局参数
            buildFuiouBizRequest(bizRequest);
            // 执行加签操作获取签名
            String url = getServerURL(tradeApiDefinitionEnum);
            // 执行HTTP post请求
            FuiouHttpResult httpResult = requestPost(url, JSONObject.toJSONString(bizRequest));
            // 反序列化响应结果
            FuiouBankPayBizResponse response = parseResponse(httpResult.getBody(), tradeApiDefinitionEnum);
            LogUtil.info(log, "【fuiou-sdk】响应结果映射结束 >> url={}, request={}, response={}", url, bizRequest, response);
            return response;
        } catch (FuiouApiException e) {
            LogUtil.error(log, "【fuiou-sdk】富友请求异常 >> tradeApiDefinition={}, bizRequest={}", e, tradeApiDefinitionEnum, bizRequest);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk】富友请求异常 >> tradeApiDefinition={}, bizRequest={}", e, tradeApiDefinitionEnum, bizRequest);
            throw new FuiouApiException(e.getMessage(), e);
        }
    }

    /**
     * 执行post请求
     *
     * @param url
     * @param requestBody
     * @return
     */
    protected FuiouHttpResult requestPost(String url, String requestBody) throws IOException, FuiouApiException {
        long beginTime = System.currentTimeMillis();
        try {
            LogUtil.info(log, "【fuiou-sdk】请求开始 >> url={}, request={}", url, requestBody);

            String httpResult = FsHttpUtil.postString(
                    url,
                    requestBody,
                    bankPayApiClientConfig.getCharset(),
                    null,
                    bankPayApiClientConfig.getConnectionTimeout(),
                    bankPayApiClientConfig.getReadTimeout(),
                    null);

            if (StringUtils.isBlank(httpResult)) {
                throw new FuiouApiException("fuiou响应结果为空");
            }
            FuiouHttpResult fuiouHttpResult = new FuiouHttpResult();
            fuiouHttpResult.setBody(URLDecoder.decode(httpResult, bankPayApiClientConfig.getCharset()));

            LogUtil.info(log, "【fuiou-sdk】请求结束 >> url={}, request={}, response={}, cost={}ms", url, requestBody,
                    fuiouHttpResult.getBody(), System.currentTimeMillis() - beginTime);
            return fuiouHttpResult;
        } catch (FuiouApiException e) {
            LogUtil.error(log, "【fuiou-sdk】网络请求异常 >> url={}, request={}, cost={}ms",
                    e, url, requestBody, (System.currentTimeMillis() - beginTime));
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk】网络请求异常 >> url={}, request={}, cost={}ms",
                    e, url, requestBody, (System.currentTimeMillis() - beginTime));
            throw new FuiouApiException("请求fuiou接口异常", e);
        }
    }

    /**
     * 请求参数校验
     *
     * @param bizRequest
     * @return
     */
    protected void checkParam(FuiouBankPayBizRequest bizRequest, IFuiouBankPayApiDefinition fuiouApiDefinition) throws FuiouApiException {
        ValidateUtil.notNull(bizRequest, "request请求参数不能为空");
        boolean checkRequestClass = StringUtils.equals(bizRequest.getClass().getCanonicalName(), fuiouApiDefinition.getRequestClass().getCanonicalName());
        if (!checkRequestClass) {
            throw new FuiouApiException("请求参数类型不正确");
        }
        // 当校验开关打开时才开启主动前置参数校验
        if (this.bankPayApiClientConfig.isCheckParam()) {
            ValidateUtil.validateWithThrow(bizRequest);
        }
    }

    /**
     * 获取请求地址
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected String getServerURL(IFuiouBankPayApiDefinition tradeApiDefinitionEnum) {
        return bankPayApiClientConfig.getApiParentURL() + tradeApiDefinitionEnum.getApiSubURI();
    }

    /**
     * 构建完整的请求参数
     *
     * @param bizRequest 业务请求参数
     * @return
     */
    protected FuiouBankPayBizRequest buildFuiouBizRequest(
            FuiouBankPayBizRequest bizRequest){
        // 补充参数
        bizRequest.setRelateInsCd(bankPayApiClientConfig.getRelateInsCd());
        String signature = MD5.create().digestHex(bizRequest.getInOrderNo() + bankPayApiClientConfig.getSignatureKey());
        bizRequest.setSignature(signature);
        return bizRequest;
    }

    /**
     * 解析响应参数
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected FuiouBankPayBizResponse parseResponse(
            String body,
            IFuiouBankPayApiDefinition tradeApiDefinitionEnum) throws FuiouApiException{
        try {
            Map<String, String> resMap = JSONObject.parseObject(body, new TypeReference<Map<String, String>>(){});
            FuiouBankPayBizResponse bizResponse = FuiouRequestUtils.getObjectFromMap(resMap, tradeApiDefinitionEnum.getResponseClass());
            bizResponse.setResMap(resMap);
            return bizResponse;
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk】响应结果反序列化异常 >> resBody={}", e, body);
            throw new FuiouApiException("[fuiou-sdk]响应结果反序列化异常");
        }
    }

    public <T extends FuiouBankPayBizResponse, R extends IFuiouBankPayApiDefinition> T execute(FuiouBankPayBizRequest<T, R> request, R tradeApiDefinitionEnum) throws FuiouApiException {
        return (T) doExecute(request, tradeApiDefinitionEnum);
    }
}