/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package
        com.fshows.fsframework.extend.lock;

import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.extend.lock.exception.RedisLockException;
import com.fshows.fsframework.extend.redis.RedisCache;
import lombok.extern.slf4j.Slf4j;

import java.text.MessageFormat;
import java.util.concurrent.TimeUnit;

/**
 * 基于redis的分布式锁服务实现
 *
 * @author Liluqing
 * @version RedisDistributedLockServiceImpl.java, v 0.1 2018-10-18 11:12
 */
@Slf4j
public class RedisDistributedLock implements DistributedLock {

    /**
     * 默认redis锁最大存活时间
     */
    private static final  long DEFAULT_LEASE_TIME = 15000;

    /**
     * 默认获取锁最大等待时间
     */
    private static final  long DEFAULT_WAIT_TIME = 2000;

    private static final  String DEFAULT_ERROR_LOG = "【分布式锁服务】tryLock-获取分布式锁失败！ 耗时 = {} ,lockKey = {}";
    private RedisCache redisCache;

    public RedisDistributedLock(RedisCache redisCache) {
        this.redisCache = redisCache;
    }

    /**
     * 获取锁，阻塞
     *
     * @param lockKey 锁的key
     */
    @Override
    public void lock(String lockKey) {
        long currentTime = System.currentTimeMillis();
        try {
            redisCache.getRedisLock(lockKey).lock(DEFAULT_LEASE_TIME, TimeUnit.MILLISECONDS);
            LogUtil.info(log, "【分布式锁服务】tryLock-获取到分布式锁成功！ 耗时 = {} ，lockKey = {}", System.currentTimeMillis() - currentTime, lockKey);
        } catch (Exception e) {
            LogUtil.error(log, DEFAULT_ERROR_LOG, e, System.currentTimeMillis() - currentTime, lockKey);
            throw new RedisLockException(MessageFormat.format("lock error, key={0}",lockKey), e);
        }
    }

    /**
     * 尝试获取锁，并立即返回结果
     *
     * @param lockKey 锁的key
     * @return 是否获取到锁
     */
    @Override
    public boolean tryLock(String lockKey) {
        return tryLock(lockKey, DEFAULT_WAIT_TIME);
    }

    /**
     * 在timeout时限内获取尝试获取锁
     *
     * @param lockKey 锁的key
     * @param timeout 获取锁的超时时间,单位毫秒
     * @return
     * @throws InterruptedException
     */
    @Override
    public boolean tryLock(String lockKey, long timeout) {
        boolean isLock = false;
        long currentTime = System.currentTimeMillis();
        try {
            isLock = redisCache.getRedisLock(lockKey).tryLock(timeout, DEFAULT_LEASE_TIME, TimeUnit.MILLISECONDS);
            LogUtil.info(log, "【分布式锁服务】tryLock-获取到分布式锁成功！ 耗时 = {} ，lockKey = {}", System.currentTimeMillis() - currentTime, lockKey);
            return isLock;
        } catch (InterruptedException e) {
            LogUtil.warn(log, DEFAULT_ERROR_LOG, e, System.currentTimeMillis() - currentTime, lockKey);
            Thread.currentThread().interrupt();
            return isLock;
        } catch (Exception e) {
            LogUtil.error(log, DEFAULT_ERROR_LOG, e, System.currentTimeMillis() - currentTime, lockKey);
            throw new RedisLockException(MessageFormat.format("trylock error, key={0}", lockKey), e);
        }
    }

    /**
     * 解锁
     *
     * @param lockKey 锁的key
     */
    @Override
    public void unlock(String lockKey) {
        long currentTime = System.currentTimeMillis();
        try {
            redisCache.getRedisLock(lockKey).unlock();
            LogUtil.info(log, "【分布式锁服务】tryLock-分布式锁解锁成功！ 耗时 = {} ，lockKey = {}", System.currentTimeMillis() - currentTime, lockKey);
        } catch (Exception e) {
            LogUtil.error(log, "【分布式锁服务】tryLock-分布式锁解锁失败！ 耗时 = {} ,lockKey = {}", e, System.currentTimeMillis() - currentTime, lockKey);
            throw new RedisLockException(MessageFormat.format("unlock error, key={0}", lockKey), e);
        }
    }
}