/**
 * fshows.com
 * Copyright (C) 2013-2024 All Rights Reserved
 */
package com.fshows.ccbpay.request.trade.pay;

import com.alibaba.fastjson.annotation.JSONField;
import com.fshows.ccbpay.enums.trade.CcbPayTradeApiDefinitionEnum;
import com.fshows.ccbpay.request.base.CcbPayJsApiBaseRequest;
import com.fshows.ccbpay.response.trade.pay.CcbAlipayJsApiPayResponse;
import lombok.Data;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.NotBlank;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * @author liubo
 * @version CcbAlipayJsApiPayRequest.java, v 0.1 2024-04-23-3:55 下午 liubo
 */
@Data
public class CcbAlipayJsApiPayRequest extends CcbPayJsApiBaseRequest<CcbAlipayJsApiPayResponse, CcbPayTradeApiDefinitionEnum> {

    private static final long serialVersionUID = -4879495627158691091L;

    /**
     * 商户号
     * 请联系开立商户的分行提供
     *
     * 示例：105421097080009
     * (必填)
     */
    @NotBlank
    @Length(max = 15, message = "merchantid长度不能超过15")
    @JSONField(name = "MERCHANTID")
    private String merchantid;

    /**
     * 柜台号
     * 请联系开立商户的分行提供
     *
     * 示例：902807340
     * (必填)
     */
    @NotBlank
    @Length(max = 9, message = "posid长度不能超过9")
    @JSONField(name = "POSID")
    private String posid;

    /**
     * 分行号
     * 请联系开立商户的分行提供（一级分行号）
     *
     * 示例：360000000
     * (必填)
     */
    @NotBlank
    @Length(max = 9, message = "branchid长度不能超过9")
    @JSONField(name = "BRANCHID")
    private String branchid;


    /**
     * 订单号
     * 由商户提供，最长 30 位
     *
     * 示例：360000000
     * (必填)
     */
    @NotBlank
    @Length(max = 30, message = "orderId长度不能超过30")
    @JSONField(name = "ORDERID")
    private String orderId;

    /**
     * 付款金额
     * 由商户提供，按实际金额给
     * 出
     * 支付完成后，请商户与收到
     * 的商户通知中的付款金额比
     * 对，确认两者金额一致；
     *
     * 示例：0.01
     * (必填)
     */
    @NotBlank
    @JSONField(name = "PAYMENT")
    private String payment;

    /**
     * 缺省为 01－人民币
     * 币种
     * 示例：1
     */
    @NotBlank
    @Length(max = 2, message = "curcode长度不能超过30")
    @JSONField(name = "CURCODE")
    private String curcode;

    /**
     * 备注1
     * 一般作为商户自定义备注信
     * 息使用，可在对账单中显示。
     * 集团主商户代理发起子商户
     * 交易时，需将集团主商户信
     * 息填入该字段，格式如下：
     * JTSH:主商户号+主柜台
     * 例如：
     * JTSH:1050000000000001234
     * 56789
     * 其中：105000000000000 为
     * 主商户号
     * 123456789 为主柜台
     */
    @Length(max = 30, message = "remark1长度不能超过30")
    @JSONField(name = "REMARK1")
    private String remark1;

    /**
     * 备注2
     * 一般作为商户自定义备注信
     * 息使用，可在对账单中显示。
     */
    @Length(max = 30, message = "remark2长度不能超过30")
    @JSONField(name = "REMARK2")
    private String remark2;

    /**
     * 交易码
     */
    @NotBlank
    @Length(max = 6, message = "txCode长度不能超过6")
    @JSONField(name = "TXCODE")
    private String txCode;

    /**
     * 返回类型
     * 0或空：返回二维码页面，只支持建行龙支付；
     * 1：返回JSON格式【二维码信息串】，只支持建行龙支付；
     * 2：返回聚合扫码二维码页面；
     * 3：返回聚合扫码JSON格式【二维码信息串】
     * 聚合扫码只能上送2或3
     * 4:返回聚合银联二维码信息串
     */
    @NotBlank
    @Length(max = 6, message = "txCode长度不能超过6")
    @JSONField(name = "RETURNTYPE")
    private String returnType;

    /**
     * 格式：
     * YYYYMMDDHHMMSS如：
     * 20120214143005
     * 银行系统时间> TIMEOUT
     * 时拒绝交易，若送空值则不
     * 判断超时。
     * 当该字段有值时参与 MAC
     * 校验，否则不参与 MAC 校
     * 验。
     */
    @Length(max = 14, message = "timeOut长度不能超过14")
    @JSONField(name = "TIMEOUT")
    private String timeOut;



    /**
     * 微信支付专有字段
     * 小程序/公众
     * 号的 APPID
     * 当前调起支付的小程序/公
     * 众号 APPID
     */
    @Length(max = 32, message = "subAppId长度不能超过16")
   // @JSONField(name = "SUB_APPID")
    private String subAppId;

    /**
     * 微信支付专有字段
     * 用户子标识
     * 用户在小程序/公众号 appid
     * 下的唯一标识，小程序通过
     * wx.login 获取，接口文档地
     * 址 ：
     * https://developers.weixin.qq.c
     * om/miniprogram/dev/api/apilogin.html?t=20161122
     */
    @Length(max = 128, message = "subOpenId长度不能超过128")
    //@JSONField(name = "SUB_OPENID")
    private String subOpenId;

    /**
     * 返回信息位图
     * 共 20 位，商户通知是否返回某
     * 个字段的位图，0 或空-不返回，
     * 1-返回。
     * 第 1 位：是否返回 OPENID 和
     * SUB_OPENID
     * 第 2 位：保留位，默认送 0
     * 第 3 位：保留位，默认送 0
     * 第 4 位：是否返回支付详细信息
     * 字段
     * 示例：10000000000000000000
     */
    @Length(max = 20, message = "returnField长度不能超过20")
    //@JSONField(name = "RETURN_FIELD")
    private String returnField;

    /**
     * 实名支付功能，包含类型、
     * 证件号、姓名三个子域（如果本字段
     * 出现，那么本字
     * 段包含的三个子域均需出现。详见下
     * 文说明5)USERPARAM字段说明）。
     * 当该字段有值时参与MAC校验，否则不
     * 参与MAC校验。
     */
    @Length(max = 2000, message = "userParam长度不能超过2000")
    @JSONField(name = "USERPARAM")
    private String userParam;

    /**
     * 商品信息
     * 客户购买的商品
     * 中文需使用 escape 编码
     */
    @Length(max = 256, message = "proInfo长度不能超过256")
    @JSONField(name = "PROINFO")
    private String proInfo;

    /**
     * 付款账户限制标识
     * 该字段不参与MAC计算
     * 1、针对微信，商户上送PAYLIMIT这个字段时，字段值填“PAYLIMIT=no_balance”;可限制用户使用微信“余额”和“零钱通”进行支付。
     * 2、针对支付宝，商户上送PAYLIMIT这个字段时，字段值分别填写“PAYLIMIT=balance,moneyFund,pcredit”；可限用户使用支付宝“余额”、“余额宝”和“花呗”进行支付。
     */
    @Length(max = 200, message = "payLimit长度不能超过200")
    @JSONField(name = "PAYLIMIT")
    private String payLimit;

    /**
     * 支付方式位图
     * 10位位图，1为开，0为关
     * 第四位：H5支付标识【0-SDK，1-H5，默认SDK】
     * 其余位数预留。
     * 例如H5上送0001000000
     * 该字段不参与MAC计算
     */
    @Length(max = 10, message = "proInfo长度不能超过10")
    @JSONField(name = "PAYMAP")
    private String payMap;


    /**
     * 银联预授权
     *
     * 0：普通消费
     * 1：银联预授权
     * 对于不送该字段和送空或者0的商户，默认为普通消费。该字段不参与MAC校验
     */
    @Length(max = 10, message = "proInfo长度不能超过10")
    @JSONField(name = "PrePay")
    private String prePay;

    /**
     * MAC 校验域
     * 采用标准 MD5 算法，由商
     * 户实现
     */
    @Length(max = 32, message = "mac长度不能超过32")
    @JSONField(name = "MAC")
    private String mac;

    /**
     * 商户URL
     * 退出支付流程时返回商户 URL，此字段不参与MAC计算
     */
    @Length(max = 32, message = "mac长度不能超过32")
    @JSONField(name = "Mrch_url")
    private String mrchUrl;


    /**
     * WXPAY：微信支付二维码
     * ALIPAY：支付宝二维码
     * 此字段不参与MAC计算
     */
    @Length(max = 10, message = "qrFlag长度不能超过10")
    @JSONField(name = "QRFLAG")
    private String qrFlag;


    /**
     * 公钥后 30 位
     * 商户从建行商户服务平台下
     * 载，截取后 30 位。
     * 仅作为源串参加 MD5 摘要，
     * 不作为参数传递
     */
    @Length(max = 30, message = "pub长度不能超过30")
    @JSONField(name = "PUB")
    private String pub;


    @Override
    public Map<String,String> getReqMap(){

        HashMap map = new LinkedHashMap();
        map.put("MERCHANTID",merchantid);
        map.put("POSID",posid);
        map.put("BRANCHID",branchid);
        map.put("ORDERID",orderId);
        map.put("PAYMENT",payment);
        map.put("CURCODE",curcode);
        map.put("TXCODE",txCode);
        map.put("REMARK1",remark1);
        map.put("REMARK2",remark2);
        map.put("RETURNTYPE",returnType);
        map.put("TIMEOUT",timeOut);
        map.put("PUB",pub);
        return map;
    }


}