/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.ark.spring.boot.starter.util;

import lombok.extern.slf4j.Slf4j;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 修改自 sofa-tracer 中的 traceId 生成
 * 生成规则: 服务器 IP + 产生 ID 时候的时间 + 自增序列 + 当前进程号
 * <a url="http://www.sofastack.tech/sofa-tracer/docs/TraceIdGeneratedRule">TraceId 生成规则</a>
 *
 * @author buhao
 * @version TraceIdGenerator.java, v 0.1 2018-09-15 14:58 buhao
 */
@Slf4j
public class TraceIdGenerator {

    private static String IP_16 = "ffffffff";
    private static AtomicInteger count = new AtomicInteger(1000);
    private static String P_ID_CACHE = null;


    static {
        String ipAddress = getInetAddress();
        if (ipAddress != null) {
            IP_16 = getIP_16(ipAddress);
        }
    }

    private static String getTraceId(String ip, long timestamp, int nextId) {
        StringBuilder appender = new StringBuilder(30);
        appender.append(ip).append(timestamp).append(nextId).append(getPID());
        return appender.toString();
    }

    /**
     * 生成 traceId
     *
     * @return
     */
    public static String generate() {
        return getTraceId(IP_16, System.currentTimeMillis(), getNextId());
    }

    private static String getIP_16(String ip) {
        String[] ips = ip.split("\\.");
        StringBuilder sb = new StringBuilder();
        for (String column : ips) {
            String hex = Integer.toHexString(Integer.parseInt(column));
            if (hex.length() == 1) {
                sb.append('0').append(hex);
            } else {
                sb.append(hex);
            }

        }
        return sb.toString();
    }

    private static int getNextId() {
        for (; ; ) {
            int current = count.get();
            int next = (current > 9000) ? 1000 : current + 1;
            if (count.compareAndSet(current, next)) {
                return next;
            }
        }
    }

    private static String getInetAddress() {
        Enumeration<NetworkInterface> interfaces = null;
        try {
            interfaces = NetworkInterface.getNetworkInterfaces();
            InetAddress address = null;
            while (interfaces.hasMoreElements()) {
                NetworkInterface ni = interfaces.nextElement();
                Enumeration<InetAddress> addresses = ni.getInetAddresses();
                while (addresses.hasMoreElements()) {
                    address = addresses.nextElement();
                    if (!address.isLoopbackAddress() && address.getHostAddress().indexOf(':') == -1) {
                        return address.getHostAddress();
                    }
                }
            }
        } catch (SocketException e ) {
            LogUtil.error(log, "获取IP地址SocketException异常,e={}", e);
        }catch (NullPointerException e){
            LogUtil.error(log, "获取IP地址NullPointerException异常,e={}", e);
        }
        return null;

    }

    /**
     * 此方法在 JDK9 下可以有更加好的方式，但是目前的几个 JDK 版本下，只能通过这个方式来搞。
     * 在 Mac 环境下，JDK6，JDK7，JDK8 都可以跑过。
     * 在 Linux 环境下，JDK6，JDK7，JDK8 尝试过，可以运行通过。
     *
     * @return 进程 ID
     */
    private static String getPID() {
        //check pid is cached
        if (P_ID_CACHE != null) {
            return P_ID_CACHE;
        }
        String processName = java.lang.management.ManagementFactory.getRuntimeMXBean().getName();

        if (StrUtil.isBlank(processName)) {
            return "";
        }

        String[] processSplitName = processName.split("@");

        if (processSplitName.length == 0) {
            return "";
        }

        String pid = processSplitName[0];

        if (StrUtil.isBlank(pid)) {
            return "";
        }
        P_ID_CACHE = pid;
        return pid;
    }
}