/**
 * fshows.com
 * Copyright (C) 2013-2024 All Rights Reserved.
 */
package
        com.fshows.ark.spring.boot.starter.core.sensitive.encrypt;

import cn.hutool.core.codec.Base64;
import cn.hutool.core.util.CharsetUtil;
import cn.hutool.core.util.StrUtil;
import com.fshows.ark.spring.boot.starter.constant.CommonConstant;
import com.fshows.ark.spring.boot.starter.core.sensitive.enums.AlgorithmTypeEnum;
import com.fshows.ark.spring.boot.starter.core.sensitive.model.SecretKeyModel;
import com.fshows.ark.spring.boot.starter.exception.DbSensitiveException;
import com.fshows.ark.spring.boot.starter.util.FsByteUtil;
import com.fshows.ark.spring.boot.starter.util.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 敏感数据加密执行器
 *
 * @author liluqing
 * @version DefaultFieldEncryptExecutor.java, v 0.1 2024-05-29 17:01
 */
@Component
@Slf4j
public class DefaultFieldEncryptExecutor implements FieldEncryptExecutor {

    @Autowired
    private FieldEncryptFactory fieldEncryptFactory;

    @Autowired
    private DefaultSecretKeyManage defaultSecretKeyManage;

    @Override
    public String encrypt(String plaintext) {
        // 从密匙管理器中获取到主密钥
        SecretKeyModel secretKeyModel = defaultSecretKeyManage.getMainEncryptSecretKey();
        return encrypt(plaintext, secretKeyModel);
    }

    /**
     * 执行加密处理
     *
     * @param plaintext
     * @param secretKeyModel
     * @return
     */
    private String encrypt(String plaintext, SecretKeyModel secretKeyModel) {
        // 如果数据为空,则无需处理
        if (StrUtil.isBlank(plaintext)) {
            return plaintext;
        }
        // 如果本身就是密文,则不需要再次加密
        if (!isPlaintext(plaintext)) {
            return plaintext;
        }
        // 算法类型
        AlgorithmTypeEnum algorithmTypeEnum = secretKeyModel.getAlgorithmType();

        // 生成数据头信息
        byte[] keyIdByte = FsByteUtil.splitBytes(secretKeyModel.getKeyId());
        byte[] algorithmTypeByte = FsByteUtil.splitBytes(algorithmTypeEnum.getValue());
        byte[] headBytes = new byte[] {
                keyIdByte[0], keyIdByte[1], algorithmTypeByte[0], algorithmTypeByte[1]
        };
        // 获得密文
        String ciphertext =  doEncrypt(headBytes, plaintext, secretKeyModel);
        // 拼接密文
        return CommonConstant.CIPHERTEXT_PREFIX + ciphertext;
    }


    /**
     * 执行加密处理
     *
     * @param plaintext
     * @param secretKeyModel
     * @return
     */
    private String doEncrypt(byte[] headBytes, String plaintext, SecretKeyModel secretKeyModel) {
        // 根据算法类型获取算法实现策略
        IFieldEncryptStrategy encryptStrategy = fieldEncryptFactory.getFieldEncryptStrategy(secretKeyModel.getAlgorithmType());
        // 得到密文数据
        byte[] cipherBytes = encryptStrategy.encrypt(StrUtil.bytes(plaintext, CharsetUtil.CHARSET_UTF_8), secretKeyModel);

        // 得到最终的密文数据
        byte[] reslutCipherBytes = FsByteUtil.mergeByteArrays(headBytes, cipherBytes);
        return Base64.encode(reslutCipherBytes);
    }


    @Override
    public String encrypt(String plaintext, String secretName) {
        // 从密匙管理器中获取到主密钥
        SecretKeyModel secretKeyModel = defaultSecretKeyManage.getSecretKeyByName(secretName);
        if (secretKeyModel == null) {
            LogUtil.error(log, "ark-spring-boot-starter >> 数据库敏感字段加密 >> 未找到secretName为'{}'的密钥信息", secretName);
            throw new DbSensitiveException("ark-spring-boot-starter >> 数据库敏感字段加密 >> 未找到secretName为" + secretName + "的密钥信息");
        }
        return encrypt(plaintext, secretKeyModel);
    }

    @Override
    public String encryptSearchKeywords(String plaintext) {
        SecretKeyModel secretKeyModel = defaultSecretKeyManage.getSearchKeywordsSecret();
        return encryptSearchKeywords(plaintext, secretKeyModel);
    }


    /**
     * 执行加密处理
     *
     * @param plaintext
     * @param secretKeyModel
     * @return
     */
    private String encryptSearchKeywords(String plaintext, SecretKeyModel secretKeyModel) {
        // 如果数据为空,则无需处理
        if (StrUtil.isBlank(plaintext)) {
            return plaintext;
        }
        // 如果本身就是密文,则不需要再次加密
        if (!isPlaintext(plaintext)) {
            return plaintext;
        }
        // 算法类型
        AlgorithmTypeEnum algorithmTypeEnum = secretKeyModel.getAlgorithmType();

        // 生成数据头信息
        byte[] keyIdByte = FsByteUtil.splitBytes(secretKeyModel.getKeyId());
        byte[] algorithmTypeByte = FsByteUtil.splitBytes(algorithmTypeEnum.getValue());
        byte[] headBytes = new byte[] {
                keyIdByte[0], keyIdByte[1], algorithmTypeByte[0], algorithmTypeByte[1]
        };

        final int length = plaintext.length();
        // 如果长度小于等于 3，直接返回加密
        if (length <= CommonConstant.ENCRYPT_QUERY_MIN_LENGTH) {
            // 拼接密文
            String ciphertext = doEncrypt(headBytes, plaintext, secretKeyModel);
            return CommonConstant.CIPHERTEXT_PREFIX + ciphertext;
        }
        StringBuilder result = new StringBuilder(1000);
        result.append(CommonConstant.CIPHERTEXT_PREFIX);
        // 从第3位开始，列举字段加密
        // eg. 123456 -> en(123)+en(1234)+en(12345)+en(123456)
        for (int i = 3; i <= length; i++) {
            String partCiphertext = doEncrypt(headBytes, StrUtil.subWithLength(plaintext, 0, i), secretKeyModel);
            result.append(partCiphertext);
            if (i != length) {
                // 加一个点分隔，防止字符串交接处，匹配字符
                result.append(CommonConstant.CIPHERTEXT_SEPARATOR);
            }
        }
        return result.toString();
    }

    @Override
    public String encryptSearchKeywords(String plaintext, String secretName) {
        SecretKeyModel secretKeyModel = defaultSecretKeyManage.getSecretKeyByName(secretName);
        return encryptSearchKeywords(plaintext, secretKeyModel);
    }

    @Override
    public String decrypt(String ciphertext) {
        // 如果当前字段不是密文,则直接返回文本本身
        if (isPlaintext(ciphertext)) {
            return ciphertext;
        }
        String[] base64DataArray = StrUtil.split(ciphertext, CommonConstant.CIPHERTEXT_SEPARATOR);
        if (base64DataArray == null || base64DataArray.length < 2) {
            return null;
        }
        // 获取base64DataArray最后一个非空的元素
        String base64Data = getLastNotBlankStr(base64DataArray);
        byte[] plain = Base64.decode(base64Data);
        // 如果解密后的数据不存在,或者没有头部4字节之外的主体数据,则返回空
        if (plain == null || plain.length <= 4) {
            return null;
        }
        // 获取密钥ID
        int keyId = FsByteUtil.mergeBytes(plain[1], plain[0]);
        // 获取算法类型
        int algorithmType = FsByteUtil.mergeBytes(plain[3], plain[2]);
        AlgorithmTypeEnum algorithmTypeEnum = AlgorithmTypeEnum.getByValue(algorithmType);

        // 根据算法类型和keyid获取密钥信息
        SecretKeyModel secretKeyModel = defaultSecretKeyManage.getSecretKey(Integer.toString(keyId), algorithmTypeEnum);

        // 根据算法类型获取算法实现策略
        IFieldEncryptStrategy encryptStrategy = fieldEncryptFactory.getFieldEncryptStrategy(algorithmTypeEnum);

        // 提取到头信息之外的密文主体数据
        byte[] body = new byte[plain.length - 4];
        System.arraycopy(plain, 4, body, 0, body.length);

        // 明文data数据
        byte[] plainData = encryptStrategy.decrypt(body, secretKeyModel);
        // 按照utf-8编码进行解码
        return StrUtil.str(plainData, CharsetUtil.CHARSET_UTF_8);
    }

    /**
     * 获取列表中最后一个非空元素
     * @param list
     * @return
     */
    private String getLastNotBlankStr(String[] list) {
        for (int i = list.length - 1 ; i >= 0; i--) {
            String str = list[i];
            if (StrUtil.isNotBlank(str)) {
                return str;
            }
        }
        return null;
    }


    /**
     * 当前字符串是否密文
     *
     * @param plaintext
     * @return
     */
    private boolean isPlaintext(String plaintext) {
        return !StrUtil.startWith(plaintext, CommonConstant.CIPHERTEXT_PREFIX);
    }
}