/**
 * fshows.com
 * Copyright (C) 2013-2021 All Rights Reserved.
 */
package
        com.fshows.ark.spring.boot.starter.core.mq.rocketmq.producer.transaction;

import com.fshows.ark.spring.boot.starter.core.mq.base.FsMessageConvert;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsMessageSendContext;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsSendResult;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.TransactionMessageManage;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.TransactionCommitCallback;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.datasource.ConnectionHolder;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import javax.sql.DataSource;
import java.sql.Connection;

/**
 * 默认事务消息生产者
 *
 * @author liluqing
 * @version DefaultTransactionProducer.java, v 0.1 2021-12-19 10:22
 */
@Slf4j
public class DefaultTransactionMessageManageImpl implements TransactionMessageManage {

    private DataSource dataSource;

    @Autowired
    private TransactionMessageDao transactionMessageDao;

    public DefaultTransactionMessageManageImpl(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    @Override
    public FsSendResult sendTransactionMsg(FsMessageSendContext context, TransactionCommitCallback transactionCommitCallback) {
        ConnectionHolder conHolder = (ConnectionHolder)TransactionSynchronizationManager.getResource(dataSource);
        Connection conn = conHolder.getConnection();
        // 将消息插入到本地消息表（包含超时时间）
        transactionMessageDao.insertTransactionMessage(conn, context);
        // 注册事务同步回调器
        TransactionSynchronizationManager.registerSynchronization(new DefaultTransactionSynchronization(context, transactionCommitCallback));
        // 构建一个支付成功的结果并返回
        return FsMessageConvert.buildSendResult(context.getFsMessage(), true);
    }

    @Override
    public boolean updateMsgSendStatusToSuccess(FsMessageSendContext context) {
        Connection conn = null;
        try {
            // 从数据源中获取连接
            conn = DataSourceUtils.getConnection(this.dataSource);
            if (DataSourceUtils.isConnectionTransactional(conn, this.dataSource)) {
                transactionMessageDao.updateMsgSendStatusToSuccess(conn, context);
                conn.commit();
            }
        } catch (Exception e) {
            log.error("ark-spring-boot-starter >> 更新本地消息表消息发送状态更新失败! message={}", e, context.getFsMessage());
            return false;
        } finally {
            // 释放连接
            DataSourceUtils.releaseConnection(conn, dataSource);
        }
        return true;
    }

    @Override
    public boolean isInTransaction() {
        // TransactionAspectSupport.currentTransactionStatus();
        // TransactionAspectSupport.getTransactionManager();
        // DataSourceUtils.getConnection(this.dataSource);
        // DataSourceUtils.isConnectionTransactional(this.connection, this.dataSource);
        // 判断当前是否处于事务中，如果在事务中,则按照事务消息逻辑进行处理,否则按照普通消息进行处理
        if (!TransactionSynchronizationManager.isActualTransactionActive() || !TransactionSynchronizationManager.isSynchronizationActive()) {
            return false;
        }
        ConnectionHolder conHolder = (ConnectionHolder)TransactionSynchronizationManager.getResource(dataSource);
        if (conHolder == null) {
            return false;
        }
        Connection conn = conHolder.getConnection();
        return conn != null;
    }
}