/**
 * fshows.com
 * Copyright (C) 2013-2021 All Rights Reserved.
 */
package
        com.fshows.ark.spring.boot.starter.core.mq.rocketmq.producer;

import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.SendResult;
import com.aliyun.openservices.ons.api.bean.ProducerBean;
import com.fshows.ark.spring.boot.starter.core.mq.base.FsMessage;
import com.fshows.ark.spring.boot.starter.core.mq.base.FsMessageConvert;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsMessageSendContext;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsSendErrorCallback;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsSendResult;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsSendSuccessCallback;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.IFshowsProducer;
import com.fshows.ark.spring.boot.starter.core.mq.rocketmq.interceptor.ProducerInterceptorActuator;
import com.fshows.ark.spring.boot.starter.enums.ProducerSendTypeEnum;
import com.fshows.ark.spring.boot.starter.enums.ProducerTypeEnum;

/**
 * 阿里云MQSDK producerImpl
 *
 * @author liluqing
 * @version AliyunFshowsProducer.java, v 0.1 2021-08-20 16:19
 */
public class SimpleFshowsProducerImpl implements IFshowsProducer {

    /**
     * 实际的生产者bean
     */
    private ProducerBean producerBean;

    /**
     * 消息内容字符集编码
     */
    private String charsetName;


    SimpleFshowsProducerImpl(ProducerBean producerBean, String charsetName) {
        this.producerBean = producerBean;
        this.charsetName = charsetName;
    }

    @Override
    public void start() {
        producerBean.start();
    }

    @Override
    public void shutdown() {
        producerBean.shutdown();
    }

    @Override
    public FsSendResult send(FsMessage fsMessage) {
        FsSendResult fsSendResult = doSend(fsMessage);
        return fsSendResult;
    }

    /**
     * 执行实际消息发送处理
     *
     * @param fsMessage
     * @return
     */
    private FsSendResult doSend(FsMessage fsMessage) {
        // 执行消息发送
        Message message = FsMessageConvert.convertMessage(fsMessage, this.charsetName);
        SendResult sendResult = producerBean.send(message);
        FsSendResult fsSendResult = FsMessageConvert.convertFsSendResult(sendResult);
        return fsSendResult;
    }

    @Override
    public void sendOneway(FsMessage fsMessage) {
        doSendOneway(fsMessage);
    }

    /**
     * 执行Oneway消息发送处理
     *
     * @param fsMessage
     */
    private void doSendOneway(FsMessage fsMessage) {
        // 执行消息发送
        Message message = FsMessageConvert.convertMessage(fsMessage, this.charsetName);
        producerBean.sendOneway(message);
    }

    @Override
    public void sendAsync(FsMessage fsMessage, FsSendSuccessCallback sendSuccessCallback, FsSendErrorCallback fsSendErrorCallback) {
        FsMessageSendContext context = this.buildMsgContext(fsMessage, ProducerSendTypeEnum.ASYNC,
                sendSuccessCallback, fsSendErrorCallback);
        this.doSendAsync(context, sendSuccessCallback, fsSendErrorCallback);
    }

    /**
     * 构建消息上下文
     */
    private FsMessageSendContext buildMsgContext(FsMessage fsMessage, ProducerSendTypeEnum sendTypeEnum, FsSendSuccessCallback successCallback, FsSendErrorCallback errorCallback) {
        FsMessageSendContext context = new FsMessageSendContext();
        context.setProducerSendTypeEnum(sendTypeEnum);
        context.setProducerTypeEnum(ProducerTypeEnum.TRANSACTION_MESSAGE_PRODUCER);
        context.setSuccessCallback(successCallback);
        context.setErrorCallback(errorCallback);
        context.setFsMessage(fsMessage);
        return context;
    }

    /**
     * 执行消息异步发送
     *
     * @param context
     * @param sendSuccessCallback
     * @param fsSendErrorCallback
     */
    private void doSendAsync(FsMessageSendContext context, FsSendSuccessCallback sendSuccessCallback, FsSendErrorCallback fsSendErrorCallback) {
        // 执行异步消息发送, 并且在消息回调中进行拦截器后置处理
        Message message = FsMessageConvert.convertMessage(context.getFsMessage(), this.charsetName);
        producerBean.sendAsync(message, new AliyunSendCallBackWrap(sendSuccessCallback, fsSendErrorCallback));
    }

    @Override
    public void sendAsync(FsMessage fsMessage, FsSendSuccessCallback sendSuccessCallback) {
        FsMessageSendContext context = this.buildMsgContext(fsMessage, ProducerSendTypeEnum.ASYNC,
                sendSuccessCallback, null);
        this.doSendAsync(context, sendSuccessCallback, null);
    }

    @Override
    public void sendAsync(FsMessage fsMessage, FsSendErrorCallback fsSendErrorCallback) {
        FsMessageSendContext context = this.buildMsgContext(fsMessage, ProducerSendTypeEnum.ASYNC,
                null, fsSendErrorCallback);
        this.doSendAsync(context, null, fsSendErrorCallback);
    }

    @Override
    public FsSendResult send(FsMessageSendContext context) {
        FsSendResult fsSendResult = null;
        if (ProducerSendTypeEnum.SYNC.equals(context.getProducerSendTypeEnum())) {
            try {
                fsSendResult = this.doSend(context.getFsMessage());
            } catch (Exception ex) {
                fsSendResult = new FsSendResult(ex);
                if (context.getRocketSendMessageMethodModel() != null
                        && context.getRocketSendMessageMethodModel().isSendErrorThrowEx()) {
                    throw ex;
                }
            }
        }
        if (ProducerSendTypeEnum.ASYNC.equals(context.getProducerSendTypeEnum())) {
            this.doSendAsync(context, context.getSuccessCallback(), context.getErrorCallback());
        }
        if (ProducerSendTypeEnum.ONE_WAY.equals(context.getProducerSendTypeEnum())) {
            this.doSendOneway(context.getFsMessage());
        }
        return fsSendResult;
    }
}