/**
 * fshows.com
 * Copyright (C) 2013-2021 All Rights Reserved.
 */
package
        com.fshows.ark.spring.boot.starter.core.mq.rocketmq.producer;

import com.fshows.ark.spring.boot.starter.annotation.mq.RocketProducer;
import com.fshows.ark.spring.boot.starter.annotation.mq.RocketProducerScan;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.PropertyValue;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.boot.autoconfigure.AutoConfigurationPackages;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.annotation.AnnotationAttributes;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.util.StringUtils;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * 自动注册mq生产者扫描器
 *
 * @author liluqing
 * @version AutoConfiguredProducerScannerRegistrar.java, v 0.1 2021-08-15 23:04
 */
@Slf4j
public class ProducerScannerRegistrar implements BeanFactoryAware, ImportBeanDefinitionRegistrar {

    /**
     * spring的beanfactory
     */
    private BeanFactory beanFactory;

    @Override
    public void registerBeanDefinitions(AnnotationMetadata importingClassMetadata, BeanDefinitionRegistry registry) {
        Set<String> packageSet = new HashSet<>();
        // 获取自动配置目录
        if (AutoConfigurationPackages.has(this.beanFactory)) {
            List<String> packages = AutoConfigurationPackages.get(this.beanFactory);
            if (log.isDebugEnabled()) {
                packages.forEach(pkg -> log.debug("Rocket MQ 生产者接口扫描  >>  检测到spring自动配置扫描 base package '{}'", pkg));
            }
            packages.forEach(packageSet::add);
        }

        // 获取@RocketProducerScan注解手动配置的生产者路路径
        AnnotationAttributes attributes = AnnotationAttributes.fromMap(
                importingClassMetadata.getAnnotationAttributes(RocketProducerScan.class.getName()));
        if (attributes != null) {
            String[] componentScanPackages = attributes.getStringArray("basePackages");
            if (componentScanPackages != null) {
                Arrays.stream(componentScanPackages).forEach(packageSet::add);
                if (log.isDebugEnabled()) {
                    Arrays.stream(componentScanPackages).forEach(pkg -> log.debug("Rocket MQ 生产者接口扫描  >> 检测到@RocketProducerScan配置的base package '{}'", pkg));
                }
            }
        }

        if (packageSet.size() == 0) {
            return;
        }

        boolean bo = registry.containsBeanDefinition(ProducerScannerConfigurer.class.getName());
        if (bo) {
            BeanDefinition definition = registry.getBeanDefinition(ProducerScannerConfigurer.class.getName());
            MutablePropertyValues mpv = definition.getPropertyValues();
            String[] originalBasePackageArr = getOriginalBasePackage(definition);
            Arrays.stream(originalBasePackageArr).forEach(packageSet::add);
            mpv.addPropertyValue("basePackage", StringUtils.collectionToCommaDelimitedString(packageSet));
            return;
        }

        BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(ProducerScannerConfigurer.class);
        builder.addPropertyValue("annotationClass", RocketProducer.class);
        builder.addPropertyValue("basePackage", StringUtils.collectionToCommaDelimitedString(packageSet));

        // 将mq生产者自动配置扫描器注册到spring容器中
        registry.registerBeanDefinition(ProducerScannerConfigurer.class.getName(), builder.getBeanDefinition());
    }

    /**
     * 获取原扫描路径
     *
     * @param definition
     * @return
     */
    private String[] getOriginalBasePackage(BeanDefinition definition) {
        MutablePropertyValues mpv = definition.getPropertyValues();
        PropertyValue value = mpv.getPropertyValue("basePackage");
        if (value == null || value.getValue() == null) {
            return new String[]{};
        }
        String v = (String) value.getValue();
        return v.split(",");
    }


    @Override
    public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
        this.beanFactory = beanFactory;
    }
}